;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2019, 2020 Jan Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2019 Rob Wieringa <rob.wieringa@verum.com>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.
;;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands verify)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 rdelim)
  #:use-module (ice-9 receive)
  #:use-module (gaiag parse)
  #:use-module (ide commands daemon)
  #:use-module (ide commander)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide commands simulate)
  #:use-module (ide util)
  #:use-module (json)
  #:use-module (gash pipe)
  #:use-module (gaiag shell-util)
  #:use-module (gaiag commands trace))

(define (parse-opts args)
  (let* ((option-spec
          '((all (single-char #\a))
            (help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))
            (queue-size (single-char #\q) (value #t))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide verify [OPTION]... [FILE]...
Run dzn verify and make a trace view available.

  -a, --all              keep going after first error
  -h, --help             display this help and exit
  -I, --import=DIR+      add directory DIR to import path
  -m, --model=MODEL      explore model MODEL
  -q, --queue-size=SIZE  use queue size=SIZE for verification [3]

View:
  ~a
  ~a
" (get-url "trace") (get-url "watch"))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (execute input procs process-line)
  (define (read-string port)
    (let loop ((res ""))
      (let ((line (read-line port)))
       (if (eof-object? line) res
           (loop (string-append res "\n" (process-line line)))))))
  (let* ((pipeline (pipeline procs))
         (pids (cddr pipeline))
         (to (car pipeline))
         (from (cadr pipeline)))
         (display input to)
    (catch #t (lambda _ (close to)) (const #f))
    (let ((output (read-string from)))
      (catch #t (lambda _ (close from)) (const #f))
      (values output (apply + (map (compose status:exit-val cdr waitpid) pids))))))

(define (display-any-status json)
  (let* ((data (json-string->scm json))
         (data (if (and (pair? data) (null? (cdr data))) (car data) data)))
    (let ((res (cond ((assoc-ref data "result") (assoc-ref data "result"))
                     ((assoc-ref data "trace") "fail")
                     (else #f))))
      (if res (format #t "verify: ~a: check: ~a: ~a\n" (assoc-ref data "model") (assoc-ref data "assert") res))))
  json)

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (all (option-ref options 'all #f))
         (queue-size (option-ref options 'queue-size #f))
         (files (option-ref options '() #f))
         (file-name (car files))
         (imports (multi-opt options 'import))
         (model (option-ref options 'model #f))
         (result EXIT_SUCCESS))

    (define (simulate-handler message)
      (log-debug  "received: ~s\n" message)
      (let ((message (parse-message message)))
        (cond ((simulate? message)
               (let ((error (handle-simulate-result message #:debug? debug? #:verbose? verbose?)))
                 (when error
                   (set! result EXIT_FAILURE)
                   (format (current-error-port) "~a" error))
                 (when verbose?
                   (format (current-error-port) "See result: ~a\n" (get-url "trace"))
                   (format (current-error-port) "            ~a\n" (get-url "watch")))))
              ((error? message)
               (display (error-stderr message) (current-error-port))
               (set! result EXIT_OTHER_FAILURE))
              (else
               (format (current-error-port)
                       "ide verify: error: received: ~s\n"
                       message)
               (set! result EXIT_OTHER_FAILURE)))
        message))

    (define* (run-verify input #:key debug? model)
      (let ((verify-command `("dzn" "--json" "-v" "verify" "--all"
                              ,@(if model `("--model" ,model) '())
                              ,@(if all `("--all") '())
                              ,@(if queue-size `("--queue-size" ,queue-size) '())
                              "-")))
        (receive (output status)
            (execute input (list verify-command) display-any-status)
          (if (and (not (zero? status)) (equal? output "")) status
              (let* ((data (string-split output #\newline))
                     (data (filter (negate (cute equal? <> "")) data))
                     (data (map json-string->scm data))
                     (data (map (lambda (d) (if (and (pair? d) (null? (cdr d))) (car d) d)) data)))
                (when debug?
                  (format #t "~a\n" data))
                (let ((errors (filter (lambda (d) (string? (assoc-ref d "trace"))) data)))
                  (if (pair? errors) errors
                      (last-pair data))))))))

    (define* (make-simulate-command verify-result #:key input)
      (let* ((error (if (null? verify-result) '() (car verify-result)))
             (model (assoc-ref error "model"))

             (trace (or (assoc-ref error "trace") ""))
             (interface-trace (assoc-ref error "interface-trace"))
             (message (assoc-ref error "message"))

             (trace (string-map (lambda (c) (if (eq? c #\newline) #\, c)) trace))
             (trace (if (mingw?) (string-append trace ",") trace)))
        (make-command "simulate"
                      `("simulate"
                        "--model" ,model
                        "--trace" ,trace
                        ,@(if message `("--error" ,message) '())
                        ,@(if interface-trace `("--spectrace" ,interface-trace) '())
                        "-")
                      #:input input
                      #:model model
                      #:trace trace)))

    (let* ((files (option-ref options '() #f))
           (file-name (car files))
           (imports (multi-opt options 'import))
           (model (option-ref options 'model #f))
           (no-browser? (ide:command-line:get 'no-browser #f))
           (input (file->stream file-name #:debug? debug? #:imports imports))
           (verify-result (run-verify input #:debug? debug? #:model model)))

      (when (and (number? verify-result)
                 (not (zero? verify-result)))
        (exit EXIT_OTHER_FAILURE))

      (daemon:start #:debug? debug? #:port %ide-port)
      (let* ((command (make-simulate-command verify-result #:input input))
             (commander (make-commander #:debug? debug? #:verbose? verbose?))
             (result-handler
              (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                       simulate-handler))
             (result
              (+ (run-command commander command
                              #:result-handler result-handler
                              #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
                 result))
             (result
              (if (or no-browser?
                      (eq? result EXIT_SUCCESS)
                      (eq? result EXIT_OTHER_FAILURE)) result
                  (let* ((command (make-command "hello" '("dzn" "hello")))
                         (commander (make-commander #:debug? debug? #:verbose? verbose?))
                         (result-handler
                          (compose (make-finalizer commander
                                                   #:debug? debug? #:verbose? verbose?)
                                   (make-hello-browse-handler
                                    "trace"
                                    #:debug? debug? #:verbose? verbose?))))
                    (+ (run-command commander command
                                    #:result-handler result-handler
                                    #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
                       result)))))
        (exit result)))))
