;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2017, 2018, 2019, 2020 Jan Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2018 Rob Wieringa <Rob.Wieringa@verum.com>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.
;;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide script)
  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 poe)
  #:use-module (system repl error-handling)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ide config)
  #:use-module (ide command-line)
  #:use-module (gaiag shell-util)
  #:export (main
            parse-opts))

(define (parse-opts args)
  (let* ((option-spec
	  '((debug (single-char #\d))
            (help (single-char #\h))
            (no-browser (single-char #\B))
	    (peg (single-char #\p))
	    (verbose (single-char #\v))
	    (version (single-char #\V))))
	 (options (getopt-long args option-spec
		               #:stop-at-first-non-option #t))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
	 (usage? (and (not help?) (null? files)))
	 (version? (option-ref options 'version #f)))
    (define (list-commands)
      (delete-duplicates
       (map (cut basename <> ".go")
            (filter (cut string-contains <> "/ide/commands/")
                    (append-map (cut find-files <> "\\.go$")
                                (filter directory-exists?
                                        %load-compiled-path))))
       string=?))
    (when version?
      (format #t "ide ~a\n" %version)
      (exit EXIT_SUCCESS))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port)))
            (commands (list-commands )))
        (format port "\
Usage: ide [OPTION]... COMMAND [COMMAND-ARGUMENT...]
  -d, --debug            enable debug ouput
  -h, --help             display this help
  -B, --no-browser       never launch a browser
  -p, --peg              use plain PEG, skip well-formedness
  -v, --verbose          be more verbose, show progress
  -V, --version          display version

Commands:~a

Use \"ide COMMAND --help\" for command-specific information.
"
                (string-join commands "\n  " 'prefix)))
      (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS)))
    options))

(define parse-opts (pure-funcq parse-opts))

(define (run-command args)
  (let* ((command (string->symbol (car args)))
         (module (resolve-module `(ide commands ,command)))
         (main (false-if-exception (module-ref module 'main))))
    (unless main
      (format (current-error-port) "ide: no such command: ~a\n" command)
      (exit EXIT_OTHER_FAILURE))
    (main args)))

(define (main args)
  (let* ((options (parse-opts args))
         (command-args (option-ref options '() '()))
         (command (and (pair? command-args) (car command-args)))
         (debug? (option-ref options 'debug #f)))
    (if (getenv "IDE_REPL") (call-with-error-handling (cut run-command command-args))
        (run-command command-args))))
