/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */


class SeparatorP5 {

  /*
   * Separator between the two diagrams 'diagram1' and 'diagram2'.
   * horizontalBar: boolean
   * For the time being the whole window is used
   * drag(dr) coordinates dragging
   */

  constructor(parent, horizontalBar, diagram1, diagram2) {
    this.horizontalBar = horizontalBar;
    this.diagram1 = diagram1;
    this.diagram2 = diagram2;

    this.in = {
      dimensions: function (px, py, width, height) {
        let fraction = this.mid / this.size;
        this.px = px;
        this.py = py;
        this.width = width;
        this.height = height;
        this.size = this.horizontalBar ? this.height : this.width;
        this.mid = this.size * fraction;
        this.divide(this.mid);
      }.bind(this)
    };
    this.out = {
      selected: function (location) {
        console.log('selected location: %j', location);
      }
    };

    // default dimensions: whole window
    this.px = 0;
    this.py = 0;
    this.width = window.innerWidth;
    this.height = window.innerHeight;
    
    this.barw = 8;
    this.bar = new BarP5(parent, horizontalBar);

    this.size = this.horizontalBar ? this.height : this.width;
    this.mid = this.size / 2;
    this.divide(this.mid);

    this.bar.out.drag = function(dr) {
      console.log('drag %j', dr);
      this.mid += dr;
      this.divide(this.mid);
    }.bind(this);

    this.bar.out.resize = function(width, height) {
      let fraction = this.mid / this.size;
      this.width = width;
      this.height = height;
      this.size = this.horizontalBar ? this.height : this.width;
      this.mid = this.size * fraction;
      this.divide(this.mid);
    }.bind(this);
  }

  divide(mid) {
    this.mid = mid;
    if (this.horizontalBar) {
      this.diagram1.in.dimensions(this.px, this.py, this.width, this.mid-this.barw/2);
      this.diagram2.in.dimensions(this.px, this.mid+this.barw/2, this.width, this.height-this.mid-this.barw/2);
      this.bar.in.dimensions(this.px, this.mid-this.barw/2, this.width, this.barw);
    } else {
      this.diagram1.in.dimensions(this.px, this.py, this.mid-this.barw/2, this.height);
      this.diagram2.in.dimensions(this.mid+this.barw/2, this.py, this.width-this.mid-this.barw/2, this.height);
      this.bar.in.dimensions(this.mid-this.barw/2, this.py, this.barw, this.height);
    }
  }
}

class BarP5 {

  /*
   * interface to the outside world, through functions
   *   in.dimensions(px, py, width, height) {
   *     change position (px, py) and size (width, height) of the canvas
   *   out.drag(dr)
   *     drag the canvas dr pixels
   *
   *   out functions can be (re)defined by the outside world.
   */

  constructor(parent, horizontalBar) {
    this.horizontalBar = horizontalBar;
    
    this.in = {
      dimensions: function (px, py, width, height) {
        this.sketch.dimensions(px, py, width, height);
      }.bind(this)
    };
    this.out = {
      drag: function(dr) {
        console.log('Bar: drag %j', dr);
      },
      resize: function(width, height) {
        console.log('Bar: resize %j, %j', width, height);
      },
    };
    // private:
    this.sketch = new p5(this.barSketch);
    this.sketch.in = this.in;
    this.sketch.out = this.out;
    this.sketch.parent = parent;
    this.sketch.horizontalBar = horizontalBar;
    this.sketch.myPx = 1;
    this.sketch.myPy = 1;
    this.sketch.myWidth = 1;
    this.sketch.myHeight = 1;
  };

  barSketch(p) {
    // interface with surrounding SeparatorP5 object:
    p.in = null;
    p.out = null;
    p.parent = null;
    p.horizontalBar = true;
    p.myPx = null;
    p.myPy = null;
    p.myWidth = null;
    p.myHeight = null;
    
    /*
     * Visualisation is done using standard functions defined in p5js
     * The World object handles the appropriate transformations.
     *   set_up: Boolean
     *     p5 setup is done
     *   setup: nil --> nil
     *     called once by p5js. Do all initialisation here.
     *     noLoop() is called to prevent continuous redrawing, so function
     *     redraw needs to be called explicitely to update the screen
     *   draw: nil --> nil
     *     called by p5
     *     show the result
     *   windowResized: nil --> nil
     *     called by p5
     *     delegate upward
     *   mouseWheel: Event --> nil
     *     called by p5
     *     noop
     *   mousePressed: Event --> nil
     *     called by p5
     *     enable dragging until mouseRelease is called.
     *   mouseDragged: Event --> nil
     *     called by p5
     *     sent out event, and redraw
     *   mouseReleased: Event --> nil
     *     called by p5
     *     stops dragging
     *   mouseMoved: Event --> nil
     *     called by p5
     *     change the cursor shape
     */

    p.set_up = false;

    p.dimensions = function(px, py, width, height) {
      p.myPx = px;
      p.myPy = py;
      p.myWidth = width;
      p.myHeight = height;
      if (p.set_up) {
        world.positionCanvas(px, py);
        world.resizeCanvas(width, height);
        p.redraw();
      }
    };

    let world = null;

    p.setup = function() {
      let px = p.myPx;
      let py = p.myPy;
      let w = p.myWidth || 4;
      let h = p.myHeight || p.windowHeight;
      world = new World(p, w, h, p.parent);
      if (px != null && py != null) world.positionCanvas(px, py);
      p.set_up = true;
      p.noLoop();
    };
    
    p.draw = function() {
      p.background('white');
      p.push();
      p.fill('#99BBDD');
      p.noStroke();
      let hgap = p.horizontalBar ? 4 : .5;
      let vgap = p.horizontalBar ? .5 : 4;
      p.rect(hgap, vgap, p.myWidth-2*hgap, p.myHeight-2*vgap);
      p.pop();
    };

    function mouseInCanvas() {
      if (!p.myWidth && !p.myHeight) return true; // canvas spans whole window
      if (p.myWidth && !(0 <= p.mouseX && p.mouseX <= p.myWidth)) return false;
      if (p.myHeight && !(0 <= p.mouseY && p.mouseY <= p.myHeight)) return false;
      return true;
    }
    
    p.mouseWheel = function(e) {
      if (!mouseInCanvas()) return true;
      return false;
    };

    p.windowResized = function() {
      p.out.resize(p.windowWidth, p.windowHeight);
//      if (!p.myWidth && !p.myHeight)
//        world.resizeCanvas(p.windowWidth, p.windowHeight);
    };

    let dragging = {
      active: false, // set/reset my mousedown and mouseup
      offset: {x: 0, y: 0} // dragging startpoint
    };
    p.mousePressed = function(e) {
      if (!mouseInCanvas()) return true;
      dragging.active = true;
      dragging.offset.x = p.mouseX;
      dragging.offset.y = p.mouseY;

      return false;
    };

    p.mouseReleased = function(e) {
      dragging.active = false;
      return false;
    };

    p.mouseDragged = function(e) {
      if (!dragging.active) return true;
      let mx = p.mouseX;
      let my = p.mouseY;
      let dr = p.horizontalBar ? my-dragging.offset.y : mx-dragging.offset.x;
      console.log('bar: drag %j dr', dr);
      if (dr != 0) {
        p.out.drag(dr);
        p.redraw();
      }
      return false;
    };

    p.mouseMoved = function(e) {
      let eps = 3;
      if ((0-eps <= p.mouseX && p.mouseX <= p.myWidth+eps) &&
          (0-eps <= p.mouseY && p.mouseY <= p.myHeight+eps) &&
          !dragging.active) {
        p.cursor(p.horizontalBar ? 'row-resize' : 'col-resize');
      } else {
        p.cursor(p.ARROW);
      }
      p.redraw();
      return false;
    };

  };

};
