/*
 * Copyright (C) 2020,2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

/***********************************************************
 * Viz couples data to visualisation
 ***********************************************************/

/*
 * Viz: the abstract base class for all visualisation classes.
 * inherits from BuildingBlock
 * elements:
 *   data: Data (which can be anything)
 *     the data to be visualised
 *   viz: BuildingBlock
 *     the object's visualisation
 * functions:
 *   new: Data --> Viz
 *     create Viz objects for all relevant sub data here.
 *   initViz: nil --> nil
 *     call initViz on all relevant sub viz object here
 *     create the object's own viz (BuildingBlock)
 *   setViz: BuildingBlock --> nil
 *     set the object's viz, and its dimensions
 *   refresh, move, show, bounds, shift, objectsAt
 *     apply these functions on the object's viz
 */
class Viz extends BuildingBlock {
  constructor(data) {
    super();
    this.klass = 'Viz';
    this.data = data;
    this.viz = new BuildingBlock();
  }

  initViz() {
    // noop
  }

  setViz(viz) {
    viz.refreshMe();
    this.viz = viz;
    this.x = viz.x;
    this.y = viz.y;
    this.width = viz.width;
    this.height = viz.height;
  }

  setParent(parent) {
    this.parent = parent;
    this.viz.setParent(parent);
  }

  bounds() {
    return this.viz.bounds();
  }

  relativeBounds(parent) {
    return this.viz.relativeBounds(parent);
  }

  refresh() {
    this.viz.refresh();
    this.x = this.viz.x;
    this.y = this.viz.y;
    this.width = this.viz.width;
    this.height = this.viz.height;
  }

  move(x, y) {
    this.viz.move(x, y);
    this.x = this.viz.x;
    this.y = this.viz.y;
  }

  hCenter(x) {
    this.viz.hCenter(x);
    this.x = this.viz.x;
    this.y = this.viz.y;
  }

  center(x, y) {
    this.viz.center(x, y);
    this.x = this.viz.x;
    this.y = this.viz.y;
  }

  shift(dx, dy) {
    this.viz.shift(dx, dy);
    this.x = this.viz.x;
    this.y = this.viz.y;
  }

  scale(sw, sh) {
    this.viz.scale(sw, sh);
    this.scaleX = this.viz.scaleX;
    this.scaleY = this.viz.scaleY;
  }

  show(p) {
    this.viz.show(p);
  }

  shift(dx, dy) {
    let oldx = this.x;
    let oldy = this.y;
    this.move(oldx+dx, oldy+dy);
  }

  objectsAt(x, y) {
    if (this.viz.atPos(x, y)) {
      return this.viz.objectsAt(x, y).concat([this]);
    } else {
      return [];
    }
  }

  highlight(on) {
    this.viz.highlight(on);
  }
}
