// Dezyne-IDE --- An IDE for Dezyne
//
// Copyright © 2019 Johri van Eerd <johri.van.eerd@verum.com>
// Copyright © 2019,2020 Rob Wieringa <Rob.Wieringa@verum.com>
// Copyright © 2020 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
//
// This file is part of Dezyne-IDE.
//
// Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
// All rights reserved.

function decouple(f, g) {
  if (1) {
    window.requestAnimationFrame(function() {
      f();
      window.requestAnimationFrame(function() {
        if (g) g();
      });
    });
  }
  else {
    f();
    if (g) g();
  }
}

var system_go = {
  diagram: null,
  selectedHeader: null,
    in: {
      set_cursor: function(shape) {
        system_go.diagram.startTransaction('Set cursor');
        system_go.diagram.defaultCursor = shape;
        system_go.diagram.commitTransaction('Set cursor');
      },
      draw: function(jsondata) {
        if (!jsondata) return;
        try {
          system_go.diagram.model = go.Model.fromJson(jsondata);
          system_go.diagram.toolManager.dragSelectingTool.isEnabled = false;
          system_go.diagram.toolManager.panningTool.isEnabled = false;
          system_go.diagram.toolManager.linkingTool.isEnabled = false;
          system_go.diagram.toolManager.resizingTool.isEnabled = false;
          system_go.diagram.toolManager.hoverDelay = 400;
          system_go.diagram.model.undoManager.isEnabled = true;
          system_go.diagram.animationManager.isEnabled = false;

          for (var it = system_go.diagram.nodes; it.next(); ) {
            var node = it.value;
            node.data.index = 10000;
            for (var it_l = node.findLinksInto(); it_l.next();) {
              var link = it_l.value;
              if (link && link.toPort && link.toPort.data && link.toPort.data.data && link.toPort.data.data.dir=='in') {
                var index = link.fromPort.data.data.index;
                node.data.index = Math.min(node.data.index, index);
              }
            }
          }

          system_go.reset_highlights();
        }
        catch (e) {
          alert(e.message);
        }
      },
      redraw: function() {
        system_go.diagram.requestUpdate();
      },
      show_trace: function(blackboxes, trace_nodes, trace_links) {
        system_go.diagram.startTransaction ('Set selection');

        system_go.reset_highlights();

        blackboxes.forEach(function(n) {
          system_go.diagram.model.setDataProperty (n, 'stroke', 'black');
          system_go.diagram.model.setDataProperty (n, 'strokeWidth', 3);
        });
        trace_nodes.forEach(function(n) {
          system_go.diagram.model.setDataProperty (n, 'stroke', color_scheme.stroke_high);
          system_go.diagram.model.setDataProperty (n, 'strokeWidth', 3);
        });
        trace_links.forEach(function(l) {
          system_go.diagram.model.setDataProperty (l, 'stroke', color_scheme.stroke_high);
          system_go.diagram.model.setDataProperty (l, 'strokeWidth', 3);
        });
        system_go.diagram.commitTransaction('Set selection');
      },
      stop: function () {
        //system_go.diagram.div.style.borderColor = color_scheme.stop_border;
        //system_go.diagram.div.style.borderWidth = color_scheme.stop_width;
      },
      init: function() {
        if (system_go.diagram) {
          // Even when deleating DIAGRAM, DIAGRAM.DIV needs to be reset to avoid
          // "Uncaught Error: Invalid div id; div already has a Diagram associated with it."
          system_go.diagram.div = null;
          delete system_go.diagram;
        }
        var $ = go.GraphObject.make;  //for conciseness in defining node templates

        var portSize = 12;

        function MyTreeLayout() {
          go.TreeLayout.call(this);
        }
        go.Diagram.inherit(MyTreeLayout, go.TreeLayout);

        MyTreeLayout.prototype.doLayout = function(coll) {
          if (coll === this.group) {
            if (!this.group.isSubGraphExpanded) {
              return;
            }
          }
          go.TreeLayout.prototype.doLayout.call(this, coll);
        }

        var layouter =
            $(MyTreeLayout,
              {
                angle: 90,
                layerSpacing: 30,
                //breadthLimit: 4000,
                setsPortSpot: false,
                alignment: go.TreeLayout.AlignmentCenterSubtrees,
                arrangement: go.TreeLayout.ArrangementHorizontal,
                arrangementSpacing: new go.Size(20,20),
                sorting: go.TreeLayout.SortingAscending,
                comparer: compare_index
              }
             );
        function compare_index(v0, v1) {
          var data0 = v0.node.data;
          var data1 = v1.node.data;
          return data0.index-data1.index;
        }

        system_go.diagram =
          $(go.Diagram, 'system',
            {
              allowDelete: false,
              allowCopy: false,
              allowDrop: false,
              allowGroup: false,
              allowInsert: false,
              allowRelink: false,
              initialContentAlignment: go.Spot.Center,
              initialAutoScale: go.Diagram.Uniform
            }
           );
        // system_go.diagram.grid.visible = true;
        // system_go.diagram.grid.gridCellSize = new go.Size(portSize, portSize);
        // system_go.diagram.toolManager.draggingTool.isGridSnapEnabled = true;
        // system_go.diagram.toolManager.resizingTool.isGridSnapEnabled = true;

        function blackbox_selected() {
          if (system_go.selectedHeader) {
            var type = system_go.selectedHeader.type;
            var blackboxed = system_go.selectedHeader.blackboxed;
            if (type == 'component' || type == 'system' || blackboxed) {
              var instances = system_go.selectedHeader.key.split('.');
              instances[0] = 'sut';
              var instance = instances.join('.');
              console.log('BLACKBOX: %j', instance);
              system_go.diagram.clearSelection();
              system_go.selectedHeader = null;
              system_go.out.blackbox(instance);
            }
          }
        };

        system_go.diagram.commandHandler.doKeyDown = function() {
          var e = system_go.diagram.lastInput;
          console.log('key press: %j', e);

          switch (e.key)
          {
            case 'H': {
              showHelp();
              break;
            }
            case 'B': {
              blackbox_selected();
              break;
            }
            default: {
              // call base method with no arguments
              go.CommandHandler.prototype.doKeyDown.call(this);
              break;
            }
          }
        };

        var tool = system_go.diagram.toolManager.clickSelectingTool;
        tool.standardMouseSelect = function() {
          var e = system_go.diagram.lastInput;
          var obj = system_go.diagram.findPartAt(e.documentPoint, false);
          if (obj) {
            system_go.diagram.select(obj);
            var iscomponent = obj.data.type == 'system' || obj.data.type == 'component';
            if (iscomponent) {
              system_go.selectedHeader = obj.data;
            } else {
              system_go.selectedHeader = null;
            }
          }
          else {
            system_go.diagram.clearSelection();
            system_go.selectedHeader = null;
          }
        };

        function portSymbol() {
          return $(go.Panel, 'Auto',
                   $(go.Shape, 'Rectangle',
                     { name: 'PORT',
                       stroke: null,
                       desiredSize: new go.Size(portSize, portSize),
                       fill: null
                     },
                     new go.Binding("data","data")
                    ),
                   $(go.Shape, 'Rectangle',
                     { stroke: 'grey',
                       desiredSize: new go.Size(portSize, portSize),
                     },
                     new go.Binding('fill', 'data', function(data) {
                       if (data.dir=='in' || data.dir=='out') {
                         if (data.external)
                           return color_scheme.component.external;
                         else
                           return color_scheme.component.port;
                       }
                       else {
                         return color_scheme.component.injection;
                       }
                     })
                    ),
                   $(go.Shape, 'Triangle',
                     { stroke: null,
                       desiredSize: new go.Size(portSize-4, portSize-4),
                       fill: '#444444',
                       angle: 180
                     },
                     new go.Binding('angle', 'data', function(data) {
                       if (data.dir=='in' || data.dir=='out') {
                         return 180;
                       }
                       else {
                         return 90;
                       }
                     })
                    ),
                   { click: function(event, obj) {
                     var shape = obj.panel.findObject('PORT');
                     if (event.control) {
                       system_go.out.click(shape.data.defloc);
                     }
                     else {
                       system_go.out.click(shape.data.fileloc);
                     }
                   },
                     toolTip:
                     $(go.Adornment, 'Auto',
                       $(go.Shape, {fill: '#FFFFCC'}),
                       $(go.TextBlock,
                         { margin: 4 },
                         new go.Binding('text', 'data', function(data) {
                           return data.label;
                         })
                        )
                      )
                   }
                  );
        };

        function nodeColor(node) {
          if (node.type == 'foreign') return color_scheme.component.foreign;
          if (node.group=='') return color_scheme.component.system;
          if (node.injected) return color_scheme.component.injection;
          if (node.type == 'system') return color_scheme.component.system;
          return color_scheme.component.component;
        };

        function minNodeSize(node) {
          if (node.injected) {
            return new go.Size(portSize*2, portSize*1);
          }
          else {
            var max_ports = Math.max(node.inPortTot, node.outPortTot);
            return new go.Size(portSize*Math.max(6, max_ports*2-1), portSize*Math.max(5, node.injPortTot*2-1));
          }
        };

        function numNodes(g, rec) {
          function numNodes_(node, g, rec) {
            var res = 1;
            if (node instanceof go.Group && (node==g||node.data.isSubGraphExpanded)) {
              for (var nit = node.memberParts; nit.next();) {
                var sub = nit.value;
                if (sub instanceof go.Node) {
                  res += numNodes_(sub,(node==g&&rec?sub:g),rec);
                }
              }
            }
            return res;
          }
          for (var nit = system_go.diagram.nodes; nit.next();) {
            var node = nit.value;
            if (node.data.group=='') {
              return numNodes_(node, g, rec);
            }
          }
          return 0;
        };

        function setSubGraphExpanded(g, expand, rec, not_top) {
          if (!(g instanceof go.Group)) return;
          var cmd = system_go.diagram.commandHandler;
          if (rec) {
            g.memberParts.each(function(m) {
              setSubGraphExpanded(m, expand, rec, true);
            })
          }
          system_go.diagram.startTransaction('expand');
          if (expand){
            if (not_top) {
              g.isSubGraphExpanded = true;
            } else {
              cmd.expandSubGraph(g);
            }
            system_go.diagram.model.setDataProperty(g.data, 'isSubGraphExpanded', true);
          }
          else {
            if (not_top) {
              g.isSubGraphExpanded = false;
            } else {
              cmd.collapseSubGraph(g);
            }
            system_go.diagram.model.setDataProperty(g.data, 'isSubGraphExpanded', false);
          }
          system_go.diagram.commitTransaction('expand');
        };

        function myclick(e, obj) {
          var group = obj.part;  // OBJ is this button
          if (group instanceof go.Adornment) group = group.adornedPart;
          if (!(group instanceof go.Group)) return;
          var cmd = system_go.diagram.commandHandler;
          if (group.isSubGraphExpanded) {
            if (!cmd.canCollapseSubGraph(group)) return;
          } else {
            if (!cmd.canExpandSubGraph(group)) return;
          }
          e.handled = true;

          var num = numNodes(group, e.control);
          setSubGraphExpanded(group, !group.isSubGraphExpanded, e.control);
        };

        function systemAdornment() {
          return $(go.Adornment, 'Auto',
                   $(go.Shape, 'Rectangle',
                     { fill: null, stroke: color_scheme.component.selected, strokeWidth: 3 }),
                   $(go.Placeholder)
                  );
        };

        system_go.diagram.groupTemplate =
          $(go.Group, 'Spot',
            {
              resizable: true,
              selectionAdorned: true,
              selectionAdornmentTemplate: systemAdornment(),
              selectionObjectName: 'COMP',
              locationObjectName: 'COMP',
              resizeObjectName: 'COMP',
              layout: layouter,
              isShadowed: true,
              shadowOffset: new go.Point(2,2),
              shadowColor: '#CCCCCC',
              itemTemplate:
              $(go.Panel, 'Spot',
                { fromSpot: go.Spot.Bottom, toSpot: go.Spot.Top,
                  fromLinkable: true, toLinkable: true, cursor: 'pointer',
                },
                new go.Binding('portId', 'portId'),
                // new go.Binding('fileloc', 'fileloc'),
                new go.Binding('alignment', 'data', function(data) {
                  if (data.dir=='in') {
                    return new go.Spot(data.relloc, 0, 0, 0);
                  }
                  else if (data.dir=='out') {
                    return new go.Spot(data.relloc, 1, 0, 0);
                  }
                  else { // data.dir=='inj'
                    return new go.Spot(1, data.relloc, 0, 0);
                  }
                }),
                new go.Binding('fromSpot', 'data', function(data) {
                  if (data.dir=='in' || data.dir=='out') {
                    return go.Spot.Bottom;
                  }
                  else { // data.dir=='inj'
                    return go.Spot.Right;
                  }
                }),
                new go.Binding('toSpot', 'data', function(data) {
                  if (data.dir=='in' || data.dir=='out') {
                    return go.Spot.Top;
                  }
                  else { // data.dir=='inj'
                    return go.Spot.Left;
                  }
                }),
                portSymbol()
               ),
              subGraphExpandedChanged: function(group) {
                var comp = group.findObject('COMP');
                if (comp) {
                  if (ast.is_a(group.data, 'system') && group.isSubGraphExpanded) {
                    comp.fill = color_scheme.component.system;
                  } else {
                    comp.fill = nodeColor(group.data);
                  }
                }
              }
            },
            new go.Binding('itemArray', 'portArray'),
            new go.Binding('key', 'key'),
            new go.Binding('location', 'loc', go.Point.parse).makeTwoWay(go.Point.stringify),
            $(go.Panel, 'Auto',
              {
                click: function(event, obj) {
                  if (event.control) {
                    system_go.out.click(obj.part.data.defloc);
                  }
                  else {
                    system_go.out.click(obj.part.data.fileloc);
                  }
                },
                //doubleClick: function(event, obj) { system_go.doubleClick(obj.part.data.label); }
              },
              $(go.Shape,
                { name: 'COMP'
                },
                new go.Binding('figure', '', function(node) {
                  return ((node.injected)?'RoundedRectangle':'Rectangle');
                }),
                new go.Binding('stroke', 'stroke'),
                new go.Binding('strokeWidth', 'strokeWidth'),

                new go.Binding('fill', '', nodeColor),

                new go.Binding('minSize', '', minNodeSize)
               ),
              $(go.Placeholder,
                { padding: portSize*2.5
                }
               ),
              $(go.Panel, 'Horizontal',
                new go.Binding('alignment', '', function(node) {
                  return ((node.size > 1)?new go.Spot(0, 0, 0, 0):new go.Spot(0.5, 0.5));
                }),
                new go.Binding('margin', '', function(node) {
                  return ((node.size > 1)?new go.Margin(5, 3, 3, 3):new go.Margin(3, 3, 3, 3));
                }),
                $('SubGraphExpanderButton',
                  { margin: new go.Margin(0, 0, 3, 3),
                    click: myclick
                  },
                  new go.Binding('visible', '', function(node) { return (node.size > 1); })
                 ),
                $(go.TextBlock,
                  { font: '10pt Sans-Serif',
                    textAlign: 'center',
                    margin: new go.Margin(3, 6, 3, 6)
                  },
                  new go.Binding('text', 'label')
                 )
               )
             )
           );

        system_go.diagram.nodeTemplate =
          $(go.Node, 'Spot',
            {
              resizable: true,
              selectionAdorned: true,
              selectionAdornmentTemplate: systemAdornment(),
              selectionObjectName: 'COMP',
              locationObjectName: 'COMP',
              resizeObjectName: 'COMP',
              isShadowed: true,
              shadowOffset: new go.Point(2,2),
              shadowColor: '#CCCCCC',
              itemTemplate:
              $(go.Panel, 'Spot',
                { fromSpot: go.Spot.Bottom, toSpot: go.Spot.Top,
                  fromLinkable: true, toLinkable: true, cursor: 'pointer',
                },
                new go.Binding('portId', 'portId'),
                // new go.Binding('fileloc', 'fileloc'),
                new go.Binding('alignment', 'data', function(data) {
                  if (data.dir=='in') {
                    return new go.Spot(data.relloc, 0, 0, 0);
                  }
                  else if (data.dir=='out') {
                    return new go.Spot(data.relloc, 1, 0, 0);
                  }
                  else { // data.dir=='inj'
                    return new go.Spot(1, data.relloc, 0, 0);
                  }
                }),
                new go.Binding('fromSpot', 'data', function(data) {
                  if (data.dir=='in' || data.dir=='out') {
                    return go.Spot.Bottom;
                  }
                  else { // data.dir=='inj'
                    return go.Spot.Right;
                  }
                }),
                new go.Binding('toSpot', 'data', function(data) {
                  if (data.dir=='in' || data.dir=='out') {
                    return go.Spot.Top;
                  }
                  else { // data.dir=='inj'
                    return go.Spot.Left;
                  }
                }),
                portSymbol()
               )
            },
            new go.Binding('key', 'key'),
            new go.Binding('itemArray', 'portArray'),
            $(go.Panel, 'Auto',
              {
                click: function(event, obj) {
                  if (event.control) {
                    system_go.out.click(obj.part.data.defloc);
                  }
                  else {
                    system_go.out.click(obj.part.data.fileloc);
                  }
                },
                //doubleClick: function(event, obj) { system_go.doubleClick(obj.part.data.label); }
              },
              // new go.Binding('fileloc', 'fileloc'),
              $(go.Shape,
                { name: 'COMP'
                },
                new go.Binding('figure', '', function(node) {
                  return ((node.injected)?'RoundedRectangle':'Rectangle');
                }),
                new go.Binding('stroke', 'stroke'),
                new go.Binding('strokeWidth', 'strokeWidth'),
                new go.Binding('fill', '', nodeColor),
                new go.Binding('minSize', '', minNodeSize)
               ),
              $(go.TextBlock,
                { font: '10pt Sans-Serif',
                  textAlign: 'center',
                  margin: new go.Margin(6, 9, 6, 9),
                },
                new go.Binding('text', 'label')
               )
             )
           );

        system_go.diagram.linkTemplate =
          $(go.Link,
            {
              routing: go.Link.AvoidsNodes,
              // reshapable: true,
              // resegmentable: true,
              corner: 4,
              click: function(event, obj) {
                console.log ('click: obj: %j', obj);
                if (event.control) {
                  system_go.out.click(obj.data.defloc);
                }
                else {
                  system_go.out.click(obj.data.fileloc);
                }
              }
            },
            // new go.Binding('fileloc', 'fileloc'),
            new go.Binding('points').makeTwoWay(),
            $(
              go.Shape,
              new go.Binding('stroke', 'stroke'),
              new go.Binding('strokeWidth', 'strokeWidth'),
            )
           );
      },
    }
  ,
  out: {
    click: function() { },
    blackbox: function(instance) {}
  }
  ,

  reset_highlights: function () {
    system_go.diagram.model.nodeDataArray.forEach (function (n) {
      system_go.diagram.model.setDataProperty (n, 'stroke', 'grey');
      system_go.diagram.model.setDataProperty (n, 'strokeWidth', 1);
    });

    system_go.diagram.model.linkDataArray.forEach (function (l) {
      system_go.diagram.model.setDataProperty (l, 'stroke', 'grey');
      system_go.diagram.model.setDataProperty (l, 'strokeWidth', 1);
    });
  }
};
