;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2020,2021 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands verify)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 rdelim)
  #:use-module (ice-9 receive)
  #:use-module (ide commands daemon)
  #:use-module (ide commander)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide commands simulate)
  #:use-module (ide pipe)
  #:use-module (ide util)
  #:use-module (ide shell-util))

(define (parse-opts args)
  (let* ((option-spec
          '((all (single-char #\a))
            (help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))
            (queue-size (single-char #\q) (value #t))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide verify [OPTION]... [FILE]...
Run dzn verify and make a trace view available.

  -a, --all              keep going after first error
  -h, --help             display this help and exit
  -I, --import=DIR+      add directory DIR to import path
  -m, --model=MODEL      explore model MODEL
  -q, --queue-size=SIZE  use queue size=SIZE for verification [3]

View:
  ~a
" (get-url "trace"))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (execute input procs process-line)
  (define (read-string port)
    (let loop ((res ""))
      (let ((line (read-line port)))
        (if (eof-object? line) res
            (loop (string-append res "\n" (process-line line)))))))
  (receive (to from pids)
      (pipeline procs)
    (display input to)
    (catch #t (lambda _ (close to)) (const #f))
    (let ((output (read-string from)))
      (catch #t (lambda _ (close from)) (const #f))
      (values output (apply + (map (compose status:exit-val cdr waitpid) pids))))))

(define (display-any-status json)
  (let* ((data (json-string->alist-scm json))
         (data (if (and (pair? data) (null? (cdr data))) (car data) data)))
    (let ((res (cond ((assoc-ref data "result") (assoc-ref data "result"))
                     ((assoc-ref data "trace") "fail")
                     (else #f))))
      (if res (format #t "verify: ~a: check: ~a: ~a\n" (assoc-ref data "model") (assoc-ref data "assert") res))))
  json)

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (all (option-ref options 'all #f))
         (queue-size (option-ref options 'queue-size #f))
         (files (option-ref options '() #f))
         (file-name (car files))
         (imports (multi-opt options 'import))
         (model (option-ref options 'model #f))
         (result EXIT_SUCCESS))

    (define (simulate-handler message)
      (log-debug  "received: ~s\n" message)
      (let ((message (parse-message message)))
        (cond ((simulate? message)
               (let ((error (handle-simulate-result message #:debug? debug? #:verbose? verbose?)))
                 (when error
                   (set! result EXIT_FAILURE)
                   (format (current-error-port) "~a" error))
                 (when (isatty? (current-error-port))
                   (format (current-error-port) "See result: ~a\n" (get-url "trace")))))
              ((error? message)
               (display (error-stderr message) (current-error-port))
               (set! result EXIT_OTHER_FAILURE))
              (else
               (format (current-error-port)
                       "ide verify: error: received: ~s\n"
                       message)
               (set! result EXIT_OTHER_FAILURE)))
        message))

    (define* (run-verify input #:key debug? model)
      (let ((verify-command `("dzn" "--json" "--verbose" "verify"
                              ,@(if model `("--model" ,model) '())
                              ,@(if all `("--all") '())
                              ,@(if queue-size `("--queue-size" ,queue-size) '())
                              "-")))
        (receive (output status)
            (execute input (list verify-command) display-any-status)
          (if (and (not (zero? status)) (equal? output "")) status
              (let* ((data (string-split output #\newline))
                     (data (filter (negate (cute equal? <> "")) data))
                     (data (map json-string->alist-scm data))
                     (data (map (lambda (d) (if (and (pair? d) (null? (cdr d))) (car d) d)) data)))
                (when debug?
                  (format #t "~a\n" data))
                (let ((errors (filter (lambda (d) (string? (assoc-ref d "trace"))) data)))
                  (if (pair? errors) errors
                      (last-pair data))))))))

    (define* (make-simulate-command verify-result #:key input)
      (let* ((error (if (null? verify-result) '() (car verify-result)))
             (model (assoc-ref error "model"))

             (trace (or (assoc-ref error "trace") ""))
             (interface-trace (assoc-ref error "interface-trace"))
             (message (assoc-ref error "message"))

             (trace (string-map (lambda (c) (if (eq? c #\newline) #\, c)) trace))
             (trace (if (mingw?) (string-append trace ",") trace))
             (interface-trace (and interface-trace
                                   (string-map (lambda (c) (if (eq? c #\newline) #\, c)) interface-trace)))
             (interface-trace (and interface-trace
                                   (if (mingw?) (string-append interface-trace ",") interface-trace))))
        (make-command "simulate"
                      `("simulate"
                        "--model" ,model
                        "--trace" ,trace
                        ,@(if message `("--error" ,message) '())
                        ,@(if interface-trace `("--spectrace" ,interface-trace) '())
                        "-")
                      #:input input
                      #:model model
                      #:trace trace)))

    (let* ((files (option-ref options '() #f))
           (file-name (car files))
           (imports (multi-opt options 'import))
           (model (option-ref options 'model #f))
           (input (piped-file->stream file-name #:debug? debug? #:imports imports))
           (verify-result (run-verify input #:debug? debug? #:model model)))

      (when (and (number? verify-result)
                 (not (zero? verify-result)))
        (exit EXIT_OTHER_FAILURE))

      (daemon:start #:debug? debug? #:port %ide-port)
      (let* ((command (make-simulate-command verify-result #:input input))
             (commander (make-commander #:debug? debug? #:verbose? verbose?))
             (result-handler
              (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                       simulate-handler))
             (result
              (+ (run-command commander command
                              #:result-handler result-handler
                              #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
                 result)))
        (exit result)))))
