;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2019 Rob Wieringa <Rob.Wieringa@verum.com>
;;; Copyright © 2019 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (ide util)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)
  #:use-module (ice-9 rdelim)
  #:use-module (ice-9 match)
  #:use-module (ide config)
  #:use-module (ide command-line)
  #:use-module (ide json)
  #:use-module (ide pipe)
  #:export (conjoin
            disjoin
            piped-file->stream
            get-url
            handle-json-error
            hash-table->alist

            json-string->alist-scm
            log-debug
            log-error
            log-format
            log-trace
            log-verbose
            pke))

;;;
;;; Utilities.
;;;
(define (disjoin . predicates)
  (lambda arguments
    (any (cut apply <> arguments) predicates)))

(define (conjoin . predicates)
  (lambda arguments
    (every (cut apply <> arguments) predicates)))

(define (get-url what)
  (let* ((ports `(,@(if (equal? %view-port %default-view-port) '()
                        (list (format #f "port=~a" %view-port)))
                  ,@(if (equal? %http-port %default-http-port) '()
                        (list (format #f "http=~a" %http-port)))))
         (params (if (null? ports) ""
                     (string-append "?" (string-join ports "&"))))
         (host (format #f "http://localhost:~a" %http-port)))
    (cond ((member what '("simulate" "trace" "verify"))
           (string-append host "/trace" params))
          ((member what '("state"))
           (string-append host "/state" params))
          ((member what '("parse" "system"))
           (string-append host "/system" params))
          (else ""))))


;;;
;;; Logging.
;;;
(define (pke . stuff)
  "Like peek (pk), writing to (CURRENT-ERROR-PORT)."
  (newline (current-error-port))
  (display ";;; " (current-error-port))
  (write stuff (current-error-port))
  (newline (current-error-port))
  (car (last-pair stuff)))

(define* (truncate-string string #:key (length
                                        (or (and=> (getenv "COLUMNS") string->number)
                                            80)))
  (if (< (string-length string) length) string
      (let ((newline? (string-suffix? "\n" string)))
        (string-append (string-take string (- length 3 (if newline? 1 0)))
                       "..."
                       (if newline? "\n" "")))))

(define (log-format port string . rest)
  (let ((time (strftime "%Y-%m-%d %T" (localtime (current-time)))))
    (apply format port (string-append "~a " string) time rest)))

(define (log-debug string . rest)
  (let ((debugity (+ (ide:debugity) (command-line:debugity))))
   (when (> debugity 1)
     (let* ((message (apply format #f string rest))
            (message (if (> debugity 2) message
                         (truncate-string message))))
       (display message (current-error-port))))))

(define (log-error string . rest)
  (apply log-format (current-error-port) string rest))

(define (log-verbose string . rest)
  (let ((debugity (+ (ide:debugity) (command-line:debugity)))
        (verbosity (+ (ide:verbosity) (command-line:verbosity))))
    (when (or (> debugity 0) (> verbosity 0))
      (let* ((message (apply log-format #f string rest))
             (message (if (> debugity 2) message
                          (truncate-string message))))
        (display message (current-error-port))))))

(define* (log-trace string . rest)
  (let ((debugity (+ (ide:debugity) (command-line:debugity))))
    (when (> debugity 1)
      (apply format (current-error-port) string rest))))


;;;
;;; JSON and alists.
;;;

(define (hash-table->alist table)
  (hash-map->list cons table))

(define (handle-json-error o)
  (lambda args
    (match args
      (('json-invalid (and (? port?) port))
       (let ((data (read-string port)))
         (format (current-error-port) "json-invalid: at=~s\n" data)
         (format (current-error-port) "while parsing :~s\n" o)
         '()))
      (('json-invalid scm)
       (format (current-error-port) "json-invalid: at=~s\n" scm)
       (format (current-error-port) "while building :~s\n" o)
       "")
      ((key . args)
       (format (current-error-port) "~a: at=~s\n" key args)
       (format (current-error-port) "while handling :~s\n" o)
       ""))))

(define (json-string->alist-scm src)
  "Compatibility between guile-json-1 (which produces hash-tables) and
guile-json-4 (which produces vectors)."
  (match src
    ((? hash-table?) (json-string->alist-scm (hash-table->alist src)))
    ((h ...) (map json-string->alist-scm src))
    ((h . t) (cons (json-string->alist-scm h) (json-string->alist-scm t)))
    (#(x ...) (json-string->alist-scm (vector->list src)))
    ((? string?) (if (or (string-prefix? "[" src)
                         (string-prefix? "{" src))
                     (catch #t (lambda _ (json-string->alist-scm (json-string->scm src))) (const src))
                     src))
    ("false" #f)
    ('false #f)
    ("true" #t)
    ('true #t)
    (_ src)))

(define* (piped-file->stream file-name #:key debug? (imports '()))
  "Run dzn --preprocess with IMPORTs on FILE-NAME and return the file stream."
  (let* ((imports (append-map (cute list "-I" <>) imports))
         (command `("dzn" ,@(if debug? '("--debug") '()) "parse"
                    "--preprocess" ,@imports ,file-name))
         (output status (pipeline->string (list command))))
    (if (zero? status) output
        (exit status))))
