/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */


class StateDiagramP5 {

  /*
   * interface to the outside world, through functions
   *   in.draw(data)
   *     show the data as a diagram
   *   in.reset()
   *     reset to show an empty canvas
   *   in.dimensions(px, py, width, height)
   *     set the location (px, py) and size (with, height) of the
   *     diagram
   *   out.selected(location)
   *     process the selected node, using its location
   *
   *   out functions can be (re)defined by the outside world.
   */

  constructor(parent) {
    this.in = {
      draw: function (data) {
        this.sketch.data = data;
        if (this.sketch.set_up) {
          this.sketch.initDiagram(data);
          this.sketch.redraw();
        }
      }.bind(this),
      reset: function () {
        this.sketch.data = null;
        if (this.sketch.set_up) {
          this.sketch.initDiagram(null);
          this.sketch.redraw();
        }
      }.bind(this),
      dimensions: function (px, py, width, height) {
        this.sketch.dimensions(px, py, width, height);
      }.bind(this)
    };
    this.out = {
      selected: function (location) {
        console.log('selected location: %j', location);
      }
    };

    // default dimensions: whole window
    this.px = 0;
    this.py = 0;
    this.fixedSize = false;
    this.width = window.innerWidth;
    this.height = window.innerHeight;

    // interface with sketch:
    this.sketch = new p5(this.stateSketch, parent);
    this.sketch.in = this.in;
    this.sketch.out = this.out;
    this.sketch.myPx = this.px;
    this.sketch.myPy = this.py;
    this.sketch.fixedSize = this.fixedSize;
    this.sketch.myWidth = this.width;
    this.sketch.myHeight = this.height;
    this.sketch.dataLimit = 4000;
  };

  stateSketch(p) {
    // interface with surrounding StateDiagramP5 object:
    p.in = null;
    p.out = null;
    p.myPx = null;
    p.myPy = null;
    p.fixedSize = false;
    p.myWidth = null;
    p.myHeight = null;
    
    /*
     * Visualisation is done using standard functions defined in p5js
     * Zooming and dragging are facilitated. The World object handles the
     * appropriate transformations.
     *   set_up: Boolean
     *     p5 setup is done
     *   data: Data
     *     data has been provided by initDiagram event
     *   initDiagram: data --> nil
     *     create the diagram from the input data in two phases.
     *     must not be called before setup()
     *   setup: nil --> nil
     *     called once by p5js. Do all Viz object initialisation here.
     *     noLoop() is called to prevent continuous redrawing, so function
     *     redraw needs to be called explicitely to update the screen
     *   draw: nil --> nil
     *     called by p5 
     *     transform world to screen coordinates, refresh all Viz objects,
     *     and show the result
     *   windowResized: nil --> nil
     *     called by p5 
     *   mouseWheel: Event --> nil
     *     called by p5 
     *     use World.mouseWheel to handle zooming
     *   mousePressed: Event --> nil
     *     called by p5 
     *     enable dragging world or Node until mouseRelease is called.
     *   mouseDragged: Event --> nil
     *     called by p5 
     *     move Node or Canvas, and redraw
     *   mouseReleased: Event --> nil
     *     called by p5 
     *     when not dragging highlight Node
     *   mouseMoved: Event --> nil
     *     called by p5 
     *     when hovering over a Node, show its details
     */

    p.set_up = false;
    p.data = null;

    p.dimensions = function(px, py, width, height) {
      p.myPx = px;
      p.myPy = py;
      p.myWidth = width;
      p.myHeight = height;
      p.fixedSize = true;
      if (p.set_up) {
        world.positionCanvas(px, py);
        world.resizeCanvas(width, height);
        p.redraw();
      }
    };
      
    let phase = {status: 'none', // {'none', 'simple', 'full', 'stable'}
                 stability: 0};

    function incTimeOut() {
      let time = p.millis();
      while (timeOut < time) timeOut += 1000;
    }
    
    // initDiagram is accessed outside sketch
    // do not call before p5 setup!
    p.initDiagram = function(data) {
      resetEnvironment();
      if (data == null) {
        diagram = null;
        p.noLoop();
      } else {
        nrNodes = data.states.length + data.transitions.length;
        if (nrNodes > p.dataLimit) {
          let msg = 
              '    more than '+ p.dataLimit + ' nodes   \n' +
              '    cowardly refusing to draw    \n\n';
          p.data = {states:[{id:'0', state:msg}], transitions:[]};
        } else  {
          p.data = data;
        }
        // start timing...
        timeOut = p.millis();
        incTimeOut();
        simpleDiagram = new StateDiagramSimple(p.data);
        simpleDiagram.initViz();
        simpleDiagram.refresh();
        phase.status = 'simple';
        phase.stability = stabilize(simpleDiagram, 1, timeOut);
        
        diagram = Message();
        p.loop();
      }
    };

    function resetEnvironment() {
      //pre: p.set_up
      world.scale = 1;
      world.x0 = 0;
      world.y0 = 0;
      diagram = null;
      simpleDiagram = null;
      fullDiagram = null;
      phase.status = 'none';
      dragging.subject = 'none';
      autoFit = true;
      highlightedObj = null;
      detail = null;
    }

    function Message() {
      let message = new RoundedBoundingBox(new Text(''));
      message.color = 'white';
      message.padding = 20;
      message.center(p.windowWidth/2, p.windowHeight/2);
      message.refresh();
      return message;
    }

    function setMessage(diagram, msg) {
      diagram.content.text = msg;
      diagram.content.refresh();
      diagram.center(p.windowWidth/2, p.windowHeight/2);
      diagram.refresh();
    }

    function stabilize(diagram, accuracy, timeOut) {
      if (diagram.maxEnergyPerNode == null) diagram.maxEnergyPerNode = diagram.energyPerNode;
      let result = 0;
      let time = p.millis();
      let iter = 0;
      while (time < timeOut && diagram.energyPerNode > accuracy) {
        let logEN = Math.log(accuracy/diagram.energyPerNode);
        let minEN = Math.min(Math.log(accuracy/diagram.maxEnergyPerNode), logEN);
        result = Math.floor((1 - logEN/minEN) * 100);
        diagram.refresh();
        iter++;
        diagram.maxEnergyPerNode = Math.max(diagram.maxEnergyPerNode, diagram.energyPerNode);
        time = p.millis();
      }
      if (diagram.energyPerNode <= accuracy) result = 100;
      console.log('at ' + Math.floor(time/1000) + ' sec: iterations: ' + iter + '; stability: ' + result);
      return result;
    };

    let world = null;
    let diagram = null;
    let simpleDiagram = null;
    let fullDiagram = null;
    let nrNodes = 0;
    let layoutAborted = false;
    let timeOut = 0;
    let accuracy = .01;
    
    p.setup = function() {
      let px = p.myPx;
      let py = p.myPy;
      let w = p.fixedSize ? p.myWidth : p.windowWidth;
      let h = p.fixedSize ? p.myHeight : p.windowHeight;
      world = new World(p, w, h);
      if (px != null && py != null) world.positionCanvas(px, py);
      p.set_up = true;
      if (p.data) {
        p.initDiagram(p.data);
      } else {
        p.noLoop();
      }
    };
    
    let dragging = {subject: 'none', // values: 'none', 'world', 'node'
                    node: null, // when subject == 'node'
                    offset: {x: 0, y: 0} // dragging startpoint
                   };
    let autoFit = true;
    
    p.draw = function() {
      p.background(255);
      if (diagram == null) {
        p.noLoop();
      } else if (phase.status == 'none') {
        p.noLoop();
      } else if (phase.status == 'simple') {
        p.push();
        world.set();

        if (phase.stability < 100) {
          setMessage(diagram,
                     'calculating layout ' + nrNodes + ' nodes\n\n' +
                     'PHASE I accuracy ' + phase.stability + '%');
          diagram.show(p);
          incTimeOut();
          phase.stability = stabilize(simpleDiagram, 1, timeOut);
        } else {
          fullDiagram = new StateDiagram(p.data);
          fullDiagram.initViz();
          fullDiagram.copyLayout(simpleDiagram);
          fullDiagram.refresh();
          console.log('----------- PHASE II -----------');
          phase.status = 'full';
          // accuracy = .01 * Math.sqrt(nrNodes);
          accuracy = .001 * nrNodes;
          phase.stability = stabilize(fullDiagram, accuracy, timeOut);
        }
        p.pop();
      } else if (phase.status == 'full') {
        p.push();
        world.set();
        incTimeOut();
        phase.stability = stabilize(fullDiagram, accuracy, timeOut);
        if (phase.stability < 100 && !layoutAborted) {
          setMessage(diagram,
                     'calculating layout ' + nrNodes + ' nodes\n\n' +
                     'PHASE I accuracy 100%\n\n' +
                     'PHASE II accuracy ' + phase.stability + '%');
          diagram.show(p);
        } else {
          diagram = fullDiagram;
          phase.status = 'stable';
        }
        p.pop();
      } else {
        if (autoFit && dragging.subject != 'node') world.fit(diagram.bounds());
        
        p.push();
        world.set();
        if (dragging.subject != 'world' && !layoutAborted) diagram.refresh();
        diagram.show(p);
        if (dragging.subject == 'node') {
          // fixate the node position to the mouse point
          let wpt = world.mousePoint();
          dragging.node.drag(wpt.x-dragging.offset.x, wpt.y-dragging.offset.y);
        } else if (diagram.energyPerNode < accuracy || layoutAborted) {
          p.noLoop();
        }
        p.pop();
        showDetails();
      }
    };

    function mouseInCanvas() {
      if (!p.fixedSize) return true; // canvas spans whole window
      if (p.myWidth && !(0 <= p.mouseX && p.mouseX <= p.myWidth)) return false;
      if (p.myHeight && !(0 <= p.mouseY && p.mouseY <= p.myHeight)) return false;
      return true;
    }
    
    p.mouseWheel = function(e) {
      if (!mouseInCanvas()) return;
      world.mouseWheel(e);
      autoFit = false;
      p.redraw();
    };

    p.windowResized = function() {
      if (!p.fixedSize)
        world.resizeCanvas(p.windowWidth, p.windowHeight);
    };

    let highlightedObj = null;

    function highlight(obj) {
      if (highlightedObj) {
        highlightedObj.highlight(false);
      }
      highlightedObj = obj;
      highlightedObj.highlight(true);
      p.redraw();
    }

    function resetHighlight() {
      if (highlightedObj) {
        highlightedObj.highlight(false);
        highlightedObj = null;
        p.redraw();
      }
    }

    p.mousePressed = function(e) {
      if (!mouseInCanvas()) return;
      if (!diagram) return;
      let wpt = world.mousePoint();
      if (e.ctrlKey) {
        dragging.node = null;
        dragging.offset.x = wpt.x;
        dragging.offset.y = wpt.y;
      }  else {
        let sel = diagram.objectsAt(wpt.x, wpt.y);
        let node = sel.find(s => (s.isA('State') || s.isA('InitialState') ||s.isA('Transition')));
        if (node) {
          dragging.node = node;
          dragging.offset.x = wpt.x - dragging.node.x;
          dragging.offset.y = wpt.y - dragging.node.y;
          p.loop();
          p.redraw();
        }
      }
    };

    p.mouseReleased = function(e) {
      if (!mouseInCanvas()) return;
      dragging.node = null;
      if (dragging.subject != 'none') {
        dragging.subject = 'none';
      } else {
        let node = nodeAtMouse();
        if (node) {
          highlight(node);
          if (node.data.location) p.out.selected({...node.data.location, 'working-directory': p.data['working-directory']});
        } else {
          resetHighlight();
        }
      }
    };

    p.mouseDragged = function(e) {
      if (!mouseInCanvas()) return;
      let wpt = world.mousePoint();
      if (e.ctrlKey) {
        dragging.subject = 'world';
        world.drag(dragging.offset.x, dragging.offset.y, wpt.x, wpt.y);
        wpt = world.mousePoint();
        dragging.offset.x = wpt.x;
        dragging.offset.y = wpt.y;
        autoFit = false;
      } else if (dragging.node) {
        dragging.subject = 'node';
        dragging.node.drag(wpt.x-dragging.offset.x, wpt.y-dragging.offset.y);
        p.loop();
      }
      p.redraw();
    };

    let detail = null;

    p.mouseMoved = function(e) {
      if (!mouseInCanvas()) return;
      let node = nodeAtMouse();
      if (node != detail) {
        if (detail) {
          if(detail.isA('Transition')) detail.hover(false);
          detail = null;
        }
        if (node) {
          detail = node;
          if(detail.isA('Transition')) detail.hover(true);
        }
        p.redraw();
      }
    }

    function nodeAtMouse() {
      if (!mouseInCanvas()) return null;
      if (!diagram) return null;
      let wpt = world.mousePoint();
      let sel = diagram.objectsAt(wpt.x, wpt.y);
      let node = sel.find(s => (s.isA('State') || s.isA('InitialState') || s.isA('Transition')));
      return node;
    }

    p.keyPressed = function() {
      return true;
    };

    p.keyReleased = function() {
      if (!mouseInCanvas()) return true;
      if (p.keyCode == p.ESCAPE) {
        layoutAborted = !layoutAborted;
        if (!layoutAborted) {
          p.loop();
          p.redraw();
        }
      } else {
        // default behaviour for unbound keys:
        return true;
      }
      // suppress default behaviour:
      return false;
    };

    function showDetails() {
      if (!mouseInCanvas()) return;
      if (detail && world.scale < .75) {
//        console.log('show detail %j', detail);
        p.translate(-detail.x+10, -detail.y+10);
        detail.show(p);
      }
    }
  };

};
