/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

class SystemDiagramP5 {

  /*
   * interface to the outside world, through functions
   *   in.draw(data)
   *     show the data as a diagram
   *   in.dimensions(px, py, width, height)
   *     set the location (px, py) and size (with, height) of the
   *     diagram
   *   out.selected(location)
   *     process the selected node, using its location
   *
   *   out functions can be (re)defined by the outside world.
   */

  constructor(parent) {
    this.in = {
      draw: function (data) {
        this.sketch.data = data;
        if (this.sketch.set_up) {
          this.sketch.initDiagram(data);
          this.sketch.redraw();
        }
      }.bind(this),
      dimensions: function (px, py, width, height) {
        this.sketch.dimensions(px, py, width, height);
      }.bind(this),
    };
    this.out = {
      selected: function (location) {
        console.log('selected location: %j', location);
      },
    };

    // default dimensions: whole window
    this.px = 0;
    this.py = 0;
    this.fixedSize = false;
    this.width = window.innerWidth;
    this.height = window.innerHeight;

    // interface with sketch:
    this.sketch = new p5(this.systemSketch, parent);
    this.sketch.in = this.in;
    this.sketch.out = this.out;
    this.sketch.myPx = this.px;
    this.sketch.myPy = this.py;
    this.sketch.fixedSize = this.fixedSize;
    this.sketch.myWidth = this.width;
    this.sketch.myHeight = this.height;
  };

  systemSketch(p) {
    // interface with surrounding StateDiagramP5 object:
    p.in = null;
    p.out = null;
    p.myPx = null;
    p.myPy = null;
    p.fixedSize = false;
    p.myWidth = null;
    p.myHeight = null;

    /*
     * Visualisation is done using standard functions defined in p5js
     * Zooming and dragging are facilitated. The World object handles the
     * appropriate transformations.
     *   setup: nil --> nil
     *     called once by p5js. Do all Viz object initialisation here.
     *     noLoop() is called to prevent continuous redrawing, so function
     *     redraw needs to be called explicitely to update the screen
     *   draw: nil --> nil
     *     transform world to screen coordinates, refresh all Viz objects,
     *     and show the result
     *   mouseWheel: Event --> nil
     *     use World.mouseWheel to handle zooming
     *   mousePressed: Event --> Boolean
     *     activate dragging until mouseRelease is called.
     *   mouseReleased: Event --> Boolean
     *     stop dragging
     *     if a Button is pressed, execute its callback function, and redraw.
     *     if over a selectable object: highlight the object,
     *     otherwise reset nighlighting
     *   keyReleased: nil -> Boolean
     *     execute the function bound to the corresponding key
     */

    p.set_up = false;
    p.data = null;

    p.dimensions = function(px, py, width, height) {
      p.myPx = px;
      p.myPy = py;
      p.myWidth = width;
      p.myHeight = height;
      p.fixedSize = true;
      if (p.set_up) {
        world.positionCanvas(px, py);
        world.resizeCanvas(width, height);
        p.redraw();
      }
    };
      
    // initDiagram is accessed outside sketch
    // do not call before p5 setup!
    p.initDiagram = function(data) {
      p.data = data;
      sut = new SUT(data).instance;
      sut.initViz(false);
      sut.move(100, 100);
      sutStack.push(sut);
      p.noLoop();
    };

    let world = null;
    let sut = null;

    p.setup = function() {
      let px = p.myPx;
      let py = p.myPy;
      let w = p.fixedSize ? p.myWidth : p.windowWidth;
      let h = p.fixedSize ? p.myHeight : p.windowHeight;
      world = new World(p, w, h);
      if (px != null && py != null) world.positionCanvas(px, py);
      p.set_up = true;
      if (p.data) {
        p.initDiagram(p.data);
      } 
      p.noLoop();
    };

    p.draw = function() {
      p.background(255);
      if (!sut) return;
      p.push();
      world.set();
      sut.refresh();
      sut.show(p);
      if (sut.changing() || focusAnimating()) p.loop(); else p.noLoop();
      p.pop();
      showDetails();
      showTooltip();
    };

    function mouseInCanvas() {
      if (!p.fixedSize) return true; // canvas spans whole window
      if (p.myWidth && !(0 <= p.mouseX && p.mouseX <= p.myWidth)) return false;
      if (p.myHeight && !(0 <= p.mouseY && p.mouseY <= p.myHeight)) return false;
      return true;
    }
    
    p.mouseWheel = function(e) {
      if (!mouseInCanvas()) return;
      world.mouseWheel(e);
      p.redraw();
    };

    p.mouseMoved = function(e) {
      if (!mouseInCanvas()) return;
      let obj = objAtMouse();
      if (obj && obj.isA('Port'))
        setTooltip(obj);
      else
        setTooltip(null);
    }

    let tooltipObj = null;
    let tooltip = null;
    
    function setTooltip(obj) {
      let old = tooltipObj;
      tooltipObj = obj;
      if (tooltipObj && tooltipObj != old) {
        if (tooltipObj.isA('Port')) {
          let text = tooltipObj.name;
          tooltip = new BoundingBox(new Text(text));
          tooltip.padding = 3;
          tooltip.color = '#FFFCB0';
          tooltip.refreshMe();
        }
      }
      if (tooltipObj || old)
        p.redraw();
    }

    p.windowResized = function() {
      if (!p.fixedSize)
        world.resizeCanvas(p.windowWidth, p.windowHeight);
    };

    let dragging = false;
    let draggingOffsetX = 0;
    let draggingOffsetY = 0;
    let highlightedObj = null;
    let highlightedExtras = [];

    function highlight(obj) {
      if (highlightedObj) {
        highlightedObj.highlight(false);
      }
      resetHighlightExtras();
      highlightedObj = obj;
      highlightedObj.highlight(true);
      p.redraw();
    }

    function resetHighlight() {
      if (highlightedObj) {
        highlightedObj.highlight(false);
        highlightedObj = null;
        resetHighlightExtras();
        p.redraw();
      }
    }

    function highlightExtras(lst) {
      highlightedExtras = lst;
      highlightedExtras.forEach(h => h.highlight(true));
      p.redraw();
    }

    function resetHighlightExtras() {
      highlightedExtras.forEach(h => h.highlight(false));
      highlightedExtras = [];
    }

    function focus(obj) {
      let sbnd = sut.bounds();
      let obnd = obj.relativeBounds(sut.viz);
      // absolute coordinates:
      let oldX = (obj == sut) ? sbnd.x : sbnd.x+obnd.x;
      let oldY = (obj == sut) ? sbnd.y : sbnd.y+obnd.y;
      let newX = oldX;
      let newY = oldY;
      let topLeft = world.canvasToWorld(0, 0);
      let bottomRight = world.canvasToWorld(world.canvas.width, world.canvas.height);
      if (oldX < topLeft.x || oldX + obnd.width > bottomRight.x) newX = 100/world.scale;
      if (oldY < topLeft.y || oldY + obnd.height > bottomRight.y) newY = 100/world.scale;

      if (newX != oldX || newY != oldY) {
        startFocusAnimation(newX - oldX, newY - oldY);
      }
    }

    let nrFocusFrames = 10;
    let focusShift = {x:0, y:0};
    let focusFrame = nrFocusFrames;

    function startFocusAnimation(dx, dy) {
      focusShift = {x: 1/nrFocusFrames*dx, y: 1/nrFocusFrames*dy};
      sut.shift(focusShift.x, focusShift.y);
      focusFrame = 1;
      p.redraw();
    }

    function focusAnimating() {
      if (focusFrame == nrFocusFrames) return false;
      sut.shift(focusShift.x, focusShift.y);
      focusFrame++;
      return true;
    }

    function highlightAndFocus(obj) {
      highlight(obj);
      focus(obj);
    }

    p.mousePressed = function(e) {
      if (!mouseInCanvas()) return;
      if (!sut) return;
      if (e.ctrlKey) {
        let wpt = world.mousePoint();
        draggingOffsetX = wpt.x - sut.x;
        draggingOffsetY = wpt.y - sut.y;
      } 
    };

    p.mouseDragged = function(e) {
      if (!mouseInCanvas()) return;
      if (!sut) return;
      let wpt = world.mousePoint();
      if (e.ctrlKey) {
        dragging = true;
        sut.move(wpt.x-draggingOffsetX, wpt.y-draggingOffsetY);
        p.redraw();
      } else dragging = false;
    };

    p.mouseReleased = function(e) {
      if (dragging) {
        dragging = false;
      } else {
        if (!mouseInCanvas()) return;
        if (!sut) return;
        let obj = objAtMouse();
        if (obj) {
          if (obj.klass == 'Button') {
            obj.callback(e.ctrlKey);
            p.redraw();
          } else {
            highlight(obj);
            if (obj.klass == 'Port') {
              highlightExtras(path(obj));
              p.redraw();
            }
            let location = obj.location;
            if (p.keyIsDown(p.CONTROL)) {
              if (obj.klass == 'Instance')
                location = obj.model && obj.model.location;
              else if (obj.klass == 'Port')
                location = obj.dinterface.location;
              else if (obj.klass == 'Binding')
                location = obj.fromPort.dinterface.location;
            }
            if (location) p.out.selected({...location, 'working-directory': p.data['working-directory']});
          }
        } else {
          resetHighlight();
        }
      }
    };

    function objAtMouse() {
      if (!mouseInCanvas()) return true;
      if (!sut) return false;
      function selectionOfKlass(sel, klass) {
        let obj = sel.find(s => s.isA(klass));
        return obj;
      }
      let wpt = world.mousePoint();
      let sel = sut.objectsAt(wpt.x, wpt.y);
      return selectionOfKlass(sel, 'Button') ||
        selectionOfKlass(sel, 'Binding') ||
        selectionOfKlass(sel, 'Port') ||
        selectionOfKlass(sel, 'Instance') ||
        selectionOfKlass(sel, 'Component') ||
        selectionOfKlass(sel, 'Foreign') ||
        selectionOfKlass(sel, 'System');
    }

    function path(port) {
      function follow(port, bnd) {
        let port2 = (bnd.fromPort == port) ? bnd.toPort : bnd.fromPort;
        let result = [port2];
        if (port2.model.klass == 'System') {
          let bnd2 = (port2.model == bnd.system) ? port2.getExternalBinding() : port2.getInternalBinding();
          if (bnd2) {
            result.push(bnd2);
            result = result.concat(follow(port2, bnd2));
          }
        }
        return result;
      }
      
      let result = [port];
      let ibnd = port.getInternalBinding();
      let ebnd = port.getExternalBinding();
      if (ibnd) {
        result.push(ibnd);
        result = result.concat(follow(port, ibnd));
      }
      if (ebnd) {
        result.push(ebnd);
        result = result.concat(follow(port, ebnd));
      }
      return result;
    }

    p.keyPressed = function() {
      if (!mouseInCanvas()) return true;
      // suppress browser zooming
      if ((p.key === '-' || p.key === '+' || p.key == '=' || p.key == '1' || p.key == '0') && p.keyIsDown(p.CONTROL)) {
        return false;
      }
      return true;
    };

    p.keyReleased = function() {
      if (!mouseInCanvas()) return true;
      if (!sut) return;
      // use p5 'key' variable for ASCII keys
      // use p5 'keyCode' variable for non-ASCII keys
      if (p.keyCode == p.ENTER) {
        enterInstance();
      } else if (p.keyCode == p.BACKSPACE) {
        exitInstance();
      } else if (p.keyCode == p.DOWN_ARROW ||
                 p.keyCode == p.UP_ARROW ||
                 p.keyCode == p.RIGHT_ARROW ||
                 p.keyCode == p.LEFT_ARROW) {
        navigateInstance(p.keyCode);
      } else if (p.key === '-') {
        if (p.keyIsDown(p.CONTROL)) {
          world.zoomAround(0, 0, world.zoomOutFactor);
          p.redraw();
        } else {
          closeSystem();
        }
      } else if (p.key === '+' || p.key == '=') {
        if (p.keyIsDown(p.CONTROL)) {
          world.zoomAround(0, 0, world.zoomInFactor);
          p.redraw();
        } else {
          openSystem();
        }
      } else if (p.key === '0') {
        if (p.keyIsDown(p.CONTROL)) {
          world.fit(sut.bounds());
          p.redraw();
        } 
      } else if (p.key === '1') {
        if (p.keyIsDown(p.CONTROL)) {
          world.scale = 1;
          world.zoomAround(0, 0, 1);
          p.redraw();
        } 
      } else {
        // default behaviour for unbound keys:
        return true;
      }
      // suppress default behaviour:
      return false;
    };

    function getHighlightedInstance() {
      if (highlightedObj && highlightedObj.isA('Instance')) return highlightedObj;
      else return null;
    }

    function getHighlightedModel() {
      if (!highlightedObj) return null;
      else if (highlightedObj.isA('Instance')) return highlightedObj.model;
      else if (highlightedObj.isA('Component') ||
               highlightedObj.isA('Foreign') ||
               highlightedObj.isA('System')) return highlightedObj;
      else return null;
    }

    function getHighlightedSystem() {
      let model = getHighlightedModel();
      if (model && model.isA('System')) return model;
      else return null;
    }

    function openSystem() {
      let system = getHighlightedSystem();
      if (system && !system.isOpen) {
        system.openClose();
        p.redraw();
      }
    }

    function closeSystem() {
      let system = getHighlightedSystem();
      if (system && system.isOpen) {
        system.openClose();
        p.redraw();
      }
    }

    /* 
     * sutStack: [Instance]
     */
    let sutStack = [];

    function enterInstance() {
      let inst = getHighlightedInstance();
      if (inst) {
        sut = inst;
        sutStack.push(sut);
        highlightAndFocus(sut);
      }
    }

    function exitInstance() {
      if (sutStack.length > 1) {
        sutStack.pop();
        sut = sutStack[sutStack.length-1];
        highlightAndFocus(sut);
      }
    }

    function navigateInstance(keyCode) {
      let inst = getHighlightedInstance();
      if (inst) {
        navigate(inst, keyCode)
      } else if (!highlightedObj) {
        highlightAndFocus(sut);
      }
    }

    function navigate(instance, keyCode) {
      if (p.keyIsDown(p.CONTROL)) {
        // navigate inward or outward
        if (keyCode == p.DOWN_ARROW) {
          if (instance.model.isA('System') && instance.model.isOpen) {
            let layer = instance.model.layers[0];
            if (layer) highlightAndFocus(layer.instances[0]);
          }
        } else if (keyCode == p.UP_ARROW) {
          // do not navigate outside sut:
          if (instance != sut) highlightAndFocus(instance.parentSystem.instance);
        }
      } else {
        // navigate in current system
        // do not navigate outside sut:
        if (instance == sut) return;
        let system = instance.parentSystem;
        if (keyCode == p.DOWN_ARROW) {
          let layer = system.nextLayer(instance);
          if (layer) highlightAndFocus(layer.instances[0]);
        } else if (keyCode == p.UP_ARROW) {
          let layer = system.previousLayer(instance);
          if (layer) highlightAndFocus(layer.instances[0]);
        } else if (keyCode == p.RIGHT_ARROW) {
          let sel = system.nextInstance(instance);
          if (sel) highlightAndFocus(sel);
        } else if (keyCode == p.LEFT_ARROW) {
          let sel = system.previousInstance(instance);
          if (sel) highlightAndFocus(sel);
        }
      }
    }

    function showDetails() {
      if (highlightedObj && world.scale < .75) {
        let model = highlightedObj.isA('Instance') ? highlightedObj.model : highlightedObj;
        let summary = model.summary;
        p.translate(-summary.x+10, -summary.y+10);
        summary.show(p);
      }
    }

    function showTooltip() {
      if (tooltipObj) {
        p.push();
        let wpt = world.mousePoint();
        p.translate(p.mouseX+20, p.mouseY);
        tooltip.show(p);
        p.pop();
      }
    }
  };
};
