/*
 * Copyright (C) 2020,2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

class World {
  constructor(p, width, height) {
    this.p = p;
    this.scale = 1,
    this.x0 = 0, // in canvas dimensions
    this.y0 = 0, // in canvas dimensions
    this.canvas = p.createCanvas(width, height);
    this.zoomInFactor = 1.05;
    this.zoomOutFactor = 0.95;
  }
  
  set() {
    this.p.translate(this.x0, this.y0);
    this.p.scale(this.scale);
  }

  
  resizeCanvas(width, height) {
    this.p.resizeCanvas(width, height);
  }

  positionCanvas(px, py) {
    this.canvas.position(px, py);
  }

  mouseWheel(e) {
    if (e.ctrlKey) {
      let zoom = (e.deltaY < 0) ? this.zoomInFactor : this.zoomOutFactor;
      this.zoomAround(this.p.mouseX, this.p.mouseY, zoom);
    } else if (e.shiftKey) {
      this.x0 -= e.delta;
    } else {
      this.y0 -= e.delta;
    }
  }

  zoomAround(mx, my, zoom) {
      this.scale *= zoom;
      this.x0 = this.x0 + (this.x0 - mx)*(zoom - 1);
      this.y0 = this.y0 + (this.y0 - my)*(zoom - 1);
  }

  drag(wfromx, wfromy, wtox, wtoy) {
    console.log('drag: %s, %s, %s, %s', wfromx, wfromy, wtox, wtoy);
    let cfrom = this.world2Canvas(wfromx, wfromy);
    let cto = this.world2Canvas(wtox, wtoy);
    this.x0 += cto.x - cfrom.x;
    this.y0 += cto.y - cfrom.y;
  }

  fit(bounds) { // bounds in World dimensions
    let sc = Math.min(this.canvas.width/bounds.width, this.canvas.height/bounds.height)*.9;
    this.scale = Math.min(this.scale, sc);
    
    let worldMiddleX = bounds.x + bounds.width/2;
    let worldMiddleY = bounds.y + bounds.height/2;
    let canvasMiddleX = this.canvas.width/2;
    let canvasMiddleY = this.canvas.height/2;
    // set this.x0 and this.y0 such that
    // canvasToWorld(canvasMiddleX, canvasMiddleY) = (worldMiddleX, worldMiddleY):
    this.x0 = canvasMiddleX - worldMiddleX * this.scale;
    this.y0 = canvasMiddleY - worldMiddleY * this.scale;
  }

  limitHorizontal(bounds) { // bounds in World dimensions
    let canvasTopLeft = this.world2Canvas(bounds.x, bounds.y);
    let canvasBottomRight = this.world2Canvas(bounds.x+bounds.width, bounds.y+bounds.height);

    if (canvasBottomRight.x - canvasTopLeft.x > this.canvas.width) {
      if (canvasBottomRight.x < this.canvas.width)
        this.x0 += this.canvas.width - canvasBottomRight.x;
      if (canvasTopLeft.x > 0)
        this.x0 -= canvasTopLeft.x;
    }
  }

  limitVertical(bounds) { // bounds in World dimensions
    let canvasTopLeft = this.world2Canvas(bounds.x, bounds.y);
    let canvasBottomRight = this.world2Canvas(bounds.x+bounds.width, bounds.y+bounds.height);

    if (canvasBottomRight.y - canvasTopLeft.y > this.canvas.height) {
      if (canvasBottomRight.y < this.canvas.height)
        this.y0 += this.canvas.height - canvasBottomRight.y;
      if (canvasTopLeft.y > 0)
        this.y0 -= canvasTopLeft.y;
    } else {
      this.y0 -= canvasTopLeft.y;
    }
  }

  canvasToWorld(cx, cy) {
    return {x: (cx - this.x0)/this.scale, y: (cy - this.y0)/this.scale};
  }

  world2Canvas(wx, wy) {
    return {x: wx * this.scale + this.x0, y: wy * this.scale + this.y0};
  }

  mousePoint() {
    return this.canvasToWorld(this.p.mouseX, this.p.mouseY);
  }
};
