;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands bye)
  #:use-module (ice-9 getopt-long)
  #:use-module (ide commander)
  #:use-module (ide commands daemon)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide util))

(define (parse-opts args)
  (let* ((option-spec
          '((force (single-char #\f))
            (help (single-char #\h))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (pair? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide bye [OPTION]...
Start Dezyne session, check daemon.

  -f, --force            kill without connecting
  -h, --help             display this help and exit
")
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (force? (option-ref options 'force #f))
         (verbose? (ide:command-line:get 'verbose))
         (pid-file (string-append %user-runtime-dir "/daemon.pid"))
         (listening? (daemon:listening? %ide-port))
         (result EXIT_SUCCESS))

    (define (bye-handler message)
      (let ((bye (parse-message message)))
        (cond ((bye? bye)
               (display "bye\n"))
              (else
               (format (current-error-port)
                       "ide bye: error: received: ~s\n"
                       message)
               (set! result EXIT_FAILURE)))
        message))

    (unless listening?
      (set! result EXIT_FAILURE)
      (format (current-error-port) "no ide daemon running on port ~a\n" %ide-port))

    (let* ((commander (make-commander #:debug? debug? #:verbose? verbose?))
           (command (make-command "bye" '("dzn" "hello")))
           (result-handler
            (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                     bye-handler))
           (result (if (or force? (not listening?)) result
                       (+ result
                          (run-command commander command
                                       #:result-handler result-handler
                                       #:ide-port %ide-port
                                       #:debug? debug? #:verbose? verbose?)))))

      (when (file-exists? pid-file)
        (when (or debug? verbose?)
          (format (current-error-port) "killing daemon...\n"))
        (daemon:kill #:force? force?))

      (exit result))))
