;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2020, 2021 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands daemon)
  #:use-module ((srfi srfi-1) #:select (filter-map))
  #:use-module ((srfi srfi-26))

  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 iconv)
  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)

  #:use-module (web request)
  #:use-module (web response)
  #:use-module (web uri)

  #:use-module (oop goops)
  #:use-module (8sync)
  #:use-module (8sync systems web)

  #:use-module (ide service)
  #:use-module (ide shell-util)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide pipe)
  #:use-module (ide util)
  #:use-module (dzn runtime)
  #:use-module (dzn decoupler)
  #:use-module (daemon daemon_interface)
  #:use-module (daemon daemon)

  #:use-module (transport libtransport)
  #:use-module (transport Websocket)

  #:export (daemon:listening?
            daemon:start
            daemon:kill
            daemon:wait-for-listening))

(define (parse-opts args)
  (let* ((option-spec
          '((debug (single-char #\d))
            (help (single-char #\h))
            (editor-port (single-char #\e) (value #t))
            (http-port (single-char #\w) (value #t))
            (log-file (single-char #\l) (value #t))
            (ide-port (single-char #\i) (value #t))
            (verbose (single-char #\v))
            (view-port (single-char #\b) (value #t))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (pair? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: dezyne daemon [OPTION]...
Start Dezyne deamon.

  -d, --debug             enable debug ouput
  -b, --view-port=PORT    browser view listen PORT [~a]
  -e, --editor-port=PORT  editor listen PORT [~a]
  -h, --help              display this help and exit
  -i, --ide-port=PORT     ide command listen PORT [~a]
  -l, --log-file=LOG      write output to LOG
  -v, --verbose           be more verbose, show progress
  -w, --http-port=PORT    web server http listen PORT [~a]
" %view-port %editor-port %ide-port %http-port)
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define-actor <sleeper> (<actor>)
  ((*init* sleeper-loop))
  (seconds #:init-value 4 #:getter .seconds #:init-keyword #:seconds)
  (verbose? #:init-value 4 #:getter .verbose? #:init-keyword #:verbose?))

(define (sleeper-loop actor message)
  (while (actor-alive? actor)
    (8sleep (.seconds actor))
    (when (.verbose? actor)
      (log-format #t "-- MARK --\n"))))

(define* (daemon:start #:key debug? force? port (verbose? #t))
  "Check for a daemon.pid file.  If it does not exists, or if FORCE?,
or if PORT and not listening, start a new daemon.  When mingw? and PORT,
wait until PORT is listening."
  (let ((pid-file (string-append %user-runtime-dir "/daemon.pid")))
    (when (or (not (file-exists? pid-file))
              force?
              (and port
                   (not (daemon:listening? port))))
      (let* ((log-file (string-append %user-cache-dir "/daemon.log"))
             (pid (start-daemon '("ide" "daemon")
                                #:directory (getcwd)
                                #:log-file log-file
                                #:pid-file pid-file
                                #:debug? debug?
                                #:verbose? verbose?)))
        ;; On mingw, connecting before PORT is listening deadlocks.  So,
        ;; even though PID-file exists, really wait for PORT.
        (when (and port (mingw?))
          (unless (daemon:wait-for-listening port)
            (format (current-error-port) "no daemon listening on ~a\n" port)
            (exit EXIT_FAILURE)))
        pid))))

(define* (daemon:kill #:key force?)
  "Check for a daemon.pid file.  If it exists, read the daemon PID and
kill it.  When FORCE?, always remove the daemon.pid file."
  (let ((pid-file (string-append %user-runtime-dir "/daemon.pid")))
    (when (file-exists? pid-file)
      (let ((pid (read-pid-file pid-file #:timeout 0)))
        (when force?
          (format (current-error-port) "deleting pid-file: ~s\n" pid-file)
          (delete-file pid-file))
        (kill-daemon pid)))))

(define (daemon:listening? port)
  "Check if something is listening on localhost:PORT."
  (let* ((sock (socket PF_INET SOCK_STREAM 0))
         (address (make-socket-address AF_INET INADDR_LOOPBACK port)))
    (catch 'system-error
      (lambda _ (connect sock address))
      (const #f))))

(define* (daemon:wait-for-listening port #:key verbose? (timeout 20))
  "Wait until daemon is listening on PORT, at most TIMEOUT seconds.
When VERBOSE?, show some progress."
  (let loop ((count 20))
    (cond ((daemon:listening? port)
           (when verbose?
             (format #t "\nfound listening port: ~a\n" port)
             (newline (current-error-port)))
           #t)
          ((zero? count)
             (format #t "\nnothing listening on port: ~a\n" port)
             #f)
          (else
           (when verbose?
             (display "." (current-error-port)))
           (sleep 1)
           (loop (1- count))))))

(define* (make-http-handler #:key root debug? verbose?)
  (lambda (request body)

    (define (request-file-components)
      (split-and-decode-uri-path (uri-path (request-uri request))))

    (define (not-found)
      (when verbose?
        (format #t "404\n"))
      (values (build-response #:code 404)
	      (string-append "Resource not found: "
			     (uri->string (request-uri request)))))

    (define (extension->mime-type extension)
      (or (assoc-ref '((".css" . text/css)
                       (".html" . text/html)
                       (".js" . text/javascript))
                     extension)
          'text/plain))

    (when debug?
      (format (current-error-port) "http-handler: request=~s\n" request))

    (let ((components (request-file-components)))

      (define (serve-file file-name)
        (let* ((file-name (string-append root "/" file-name))
               (content (false-if-exception (with-input-from-file file-name read-string)))
               (extension (file-name-extension file-name))
               (mime-type (extension->mime-type extension)))
          (if (not content) (not-found)
              (begin
                (when (or debug? verbose?)
                  (format #t "GET ~a\n" file-name))
                (values `((content-type . (,mime-type))) content)))))

      (when verbose?
        (log-format #t ""))
      (match components
        (()
         (serve-file "index.html"))
        (((and (or "state" "system" "trace") base))
         (serve-file (string-append base ".html")) )
        (((? string?) ...)
         (serve-file (string-join components "/")))
        (_ (not-found))))))

(define* (cleanup #:key debug?)
  (let ((pid-file (string-append %user-runtime-dir "/daemon.pid")))
    (when (defined? 'SIGHUP)
      (sigaction SIGHUP SIG_DFL))
    (sigaction SIGINT SIG_DFL)
    (sigaction SIGTERM SIG_DFL)
    (when (file-exists? pid-file)
      (when debug?
        (format (current-error-port) "deleting pid-file: ~s\n" pid-file))
      (delete-file pid-file))
    (primitive-exit 1)))

(define (illegal-handler)
  (log-format #t "caught illegal, exiting\n")
  (display-backtrace (make-stack #t) (current-error-port))
  (cleanup))

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (locator (make <dzn:locator>))
         (debug? (or (command-line:get 'debug)
                     (ide:command-line:get 'debug)))
         (verbose? (or (command-line:get 'verbose)
                       (ide:command-line:get 'verbose)))
         (debugity (+ (length (ide:multi-opt 'debug))
                      (length (multi-opt options 'debug))))
         (locator (if (> debugity 1) locator
                      (dzn:set! locator (const #t) 'trace)))
         (locator (dzn:set! locator (if (or debug? verbose?) log-verbose (const #t)) 'log))
         (locator (dzn:set! locator (if (or debug? verbose?) log-debug (const #t)) 'log-debug))
         (verbosity (+ (length (ide:multi-opt 'verbose))
                       (length (multi-opt options 'verbose))))
         (locator (if (zero? verbosity) locator
                      (dzn:set! locator verbosity 'verbosity)))
         (runtime (make <dzn:runtime> #:illegal illegal-handler))
         (locator (dzn:set! locator runtime))
         (hive (make-hive))
         (sleeper (bootstrap-actor hive <sleeper> #:seconds 300 #:verbose? #t))
         (locator (dzn:set! locator hive))
         (decoupler (bootstrap-actor hive <decoupler>))
         (decoupler-actor ((@@ (8sync actors) hive-resolve-local-actor) hive decoupler))
         (locator (dzn:set! locator decoupler-actor))
         (daemon (make <daemon:Daemon> #:locator locator #:name "daemon"))
         (http-port (or (and=> (option-ref options 'http-port #f) string->number)
                        %http-port))
         (http-handler (make-http-handler #:root %web-root #:debug? (> debugity 2) #:verbose? verbose?))
         (web-server (bootstrap-actor hive <web-server>
                                      #:host "127.0.0.1"
                                      #:port http-port
                                      #:http-handler http-handler))
         (editor-port (or (and=> (option-ref options 'editor-port #f) string->number)
                          %editor-port))
         (ide-port (or (and=> (option-ref options 'ide-port #f) string->number)
                       %ide-port))
         (view-port (or (and=> (option-ref options 'view-port #f) string->number)
                        %view-port))
         (ide-config ide-port)
         (view-config view-port)
         (editor-config editor-port)
         (log-file (option-ref options 'log-file #f)))

    (update-daemon-config-file #:http-port http-port
                               #:ide-port ide-port
                               #:view-port view-port
                               #:editor-port editor-port)
    (set! (.set_up (.out (.start daemon)))
          (lambda _
            (define (signal-handler signal)
              (log-format #t "caught signal ~a, exiting\n" signal)
              (cleanup #:debug? debug?))
            (when verbose?
              (log-format #t "setup done\n"))
            (when (defined? 'SIGHUP)
              (sigaction SIGHUP signal-handler))
            (sigaction SIGINT signal-handler)
            (sigaction SIGTERM signal-handler)
            (mkdir-p %user-runtime-dir)
            (let ((pid-file (string-append %user-runtime-dir "/daemon.pid")))
              (with-output-to-file pid-file
                (cut display (getpid))))
            (<- decoupler 'decouple
                (lambda _ #t))))

    (set! (.error (.out (.start daemon)))
          (lambda (e)
            (format (current-error-port) "daemon error -- exiting\n")
            (primitive-exit 1))) ;; FIXME: stop hive

    (when log-file
      (mkdir-p (dirname log-file))
      (let ((log-port (open-file log-file "a")))
        (set-unbuffered! log-port)
        (set-current-output-port log-port)
        (set-current-error-port log-port)))

    (run-hive hive (list (bootstrap-message
                          hive decoupler 'decouple
                          (lambda _
                            (action daemon .start .in .setup ide-config view-config editor-config)))))))
