/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

class SequenceDiagramP5 extends DiagramP5 {
  
  constructor(parent) {
    super(parent);
    /*
     * super defines interface to the outside world, through functions
     *   in.draw(data)
     *   in.dimensions(px, py, width, height)
     *   out.selected(location)
     *
     * extension:
     */
    this.out.event = function (event) {
        console.log('event: %j', event);
    };

    this.message = null;
    
    this.nrFocusFrames = 10;
    this.focusShift = {x:0, y:0};
    this.focusFrame = this.nrFocusFrames;
    
    this.activeEventIndex = 0;
    this.highlightedEvent = null;
    this.activeLifelineIndex = -1;
    this.highlightedLifeline = null;
  }
  
  /*
   * These abstrace methods have to be defined in this class:
   *
   * initDiagram(data) { }
   * draw(p) { }
   * selection(px, py) { }
   * handleKey(p) { }
   * handleMouseClick(p, e) { }
   * dragIt(px, py) { }
   */

  initDiagram(data) {
    try {
      this.data = data;
      this.diagram = new SequenceDiagram(data);
      this.diagram.initViz();
      this.diagram.shift(10,0);
      this.shiftToBottom();
      this.activeEventIndex = this.diagram.body.events.length-1;
      this.highlightActiveEvent();
      this.sketch.cursor(this.sketch.ARROW);
      this.sketch.noLoop();
    } catch(e) {
      this.diagram = null;
      this.message = new BoundingBox(new Text(''));
      this.message.padding = 30;
      this.message.content.size = 30;
      this.message.color = '#F37361';
      this.setMessage(this.message, 'invalid input: see console [F12] for details');
      console.log('%j: %j', e, data);
      this.sketch.noLoop();
    }
  }
  
  setMessage(diagram, msg) {
    diagram.content.text = msg;
    diagram.content.refresh();
    diagram.center(this.sketch.windowWidth/2, this.sketch.windowHeight/2);
    diagram.refresh();
  }
  
  draw(p) {
    // for svg: use this.world.graphics to generate image i.s.o. p
    let gr = this.world.graphics || p;
    gr.clear();
    gr.background(255);
    gr.push();
    this.world.set(gr);
    if (this.diagram) {
      gr.push();
      gr.translate(this.diagram.x, this.diagram.y);
      this.diagram.body.show(gr);
      gr.pop();
      // always at top of screen:
      gr.push();
      let w0 = this.world.canvasToWorld(0,0);
      gr.translate(this.diagram.x, w0.y);
      this.diagram.header.show(gr);
      gr.pop();
      if (this.focusAnimating()) {
        p.loop();
      } else {
        p.noLoop();
        //FIXME: repeat must stop when browser loses focus
        //When Chromium loses focus, the repeat happily continues
        //making Emacs switch windows without much possibility to
        //switch back to Chromium
        //this.checkRepeatKey(p);
      }
    } else if (this.message) {
      this.message.show(gr);
      p.noLoop();
    }
    gr.pop();
    if (this.world.graphics)
      p.image(gr, 0, 0);
  }
  
  selection(px, py) {
    let sel = this.diagram.objectsAt(px, py);
    return this.selectionOfKlass(sel, 'Header')
      || this.selectionOfKlass(sel, 'HeaderGroup')
      || this.selectionOfKlass(sel, 'Button')
      || this.selectionOfKlass(sel, 'Event')
      || this.selectionOfKlass(sel, 'Lifeline');
  }
  
  lastEvent() {
    let nrEvents = this.diagram.body.events.length;
    return (nrEvents > 0) ? this.diagram.body.events[nrEvents-1] : null;
  }
  
  hasMatchingLifeline(activity, lifeline) {
    return activity && lifeline && activity.lifeline.index == lifeline.index;
  }
  
  selectNextActiveEvent(down) {
    let matches = function(eventIndex, lifeline) {
      if (!lifeline) return true;
      let event = this.diagram.body.events[eventIndex];
      return this.hasMatchingLifeline(event.from, lifeline) || this.hasMatchingLifeline(event.to, lifeline);
    }.bind(this);
    let nrEvents = this.diagram.body.events.length;
    if (down) {
      if (this.activeEventIndex < nrEvents-1)
        this.activeEventIndex++;
      while (this.activeEventIndex < nrEvents-1 && !matches(this.activeEventIndex, this.highlightedLifeline))
        this.activeEventIndex++;
    } else {
      if (this.activeEventIndex >= 0)
        this.activeEventIndex--;
      while (this.activeEventIndex >= 0 && !matches(this.activeEventIndex, this.highlightedLifeline))
        this.activeEventIndex--;
    }
    this.highlightActiveEvent();
    this.sketch.redraw();
  }

  selectActivity(event, selectedLifeline) {
    if(event.from && selectedLifeline &&
       event.from.lifeline.index == selectedLifeline.index)
      return event.from;
    else(event.to && selectedLifeline &&
         event.to.lifeline.index == selectedLifeline.index)
      return event.to;
    return null;
  }

  highlightActiveEvent() {
    if (this.highlightedEvent) {
      this.highlightedEvent.highlight(false);
    }
    if (this.activeEventIndex >= 0) {
      this.highlightedEvent = this.diagram.body.events[this.activeEventIndex];
      this.highlightedEvent.highlight(true);
      this.focus(this.highlightedEvent);
      let activity = this.selectActivity(this.highlightedEvent, this.highlightedLifeline);
      if(activity) {
        let location = activity.location;
        if (location) this.out.selected({...location,
                                         'working-directory': this.data['working-directory']});
      }
    } else {
      this.highlightedEvent = null;
      // do not change focus upon deselect
      // this.focus(null);
    }
    let index = Math.min(this.activeEventIndex+1, this.diagram.states.length-1);
    this.diagram.setActive(index);
  }
  
  resetHighlightEvent() {
    this.activeEventIndex = -1;
    this.highlightActiveEvent();
  }
  
  setActiveLifeline(index) {
    // deselect by second activate:
    this.activeLifelineIndex = (this.activeLifelineIndex == index) ? -1 : index;
    
    if (this.highlightedLifeline) {
      this.highlightedLifeline.highlight(false);
    }
    if (this.activeLifelineIndex >= 0) {
      this.highlightedLifeline = this.diagram.body.findLifeline(this.activeLifelineIndex);
      this.highlightedLifeline.highlight(true);
      this.sketch.redraw();
    } else {
      this.highlightedLifeline = null;
    }
  }
  
  focus(obj) {
    // obj isA Event or null 
    let topLeft = this.world.canvasToWorld(0, 0);
    let bottomRight = this.world.canvasToWorld(this.world.canvas.width, this.world.canvas.height);
    let oldX = 0;
    let oldY = 0;
    let newX = 0;
    let newY = 0;
    let margin = 10/this.world.scale;
    let dbnd = this.diagram.bounds();
    let bodybnd = this.diagram.body.bounds();
    let headerbnd = this.diagram.header.bounds();
    if (obj) {
      let objbnd = obj.relativeBounds(this.diagram.viz);
      // extend object bounds with the width of both headers:
      let extendWidth = function(hdr) {
        let bnd = hdr.relativeBounds(this.diagram.viz);
        if (bnd.x < objbnd.x) {
          let diff = objbnd.x - bnd.x;
          objbnd.x -= diff;
          objbnd.width += diff;
        }
        if (bnd.x + bnd.width > objbnd.x + objbnd.width) {
          let diff = (bnd.x + bnd.width) - (objbnd.x + objbnd.width);
          objbnd.width += diff;
        }
      }.bind(this);
      if (obj.from) {
        let hdr = obj.from.lifeline.header;
        extendWidth(hdr);
      }
      if (obj.to) {
        let hdr = obj.to.lifeline.header;
        extendWidth(hdr);
      }
      // absolute coordinates obj top:
      oldX = dbnd.x + objbnd.x;
      newX = (objbnd.width > (bottomRight.x - topLeft.x)) ? (bottomRight.x - topLeft.x)/2 - objbnd.width/2
        : (oldX < topLeft.x) ? topLeft.x + margin
        : (oldX + objbnd.width > bottomRight.x) ? bottomRight.x - objbnd.width - margin
        : oldX;
      oldY = dbnd.y + objbnd.y;
      newY = (oldY < topLeft.y + headerbnd.height) ? topLeft.y + headerbnd.height + margin
        : (oldY + objbnd.height > bottomRight.y) ? bottomRight.y - objbnd.height - margin
        : oldY;
      if (obj.index == this.diagram.body.events.length-1) {
        // scroll down the bottom of body, to show the eligibles
        oldY = dbnd.y + dbnd.height;
        newY = (oldY > bottomRight.y) ? bottomRight.y - margin : oldY;
      }
    } else {
      // assure right side header is left-aligned to view,
      oldX = dbnd.x;
      newX = (oldX + dbnd.width > bottomRight.x) ? bottomRight.x - dbnd.width - margin
        : oldX;
      newX = (newX < topLeft.x) ? topLeft.x + margin
        : newX;
      // scroll up to show body top
      oldY = dbnd.y;
      newY = (oldY < topLeft.y + dbnd.height) ? headerbnd.y + headerbnd.height + margin
        : oldY;
    }
    if (newX != oldX || newY != oldY) {
      this.startFocusAnimation(newX - oldX, newY - oldY);
    }
  }
  
  shiftToBottom() {
    let bottomRight = this.world.canvasToWorld(this.world.canvas.width, this.world.canvas.height);
    let margin = 10/this.world.scale;
    let bodybnd = this.diagram.body.bounds();
    // scroll up to show body bottom
    let oldY = bodybnd.y + bodybnd.height;
    if (oldY > bottomRight.y) {
      let newY = bottomRight.y - margin;
      this.diagram.shift(0, newY - oldY);
    }
  }
  
  startFocusAnimation(dx, dy) {
    this.focusShift = {x: 1/this.nrFocusFrames*dx, y: 1/this.nrFocusFrames*dy};
    this.diagram.shift(this.focusShift.x, this.focusShift.y);
    //this.diagram.header.shift(this.focusShift.x, 0);
    this.focusFrame = 1;
  }
  
  focusAnimating() {
    if (this.focusFrame == this.nrFocusFrames) return false;
    this.diagram.shift(this.focusShift.x, this.focusShift.y);
    //this.diagram.header.shift(this.focusShift.x, 0);
    this.focusFrame++;
    return true;
  }
  
  limitVertical() {
    // limit vertical scrolling
    let dbnd = this.diagram.bounds();
    let bnd = {x: dbnd.x, y: dbnd.y, width: dbnd.width, height: dbnd.height};
    // allow some space at the bottom
    bnd.height += 15;
    this.world.limitVertical(bnd);
  }
  
  handleKey(p) {
    // use p5 'key' variable for ASCII keys
    // use p5 'keyCode' variable for non-ASCII keys
    if (p.keyCode == p.DOWN_ARROW) {
      this.selectNextActiveEvent(true); // down
    } else if (p.keyCode == p.UP_ARROW) {
      this.selectNextActiveEvent(false); // up
    } else if (p.key === '-') {
      if (p.keyIsDown(p.CONTROL)) {
        this.world.zoomAround(0, 0, this.world.zoomOutFactor);
        p.redraw();
      } 
    } else if (p.key === '+' || p.key == '=') {
      if (p.keyIsDown(p.CONTROL)) {
        this.world.zoomAround(0, 0, this.world.zoomInFactor);
        p.redraw();
      } 
    } else if (p.key === '0') {
      if (p.keyIsDown(p.CONTROL)) {
        this.world.scale = 1;
        this.world.fit(this.diagram.header.bounds());
        this.limitVertical();
        p.redraw();
      } 
    } else if (p.key === '1') {
      if (p.keyIsDown(p.CONTROL)) {
        this.world.scale = 1;
        this.world.zoomAround(0, 0, 1);
        this.limitVertical();
        p.redraw();
      } 
    } else if (p.key == 's') {
      if (p.keyIsDown(p.CONTROL)) {
        this.saveAsSvg(p, 'sequence.svg');
      }
    } else {
      // default behaviour for unbound keys:
      return;
    }
    // suppress default behaviour:
    return false;
  }

  startTimeout() {
    this.timeout.running = true;
    this.timeout.wait = this.timeout.delay * 15; // initial delay is larger
    this.timeout.start = Date.now();
  }
  
  stopTimeout() {
    this.timeout.running = false;
    if (this.timeout.handler != null) {
      clearTimeout(this.timeout.handler);
      this.timeout.handler = null;
    }
  }
  
  checkRepeatKey(p) {
    if (this.timeout.running && this.timeout.handler == null) {
      let now = Date.now();
      let delta = now - this.timeout.start;
      let rerun = function() {
        // restart timer:
        this.timeout.start = Date.now();
        this.timeout.wait = this.timeout.delay;
        this.timeout.handler = null;
        this.handleKey(p);
      }.bind(this);
      this.timeout.handler = setTimeout(rerun, this.timeout.wait - delta);
    }
  }
  
  dragIt(px, py) {
    // only drag whole diagram....
    if (this.drag.ctrl) {
      this.diagram.move(px, py);
      // limit vertical dragging for body:
      let topLeft = this.world.canvasToWorld(0, 0);
      let bottomRight = this.world.canvasToWorld(this.world.canvas.width, this.world.canvas.height);
      let margin = 10/this.world.scale;
      let bnd = this.diagram.bounds();
      // too high? move down to fit bottom:
      if (bnd.y + bnd.height < bottomRight.y)
        this.diagram.move(bnd.x, bottomRight.y - bnd.height - margin);
      // too low? move up to fit top:
      bnd = this.diagram.bounds();
      if (bnd.y > 0)
        this.diagram.move(bnd.x, 0);
    }
  }
  
  handleMouseClick(p, e) {
    let wpt = this.world.mousePoint();
    // take care: handle body and header separately
    this.resetHighlight();
    let obj = this.selection(wpt.x, wpt.y);
    if (obj) {
      if (obj.klass == 'Button') {
        obj.callback();
        p.cursor(p.WAIT);
        this.out.event (obj.manager.text);
      } else if (obj.klass == 'Event') {
        this.activeEventIndex = obj.index;
        this.highlightActiveEvent();
      } else if (obj.klass == 'Header') {
        this.setActiveLifeline(obj.lifeline.index);
      } else if (obj.klass == 'HeaderGroup') {
        // noop
      } else if (obj.klass == 'Lifeline') {
        this.setActiveLifeline(obj.index);
      } 
    } else {
      this.resetHighlightEvent();
    }
    p.redraw();
  }

  // override super
  mouseWheel(p, e) {
    if (!this.mouseInCanvas(p)) return;
    if (!this.diagram) return;
    // HACK: override this.world.mouseWheel to force zooming around y=0
    if (e.ctrlKey) {
      let zoom = (e.deltaY < 0) ? this.world.zoomInFactor : this.world.zoomOutFactor;
      this.world.zoomAround(p.mouseX, 0, zoom);
    } else {
      this.world.mouseWheel(e);
    }
    this.limitVertical();
    p.redraw();
    return false;
  }
}
