/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

class SystemDiagramP5 extends DiagramP5 {
  
  constructor(parent) {
    super(parent);
    /*
     * super defines interface to the outside world, through functions
     *   in.draw(data)
     *   in.dimensions(px, py, width, height)
     *   out.selected(location)
     *
     * extension: none
     */
    
    this.tooltipObj = null;
    this.tooltip = null;
    this.highlightedExtras = [];
    
    this.nrFocusFrames = 10;
    this.focusShift = {x:0, y:0};
    this.focusFrame = this.nrFocusFrames;
    
    this.diagramStack = []; // [Instance]
  }  
  
  /*
   * These abstrace methods have to be defined in this class:
   *
   * initDiagram() { }
   * draw(p) { }
   * selection(px, py) { }
   * handleKey(p) { }
   * handleMouseClick(p, e) { }
   * dragIt(px, py) { }
   * help() { }
   */

  initDiagram() {
    this.diagram = new SUT(this.data).instance;
    this.diagram.initViz(false);
    this.diagram.move(100, 100);
    this.diagramStack.push(this.diagram);
    this.setCursor(this.sketch, 'default');
    this.sketch.noLoop();
    this.helpMessage.initViz();
  }
  
  draw(p) {
    // for svg: use this.world.graphics to generate image i.s.o. p
    let gr = this.world.graphics || p;
    gr.clear();
    gr.background(255);
    if (!this.diagram) return;
    gr.push();
    this.world.set(gr);
    this.diagram.refresh();
    this.diagram.show(gr);
    if (this.diagram.changing() || this.focusAnimating()) p.loop();
    else {
      p.noLoop();
      this.checkRepeatKey(p);
    }
    gr.pop();
    this.showDetails(gr);
    this.showTooltip(gr, p.mouseX, p.mouseY);
    if (this.showHelp) {
      gr.push();
      gr.translate(10,10);
      this.helpMessage.show(gr);
      gr.pop();
    }
    if (this.world.graphics)
      p.image(gr, 0, 0);
  }
  
  mouseMoved(p, e) {
    super.mouseMoved(p, e);
    if (!this.mouseInCanvas(p)) return;
    let obj = this.objAtMouse(p);
    if (obj && obj instanceof Port)
      this.setTooltip(obj);
    else
      this.setTooltip(null);
  }
  
  setTooltip(obj) {
    let old = this.tooltipObj;
    this.tooltipObj = obj;
    if (this.tooltipObj && this.tooltipObj != old) {
      if (this.tooltipObj instanceof Port) {
        let text = this.tooltipObj.name;
        this.tooltip = new BoundingBox(new Text(text));
        this.tooltip.padding = 3;
        this.tooltip.color = '#FFFCB0';
        this.tooltip.refreshMe();
      }
    }
    if (this.tooltipObj || old)
      this.redrawNeeded = true;
  }
  
  highlight(obj) {
    super.highlight(obj);
    this.resetHighlightExtras();
  }
  
  resetHighlight() {
    super.resetHighlight();
    this.resetHighlightExtras();
    this.redrawNeeded = true;
  }
  
  highlightExtras(lst) {
    this.highlightedExtras = lst;
    this.highlightedExtras.forEach(h => h.highlight(true));
    this.redrawNeeded = true;
  }
  
  resetHighlightExtras() {
    this.highlightedExtras.forEach(h => h.highlight(false));
    this.highlightedExtras = [];
  }
  
  focus(obj) {
    let sbnd = this.diagram.scaledBounds();
    let obnd = obj.relativeBounds(this.diagram.viz);
    // absolute coordinates:
    let oldX = (obj == this.diagram) ? sbnd.x : sbnd.x+obnd.x;
    let oldY = (obj == this.diagram) ? sbnd.y : sbnd.y+obnd.y;
    let newX = oldX;
    let newY = oldY;
    let topLeft = this.world.canvasToWorld(0, 0);
    let bottomRight = this.world.canvasToWorld(this.world.canvas.width, this.world.canvas.height);
    if (oldX < topLeft.x || oldX + obnd.width > bottomRight.x) newX = 100/this.world.scale;
    if (oldY < topLeft.y || oldY + obnd.height > bottomRight.y) newY = 100/this.world.scale;
    
    if (newX != oldX || newY != oldY) {
      this.startFocusAnimation(newX - oldX, newY - oldY);
    }
  }
  
  startFocusAnimation(dx, dy) {
    this.focusShift = {x: 1/this.nrFocusFrames*dx, y: 1/this.nrFocusFrames*dy};
    this.diagram.shift(this.focusShift.x, this.focusShift.y);
    this.focusFrame = 1;
    this.redrawNeeded = true;
  }
  
  focusAnimating() {
    if (this.focusFrame == this.nrFocusFrames) return false;
    this.diagram.shift(this.focusShift.x, this.focusShift.y);
    this.focusFrame++;
    return true;
  }
  
  highlightAndFocus(obj) {
    this.highlight(obj);
    this.focus(obj);
  }
  
  handleMouseClick(p, e) {
    let obj = this.objAtMouse(p);
    if (obj) {
      if (obj instanceof Button) {
        obj.callback(e.ctrlKey);
        this.redrawNeeded = true;
      } else {
        this.highlight(obj);
        if (obj instanceof Port) {
          this.highlightExtras(this.path(obj));
          this.redrawNeeded = true;
        }
        let location = obj.location;
        if (p.keyIsDown(p.CONTROL)) {
          if (obj instanceof Instance)
            location = obj.model && obj.model.location;
          else if (obj instanceof Port)
            location = obj.dinterface.location;
          else if (obj instanceof Binding)
            location = obj.fromPort.dinterface.location;
        }
        if (location) this.out.selected({...location, 'working-directory': this.data['working-directory']});
      }
    } else {
      this.resetHighlight();
    }
  }
  
  objAtMouse(p) {
    if (!this.mouseInCanvas(p)) return null;
    if (!this.diagram) return null;
    let wpt = this.world.mousePoint();
    return this.selection(wpt.x, wpt.y);
  }

  selection(px, py) {
    let sel = this.diagram.objectsAt(px, py);
    return this.selectionOfKlass(sel, Button) ||
      this.selectionOfKlass(sel, Binding) ||
      this.selectionOfKlass(sel, Port) ||
      this.selectionOfKlass(sel, Instance) ||
      this.selectionOfKlass(sel, Component) ||
      this.selectionOfKlass(sel, Foreign) ||
      this.selectionOfKlass(sel, System);
  }
  
  path(port) {
    function follow(port, bnd) {
      let port2 = (bnd.fromPort == port) ? bnd.toPort : bnd.fromPort;
      let result = [port2];
      if (port2.model instanceof System) {
        let bnd2 = (port2.model == bnd.system) ? port2.getExternalBinding() : port2.getInternalBinding();
        if (bnd2) {
          result.push(bnd2);
          result = result.concat(follow(port2, bnd2));
        }
      }
      return result;
    }
    
    let result = [port];
    let ibnd = port.getInternalBinding();
    let ebnd = port.getExternalBinding();
    if (ibnd) {
      result.push(ibnd);
      result = result.concat(follow(port, ibnd));
    }
    if (ebnd) {
      result.push(ebnd);
      result = result.concat(follow(port, ebnd));
    }
    return result;
  }
  
  help() {
    let help = [
      ['', 'help'],
      ['F1 or ?', 'toggle this help pop-up'],
      ['','zooming and scrolling'],
      ['ctrl mouse scroll', 'zoom in or out around the mouse pointer location'],
      ['ctrl +','zoom in around the mouse pointer location'],
      ['ctrl -','zoom out around the mouse pointer location'],
      ['ctrl 1','reset the zoom factor to 1'],
      ['ctrl 0','zoom such that the whole diagram fits on the canvas'],
      ['mouse scroll','scroll the diagram up or down'],
      ['shift mouse scroll','scroll the diagram left or right'],
      ['','dragging'],
      ['ctrl mouse drag','drag the canvas'],
      ['','selecting'],
      ['mouse click element','select the element and send its file location to the IDE:'],
      ['    instance','select the instance'],
      ['    port','select the port, and all connected bindings and ports'],
      ['    binding','select the binding'],
      ['ctrl mouse click element','select as above, but send the following file location to the IDE:'],
      ['    instance','the defining component'],
      ['    port','the defining interface'],
      ['    binding','the defining interface'],
      ['(ctrl) mouse click canvas','deselect any element'],
      ['', 'system opening and closing'],
      ['mouse click on system [-] or [+] button','close or open the system'],
      ['ctrl mouse click on system [-] or [+] button','close or open the system and all its sub systems'],
      ['', 'component navigation'],
      ['ctrl down arrow','select first sub component of the selected system'],
      ['ctrl up arrow','select the parent system of the selected component'],
      ['right arrow','select the component to the right of the selected component'],
      ['left arrow','select the component to the left of the selected component'],
      ['down arrow','select the first component in the row below the selected component'],
      ['up arrow','select the first component in the row above the selected component'],
      ['- on system selection','close the system'],
      ['+ on system selection','open the system'],
      ['enter','limit the view to the selected component'],
      ['backspace','undo the enter effect'],
      ['','save'],
      ['ctrl s','save the diagram as an svg file'],
    ];
    return new Help(help);
  }
  
  handleKey(p) {
    // use p5 'key' variable for ASCII keys
    // use p5 'keyCode' variable for non-ASCII keys
    if (p.keyCode == 112 || p.key == '?') { // 112: F1 key
      this.showHelp = ! this.showHelp;
      this.redrawNeeded = true;
    } else if (p.keyCode == p.ENTER) {
      // hack: avoid checkRepeatKey loop, since keyReleased is not detected
      this.stopTimeout();
      this.enterInstance();
    } else if (p.keyCode == p.BACKSPACE) {
      this.exitInstance();
    } else if (p.keyCode == p.DOWN_ARROW ||
               p.keyCode == p.UP_ARROW ||
               p.keyCode == p.RIGHT_ARROW ||
               p.keyCode == p.LEFT_ARROW) {
      this.navigateInstance(p, p.keyCode);
    } else if (p.key === '-' && !p.keyIsDown(p.CONTROL)) {
      this.closeSystem();
    } else if ((p.key === '+' || p.key == '=') && !p.keyIsDown(p.CONTROL)) {
      this.openSystem();
    } else if (p.key === '-' && p.keyIsDown(p.CONTROL)) {
      this.world.zoomAround(p.mouseX, p.mouseY, this.world.zoomOutFactor);
      this.redrawNeeded = true;
    } else if ((p.key === '+' || p.key == '=') && p.keyIsDown(p.CONTROL)) {
      this.world.zoomAround(p.mouseX, p.mouseY, this.world.zoomInFactor);
      this.redrawNeeded = true;
    } else if (p.key === '0' && p.keyIsDown(p.CONTROL)) {
      this.world.fit(this.diagram.scaledBounds());
      this.redrawNeeded = true;
    } else if (p.key === '1' && p.keyIsDown(p.CONTROL)) {
      this.world.zoomAround(p.mouseX, p.mouseY, 1/this.world.scale);
      this.redrawNeeded = true;
    } else if (p.key == 's') {
      if (p.keyIsDown(p.CONTROL)) {
        this.saveAsSvg(p, 'system.svg');
      }
    } else {
      // default behaviour for unbound keys:
      return;
    }
    // suppress default behaviour:
    return false;
  }
  
  getHighlightedInstance() {
    if (this.highlightedObj && this.highlightedObj instanceof Instance) return this.highlightedObj;
    else return null;
  }
  
  getHighlightedModel() {
    if (!this.highlightedObj) return null;
    else if (this.highlightedObj instanceof Instance) return this.highlightedObj.model;
    else if (this.highlightedObj instanceof Component ||
             this.highlightedObj instanceof Foreign ||
             this.highlightedObj instanceof System) return this.highlightedObj;
    else return null;
  }
  
  getHighlightedSystem() {
    let model = this.getHighlightedModel();
    if (model && model instanceof System) return model;
    else return null;
  }
  
  openSystem() {
    let system = this.getHighlightedSystem();
    if (system && !system.isOpen) {
      system.openClose();
      this.redrawNeeded = true;
    }
  }
  
  closeSystem() {
    let system = this.getHighlightedSystem();
    if (system && system.isOpen) {
      system.openClose();
      this.redrawNeeded = true;
    }
  }
  
  enterInstance() {
    let inst = this.getHighlightedInstance();
    if (inst) {
      this.diagram = inst;
      this.diagramStack.push(this.diagram);
      this.highlightAndFocus(this.diagram);
    }
  }
  
  exitInstance() {
    if (this.diagramStack.length > 1) {
      this.diagramStack.pop();
      this.diagram = this.diagramStack[this.diagramStack.length-1];
      this.highlightAndFocus(this.diagram);
    }
  }
  
  navigateInstance(p, keyCode) {
    let inst = this.getHighlightedInstance();
    if (inst) {
      this.navigate(inst, p, keyCode)
    } else if (!this.highlightedObj) {
      this.highlightAndFocus(this.diagram);
    }
  }
  
  navigate(instance, p, keyCode) {
    if (p.keyIsDown(p.CONTROL)) {
      // navigate inward or outward
      if (keyCode == p.DOWN_ARROW) {
        if (instance.model instanceof System && instance.model.isOpen) {
          let layer = instance.model.layers[0];
          if (layer) this.highlightAndFocus(layer.instances[0]);
        }
      } else if (keyCode == p.UP_ARROW) {
        // do not navigate outside diagram:
        if (instance != this.diagram) this.highlightAndFocus(instance.parentSystem.instance);
      }
    } else {
      // navigate in current system
      // do not navigate outside diagram:
      if (instance == this.diagram) return;
      let system = instance.parentSystem;
      if (keyCode == p.DOWN_ARROW) {
        let layer = system.nextLayer(instance);
        if (layer) this.highlightAndFocus(layer.instances[0]);
      } else if (keyCode == p.UP_ARROW) {
        let layer = system.previousLayer(instance);
        if (layer) this.highlightAndFocus(layer.instances[0]);
      } else if (keyCode == p.RIGHT_ARROW) {
        let sel = system.nextInstance(instance);
        if (sel) this.highlightAndFocus(sel);
      } else if (keyCode == p.LEFT_ARROW) {
        let sel = system.previousInstance(instance);
        if (sel) this.highlightAndFocus(sel);
      }
    }
  }

  dragIt(px, py) {
    if (this.drag.ctrl) {
      // dragged the canvas
      // nothing diagram specific
    }
  }
  
  showDetails(p) {
    if (this.highlightedObj && this.world.scale < .75) {
      let model = this.highlightedObj instanceof Instance ? this.highlightedObj.model : this.highlightedObj;
      let summary = model.summary;
      p.push();
      p.translate(-summary.bounds.x+10, -summary.bounds.y+10);
      summary.show(p);
      p.pop();
    }
  }
  
  showTooltip(p, mx, my) {
    if (this.tooltipObj) {
      p.push();
      p.translate(mx+20, my);
      this.tooltip.show(p);
      p.pop();
    }
  }
}
