;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2019,2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2019 Rob Wieringa <Rob.Wieringa@verum.com>
;;; Copyright © 2020 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;; Copyright © 2020 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (daemon Dzn_proxy)
  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)
  #:use-module (srfi srfi-1)
  #:use-module ((srfi srfi-26) #:select (cute))
  #:use-module ((srfi srfi-71))
  #:use-module (dzn runtime)
  #:use-module (dzn decoupler)
  #:use-module (oop goops)
  #:use-module (daemon daemon_interface)
  #:use-module (ide pipe)
  #:use-module (ide util)
  #:use-module (ide port-util)
  #:use-module (8sync)
  #:use-module (8sync actors)
  #:use-module (transport libenvelope)

  #:use-module (dzn ast)
  #:use-module (dzn parse)
  #:use-module (dzn simulate)
  #:use-module (dzn goops)
  #:use-module (dzn vm goops)
  #:use-module (dzn vm normalize)
  #:use-module (dzn vm run)
  #:use-module (dzn vm runtime)

  #:export (<daemon:Dzn_proxy> .dz))

(define-class <daemon:Dzn_proxy> (<dzn:component>)
  (dz #:accessor .dz #:init-form (make <daemon:Idzn>) #:init-keyword #:dz))

(define-method (initialize (o <daemon:Dzn_proxy>) args)
  (next-method)
  (set! (.dz o)
    (make <daemon:Idzn>
      #:in (make <daemon:Idzn.in>
        #:name "dz"
        #:self o
        #:command (lambda args (call-in o (lambda _ (apply dzn-command (cons o args))) `(,(.dz o) command))))
      #:out (make <daemon:Idzn.out>))))

(define-actor <pipeline> (<actor>)
  ((close pipeline-close)
   (open pipeline-open)
   (read pipeline-read)
   (write pipeline-write))

  (pids #:accessor .pids #:init-value #f #:init-keyword #:pids)
  (input-port #:accessor .input-port #:init-value #f #:init-keyword #:input-port)
  (output-port #:accessor .output-port #:init-value #f #:init-keyword #:output-port)
  (error-input-port #:accessor .error-input-port #:init-value #f #:init-keyword #:error-input-port)
  (error-output-port #:accessor .error-output-port #:init-value #f #:init-keyword #:error-output-port)
  (on-read #:accessor .on-read #:init-keyword #:on-read))

(define-method (pipeline-open (p <pipeline>) message commands)
  (match (pipe)
    ((error-input-port . error-output-port)
     (set-unbuffered! (set-non-blocking! error-input-port))
     (move->fdes error-output-port 2)
     (let ((output-port input-port pids
                        (with-error-to-port error-output-port
                          (cute pipeline commands))))
       (set! (.error-input-port p) error-input-port)
       (set! (.error-output-port p) error-output-port)
       (set! (.input-port p) (set-unbuffered! (set-non-blocking! input-port)))
       (set! (.output-port p) (set-unbuffered! (set-non-blocking! output-port)))
       (set! (.pids p) pids)))))

(define-method (pipeline-close (p <pipeline>) message)
  (self-destruct p))

(define-method (pipeline-read (p <pipeline>) message)
  ((.on-read p) p))

(define-method (pipeline-write (p <pipeline>) message data)
  (display data (.output-port p))
  (close (.output-port p)))

(define-method (8sync:pipe-command (o <daemon:Dzn_proxy>) name argv input)
  (let* ((hive (dzn:get (.locator o) <hive>))
         (pipeline
          (create-actor
           hive <pipeline>
           #:on-read
           (lambda (pipeline)
             (let* ((stdout (read-string (.input-port pipeline)))
                    (close-stderr-output (close (.error-output-port pipeline)))
                    (stderr (read-string (.error-input-port pipeline)))
                    (status (apply + (map
                                      (compose status:exit-val cdr waitpid)
                                      (.pids pipeline))))
                    (result (libenvelope:wrap
                             name
                             `((stdout . ,stdout)
                               (stderr . ,stderr)
                               (status . ,status)))))
               (action o .dz .out .result result)
               (<- (actor-id pipeline) 'close))))))

    (<- pipeline 'open `(,argv))
    (<- pipeline 'read)
    (<- pipeline 'write input)))

(define-method (blocking:pipe-command (o <daemon:Dzn_proxy>) name argv input)
  (let* ((stdout status stderr (pipeline*->string (list argv) #:input input))
         (result (libenvelope:wrap name `((stdout . ,stdout)
                                          (stderr . ,stderr)
                                          (status . ,status)))))
    (action o .dz .out .result result)))

(define-method (dzn-pipe-command (o <daemon:Dzn_proxy>) command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (let* ((name (assoc-ref command "name"))
           (argv (assoc-ref command "argv"))
           (input (assoc-ref command "input")))
      (log-debug "Dzn_proxy: piping: ~s\n" (and argv (string-join argv)))
      (if (blocking-pipe?) (blocking:pipe-command o name argv input)
          (8sync:pipe-command o name argv input)))))

(define* (input->root input #:key debug? model-name)
  (let* ((root (call-with-handle-exceptions
                (cute string->ast input)
                #:backtrace? debug?))
         (root (filter-root root #:model-name model-name)))
    (vm:normalize root)))

(define dzn-library-command
  (let ((input #f)
        (model #f)
        (root #f)
        (sut #f)
        (instances #f))
    (lambda (o command)
      (close (current-input-port)) ;; readline
      (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
        (let* ((argv (assoc-ref command "argv"))
               (command-input (assoc-ref command "input"))
               (model-name (assoc-ref command "model"))
               (trail (assoc-ref command "trace"))
               (trail trail-model (string->trail+model trail))
               (model-name (or model-name trail-model))
               (queue-size (or (and=> (assoc-ref command "queue-size") string->number) 3))
               (reset? (or (not (equal? model model-name))
                           (not (equal? input command-input)))))
          (log-debug "Dzn_proxy: calling library as if: ~s\n" (and argv (string-join argv)))
          (log-trace "trail: ~a\n" (string-join trail ","))

          (when reset?
            (log-debug "Dzn_proxy: RESET\n"))

          (let* ((stdout stderr
                         (with-output+error-to-string
                          (lambda _
                            (catch #t
                              (lambda _

                                (when reset?
                                  (set! input command-input)
                                  (set! model model-name)
                                  (set! root (input->root input #:model-name model-name))
                                  (set! sut (runtime:get-sut root (ast:get-model root model-name)))
                                  (set! instances (runtime:create-instances sut)))

                                (simulate** sut instances trail
                                            #:deadlock-check? #t
                                            #:interface-determinism-check? #t
                                            #:refusals-check? #t
                                            #:queue-size queue-size
                                            #:strict? #f
                                            #:locations? #t
                                            #:trace "trace"
                                            #:state? #t
                                            #:verbose? #f))
                              (const #f)
                              (lambda (key . args)
                                (case key
                                  ((error)
                                   (apply format (current-error-port) "~a\n" args))
                                  (else
                                   (format (current-error-port) "Simulate failed as if: ~a\n"
                                           (and argv (string-join argv)))
                                   (format (current-error-port) "Caught exception: ~a ~s\n" key args)
                                   (let ((stack (make-stack #t)))
                                     (display-backtrace stack (current-error-port))))))))))
                 (decoupler (actor-id (dzn:get (.locator o) <decoupler>)))
                 (status (if (string-null? stderr) 0 1))
                 (result (libenvelope:wrap "simulate"
                                           `((stdout . ,stdout)
                                             (stderr . ,stderr)
                                             (status . ,status)))))
            (<- decoupler 'decouple
                (lambda _
                  (action o .dz .out .result result)))))))))

(define-method (dzn-command (o <daemon:Dzn_proxy>) msg)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Dzn_proxy: and now run ~s\n" msg)
    (let* ((command (libenvelope:data msg))
           (name (assoc-ref command "name")))
      (if (equal? name "simulate")
          (dzn-library-command o command)
          (dzn-pipe-command o command)))))
