;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020, 2022 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands info)
  #:use-module (ice-9 rdelim)
  #:use-module (ice-9 getopt-long)
  #:use-module (ide commander)
  #:use-module (ide commands daemon)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide util))

(define (parse-opts args)
  (let* ((option-spec
          '((help (single-char #\h))
            (trail (single-char #\t))
            (verbose (single-char #\v))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (pair? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide info [OPTION]...
Query the daemon.

  -h, --help             display this help and exit
  -v, --verbose          be verbose; show connections with daemon
  -t, --trail            show the latest simulate trail
")
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (force? (option-ref options 'force #f))
         (verbose? (or (command-line:get 'verbose)
                       (ide:command-line:get 'verbose)))
         (local-result EXIT_SUCCESS))

    (define (info-handler message)
      (let ((info (parse-message message)))
        (cond ((info? info)
               (let* ((data (info-data info))
                      (stdout (assoc-ref data "stdout")))
                 (write-line stdout)))
              (else
               (format (current-error-port)
                       "ide info: error: received: ~s\n"
                       message)
               (set! local-result EXIT_FAILURE)))
        message))

    (daemon:start #:debug? debug? #:force? force? #:port %ide-port)
    (let* ((commander (make-commander #:debug? debug? #:verbose? verbose?))
           (command (make-command "info" '(trail)))
           (result-handler
            (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                     info-handler))
           (result
            (run-command commander command
                         #:result-handler result-handler
                         #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)))
      (exit (+ result local-result)))))
