;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (ide daemon-config)
  #:use-module (srfi srfi-9)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 match)
  #:use-module (ice-9 pretty-print)
  #:use-module (ide shell-util)
  #:use-module (ide config)
  #:export (daemon-config?
            daemon-config-http-port
            daemon-config-ide-port
            daemon-config-view-port
            daemon-config-editor-port

            daemon-config-set!
            read-daemon-config
            read-daemon-config-file
            update-daemon-config-file
            write-daemon-config
            write-daemon-config-file))

(define-record-type <daemon-config>
  (make-daemon-config http-port ide-port view-port editor-port)
  daemon-config?
  (http-port   daemon-config-http-port)    ;number
  (ide-port    daemon-config-ide-port)     ;number
  (view-port   daemon-config-view-port)    ;number
  (editor-port daemon-config-editor-port)) ;number

(define (daemon-config->sexp daemon-config)
  "Return DAEMON-CONFIG serialized as an sexp."
  (match daemon-config
    (($ <daemon-config> http-port ide-port view-port editor-port)
     `(daemon-config (version 0)
                     (http-port   ,http-port)
                     (ide-port    ,ide-port)
                     (view-port   ,view-port)
                     (editor-port ,editor-port)))))

(define* (read-daemon-config port #:key
                             (http-port   %default-http-port)
                             (ide-port    %default-ide-port)
                             (view-port   %default-view-port)
                             (editor-port %default-editor-port))
  "Read from PORT daemon config in the format expected for the '.daemon'
file.  Return a <daemon-config> record, or raise an error if valid
metadata could not be read from PORT."
  (match (read port)
    (('daemon-config ('version 0) properties ...)
     (let ((http-port   (or (and=> (assoc-ref properties 'http-port)   car)
                            http-port))
           (ide-port    (or (and=> (assoc-ref properties 'ide-port)    car)
                            ide-port))
           (view-port   (or (and=> (assoc-ref properties 'view-port)   car)
                            view-port))
           (editor-port (or (and=> (assoc-ref properties 'editor-port) car)
                            editor-port)))
       (make-daemon-config http-port ide-port view-port editor-port)))
    ((and ('channel ('version version) _ ...) sexp)
     (throw 'wrong-type-arg
            (format #f "~a: unsupported version: ~s\n" (port-filename port) version)))
    (sexp
     (throw 'wrong-type-arg
            (format #f "~a: invalid config file" (port-filename port))))))

(define* (read-daemon-config-file #:key
                                  (file-name   %daemon-config-file)
                                  (http-port   %default-http-port)
                                  (ide-port    %default-ide-port)
                                  (view-port   %default-view-port)
                                  (editor-port %default-editor-port))
  "Return a daemon-config record read from FILE-NAME, or return the
default daemon-config record if FILE-NAME doesn't exist."
  (catch 'system-error
    (lambda ()
      (call-with-input-file file-name
        read-daemon-config))
    (lambda args
      (if (= ENOENT (system-error-errno args))
          (make-daemon-config http-port ide-port view-port editor-port)
          (apply throw args)))))

(define* (write-daemon-config daemon-config port)
  "Serialize DAEMON-CONFIG and write to PORT."
  (pretty-print (daemon-config->sexp daemon-config) port))

(define* (write-daemon-config-file daemon-config #:key
                                   (file-name %daemon-config-file))
  "Serialize DAEMON-CONFIG and write to FILE-NAME."
  (let ((dir (dirname file-name)))
    (unless (directory-exists? dir)
      (mkdir-p dir)))
  (call-with-output-file file-name
    (cute write-daemon-config daemon-config <>)))

(define (daemon-config-set!)
  (let ((daemon-config (read-daemon-config-file)))
    (set! %http-port (daemon-config-http-port daemon-config))
    (set! %ide-port (daemon-config-ide-port daemon-config))
    (set! %view-port (daemon-config-view-port daemon-config))
    (set! %editor-port (daemon-config-editor-port daemon-config))))

(define* (update-daemon-config-file #:key
                                    (file-name   %daemon-config-file)
                                    (http-port   %default-http-port)
                                    (ide-port    %default-ide-port)
                                    (view-port   %default-view-port)
                                    (editor-port %default-editor-port))
  "If any of HTTP-PORT, IDE-PORT, VIEW-PORT, or EDITOR-PORT has been
modified, write a new daemon configuration to FILE-NAME."
  (when (or (not (eq? http-port   %http-port))
            (not (eq? ide-port    %ide-port))
            (not (eq? view-port   %view-port))
            (not (eq? editor-port %editor-port)))
      (let ((daemon-config (make-daemon-config http-port ide-port view-port editor-port)))
        (write-daemon-config-file daemon-config #:file-name file-name))))
