;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2021 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;; Copyright © 2021 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;; Commentary:
;;
;; Dezyne Language Server for the Language Server Protocol
;; https://microsoft.github.io/language-server-protocol

;;; Code:

(define-module (ide lsp)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-9 gnu)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)
  #:use-module (ice-9 binary-ports)
  #:use-module (ice-9 iconv)
  #:use-module (ice-9 match)
  #:use-module (ice-9 poe)
  #:use-module (ice-9 rdelim)
  #:use-module (web http)
  #:use-module (web uri)

  #:use-module (dzn parse)
  #:use-module (dzn parse peg)
  #:use-module (dzn parse complete)
  #:use-module (dzn parse lookup)
  #:use-module (dzn parse tree)
  #:use-module (dzn parse util)

  #:use-module (ide json)
  #:use-module (ide shell-util)
  #:use-module (ide util)

  #:export (lsp:stdio))

;;; Print debug messages?
(define %debug? (make-parameter #f))


;;;
;;; RPC messages.
;;;

(define (rpc:get-id jsonrpc)
  (assoc-ref jsonrpc "id"))

(define (rpc:get-jsonrpc jsonrpc)
  (assoc-ref jsonrpc "jsonrpc"))

(define (rpc:get-method jsonrpc)
  (assoc-ref jsonrpc "method"))

(define (rpc:get-params jsonrpc)
  (assoc-ref jsonrpc "params"))

(define (rpc:get-root-uri-string jsonrpc)
  (assoc-ref jsonrpc "rootUri"))

(define (rpc:get-root-uri-string jsonrpc)
  (and=> (assoc-ref jsonrpc "params")
         rpc:params:get-root-uri-string))

(define (rpc:params:get-id params)
  (assoc-ref params "id"))

(define (rpc:params:get-text params)
  (and=> (assoc-ref params "textDocument")
         (cute assoc-ref <> "text")))

(define (rpc:params:get-content-changes params)
  (and=> (assoc-ref params "contentChanges")
         (compose (cute assoc-ref <> "text") car)))

(define (rpc:params:get-root-uri-string params)
  (assoc-ref params "rootUri"))

(define (rpc:params:get-uri-string params)
  (and=> (assoc-ref params "textDocument")
         (cute assoc-ref <>  "uri")))

(define (rpc:params:get-position params)
  (assoc-ref params "position"))

(define (rcp:make-message sexp)
  "Return an JSON RPC message with SEXP."
  (let* ((body    (scm->json-string sexp))
         (text    (string-append body "\r\n"))
         (size    (string-length text)))
    (format #f "Content-Length: ~a\r\n\r\n~a" size text)))

(define* (rpc:read-message #:optional (input-port (current-input-port)))
  (let* ((headers (read-headers input-port))
         (size    (assoc-ref headers 'content-length)))
    (when size
      (let* ((bv (get-bytevector-n input-port size))
             (json-data (bytevector->string bv "utf-8")))
        (json-string->alist-scm json-data)))))


;;;
;;; Parse errors.   XXX Split to util/peg;
;;;

;;; XXX TODO use error-/error: prefix?
;;; Use one <error> type?
(define-immutable-record-type <error>
  (make-error location message)
  error?
  (location error-location)
  (message  error-message))

(define-immutable-record-type <missing>
  (make-missing location expectations)
  missing?
  (location     missing-location)
  (expectations missing-expectations))

(define-immutable-record-type <skipped>
  (make-skipped location skipped location-end)
  skipped?
  (location     skipped-location)
  (skipped      skipped-skipped)
  (location-end skipped-location-end))

(define (error:location error)
  (match error
    ((or ($ <error> location)
         ($ <missing> location)
         ($ <skipped> location))
     location)))

(define (error:location-end error)
  (match error
    (($ <error> location)
     location)
    (($ <missing> location)
     location)
    (($ <skipped> location skipped location-end)
     location-end)))

(define* (error:message error #:key without-location?)
  (match error
    (($ <error> location message)
     (let* ((type (error:type error))
            (location (and location (location->string location)))
            (message (format #f "error: ~a" message)))
       (if without-location? message
           (format #f "~a:~a: ~a" location type message))))
    (($ <missing> location missing)
     (let* ((type (error:type error))
            (location (location->string location))
            (message (format #f "missing: ~a" missing)))
       (if without-location? message
           (format #f "~a:~a: ~a" location type message))))
    (($ <skipped> location skipped)
     (let* ((type (error:type error))
            (location (location->string location))
            (message "skipping."))
       (if without-location? message
           ;; XXX TODO: location range
           (format #f "~a:~a: ~a" location type message))))))

(define (error:type error)
  (match error
    ((or ($ <error>) ($ <missing>) ($ <skipped>))
     'error)))

(define (error:source error)
  (match error
    ((or ($ <error>) ($ <missing>) ($ <skipped>))
     "dzn parse")))

(define (peg:file-name+offset+error->error file-name offset error text)
  (let ((location (file-offset->location file-name offset text)))
    (match error
      (('missing missing)
       (make-missing location missing))
      (('skipped skipped)
       (let* ((offset-end   (+ offset (string-length skipped)))
              (location-end (file-offset->location file-name offset-end text)))
         (make-skipped location skipped location-end)))
      (_
       (make-error location error)))))

(define string->parse-tree+errors-
  (pure-funcq
   (lambda (text file-name)
     (let ((errors '())
           (file-name (symbol->string file-name)))
       (define (add-error! offset string error)
         (let ((error (peg:file-name+offset+error->error file-name offset error text)))
           (set! errors (cons error errors))))
       (parameterize ((%peg:fall-back? #t)
                      (%peg:error add-error!))
         (cons
          (catch #t
            (lambda _
              (string->parse-tree text #:content-alist '() #:file-name file-name))
            (const #f)
            (lambda (key . args)
              (let ((stack (make-stack #t)))
                (let ((backtrace (call-with-output-string
                                   (cute display-backtrace stack <>))))
                  (log-debug "lsp:peg:collect-errors file-name=~a\n" file-name)
                  (log-debug "lsp:peg:collect-errors file\n~a\n" backtrace))
                '())))
          errors))))))

(define* (string->parse-tree+errors text #:key (content-alist '()) (file-name "-"))
  (match (string->parse-tree+errors- text (string->symbol file-name))
    ((parse-tree . errors) (values parse-tree errors))))


;;;
;;; LSP library.
;;;

(define-immutable-record-type <lsp>
  (make-lsp- root-uri-string imports content-alist parse-alist)
  lsp?
  (root-uri-string lsp-root-uri-string) ;string
  (imports         lsp-imports)         ;list of strings
  (content-alist   lsp-content-alist)   ;alist ((file-name . content) ...)
  (parse-alist     lsp-parse-alist))    ;alist ((file-name . parse-tree) ...)

(define* (make-lsp uri-string #:key (imports '()) (content-alist '())
                   (parse-alist '()))
  (make-lsp- uri-string imports content-alist parse-alist))

(define* (lsp:resolve-file lsp file-name #:key (imports '()))
  (let ((imports (append imports (lsp-imports lsp))))
    (and=> (search-path imports file-name) canonicalize-path)))

(define (lsp:get-text lsp file-name)
  (let* ((file-name (or (lsp:resolve-file lsp file-name)
                        file-name))
         (text (assoc-ref (lsp-content-alist lsp) file-name)))
    (if text (values lsp text)
        (let* ((text (false-if-exception*
                      (with-input-from-file file-name read-string)
                      #:backtrace? (%debug?)
                      #:warning "lsp:get-text ~a\n" file-name))
               (lsp (lsp:set-text lsp file-name text)))
          (values lsp text)))))

(define (lsp:get-parse-tree lsp file-name)
  (let* ((parse-alist (lsp-parse-alist lsp))
         (file-name (or (lsp:resolve-file lsp file-name)
                        file-name))
         (tree (assoc-ref parse-alist file-name)))
    (cond (tree (values lsp tree))
          (file-name (let* ((lsp (lsp:set-file lsp file-name))
                            (parse-alist (lsp-parse-alist lsp))
                            (tree (assoc-ref parse-alist file-name)))
                       (values lsp tree)))
          (else (values lsp #f)))))

(define (lsp:file-name+position->location file-name position)
  (let ((line   (assoc-ref position "line"))
        (column (assoc-ref position "character")))
    (make-location file-name (1+ line) column)))

(define (lsp:file-name+offset->location lsp file-name offset)
  (let ((lsp text (lsp:get-text lsp file-name)))
    (file-offset->location file-name offset text)))

(define* (lsp:uri-string->file-name uri-string)
  (let* ((uri       (string->uri uri-string))
         (host      (uri-host uri))
         (path      (uri-path uri))
         (parts     (split-and-decode-uri-path path))
         (file-name (string-join parts "/"))
         (file-name (if (or (string-prefix? "/" file-name)
                            ;; XXX for tests: file://localhost/some/file/name
                            (equal? host "localhost")
                            (mingw?)) file-name
                            (string-append "/" file-name))))
    (or (false-if-exception*
         (canonicalize-path file-name)
         #:backtrace? (%debug?)
         #:warning "lsp:uri-string->file-name: No such file: ~a\n" file-name)
        file-name)))

(define* (lsp:file-name->uri-string lsp file-name)
  (let* ((file-name (or (lsp:resolve-file lsp file-name)
                        file-name))
         (parts     (string-split file-name #\/)))
    (string-append "file://" (if (mingw?) "/" "")
                   (encode-and-join-uri-path parts))))

(define* (lsp:location->result lsp location #:key (location-end location))
  (let* ((file-name  (location-file location))
         (uri-string (lsp:file-name->uri-string lsp file-name))
         (line       (1- (location-line location)))
         (column     (location-column location))
         (end-line   (1- (location-line location-end)))
         (end-column (location-column location-end)))
    `((uri   . ,uri-string)
      (range . ((start . ((line      . ,line)
                          (character . ,column)))
                (end   . ((line      . ,end-line)
                          (character . ,end-column))))))))

(define (lsp:error->diagnostic lsp error)
  (define (type->severity type)
    (case type
      ((error) 1)
      ((warn) 2)
      ((info) 3)
      ((hint) 4)))
  (let* ((message  (error:message error #:without-location? #t))
         (location (error:location error))
         (end      (error:location-end error))
         (type     (error:type error))
         (severity (type->severity type)))
    `(,@(lsp:location->result lsp location #:location-end end)
      (severity . ,severity)
      (source   . "dzn parse")
      (message  . ,message))))

(define (lsp:errors->diagnostics lsp file-name errors)
  (let* ((diagnostics (map (cute lsp:error->diagnostic lsp <>) errors))
         (diagnostics (delete-duplicates diagnostics))
         (uri-string  (lsp:file-name->uri-string lsp file-name)))
    `((method . "textDocument/publishDiagnostics")
      (params . ((uri         . ,uri-string)
                 (diagnostics . ,diagnostics))))))

(define (lsp:completion->result completion)
  `(("label" . ,completion) ("kind" . 11)))

(define (lsp:make-message data)
  "Return an LSP message with DATA."
  (let ((message `((jsonrpc . "2.0")
                   ,@data)))
    (rcp:make-message message)))

(define (lsp:make-reply-message request result)
  "Return an LSP relpy message for REQUEST with RESULT."
  (let* ((id      (rpc:get-id request))
         (jsonrpc (rpc:get-jsonrpc request))
         (message `((jsonrpc . ,jsonrpc)
                    (id      . ,id)
                    (result  . ,result))))
    (rcp:make-message message)))

(define (lsp:context lsp location)
  (let* ((file-name (location-file location))
         (lsp text  (lsp:get-text lsp file-name)))
    (if (not text) (values lsp #f)
        (let* ((offset    (location->offset location text))
               (lsp tree  (lsp:get-parse-tree lsp file-name))
               (imports   (lsp-imports lsp)))
          (values lsp (complete:context tree offset))))))

(define (lsp:completions lsp location)
  (let* ((lsp
          context   (lsp:context lsp location))
         (token     (and=> context .tree))
         (file-name (location-file location))
         (lsp text  (lsp:get-text lsp file-name))
         (offset    (location->offset location text)))
    (define (get-parse-tree file-name)
      (let ((lsp* tree (lsp:get-parse-tree lsp file-name)))
        (set! lsp lsp*)
        tree))
    (define* (resolve-file file-name #:key (imports '()))
      (or (lsp:resolve-file lsp file-name #:imports imports)
          file-name))
    (values lsp
            (false-if-exception*
             (complete token context offset
                       #:file-name->parse-tree get-parse-tree
                       #:imports (lsp-imports lsp)
                       #:resolve-file resolve-file)
             #:backtrace? (%debug?)
             #:warning "lsp:completions: ~a\n" location))))

(define (lsp:lookup-definition lsp location)
  (let* ((lsp
          context     (lsp:context lsp location))
         (token       (and=> context .tree))
         (file-name   (location-file location)))
    (define (get-text file-name)
      (let ((lsp* text (lsp:get-text lsp file-name)))
        (set! lsp lsp*)
        text))
    (define (get-parse-tree file-name)
      (let ((lsp* tree (lsp:get-parse-tree lsp file-name)))
        (set! lsp lsp*)
        tree))
    (define* (resolve-file file-name #:key (imports '()))
      (or (lsp:resolve-file lsp file-name #:imports imports)
          file-name))
    (values lsp
            (and context
                 (false-if-exception*
                  (lookup-location token context
                                   #:file-name file-name
                                   #:file-name->text get-text
                                   #:file-name->parse-tree get-parse-tree
                                   #:resolve-file resolve-file)
                  #:backtrace? (%debug?)
                  #:warning "lsp:lookup-definition: ~a\n" location)))))

(define (lsp:import-errors lsp file-name)
  (let* ((lsp parse-tree (lsp:get-parse-tree lsp file-name))
         (imports (tree:import* parse-tree))
         (lsp text (lsp:get-text lsp file-name)))
    (filter-map
     (lambda (import)
       (let* ((location (tree:->location import text #:file-name file-name))
              (dir (dirname file-name))
              (file-name (.file-name import)))
         (and (not (lsp:resolve-file lsp file-name #:imports (cons dir (lsp-imports lsp))))
              (make-error location (format #f "No such file: ~a\n" file-name)))))
     imports)))

(define (lsp:update-parse lsp file-name)
  (let* ((content-alist (lsp-content-alist lsp))
         (parse-alist   (lsp-parse-alist lsp))
         (parse-alist   (alist-delete file-name parse-alist))
         (lsp content   (lsp:get-text lsp file-name))
         (parse-tree    (string->parse-tree+errors
                         content
                         #:file-name file-name
                         #:content-alist content-alist))
         (parse-alist   (acons file-name parse-tree parse-alist)))
    (set-field lsp (lsp-parse-alist) parse-alist)))

(define (lsp:set-file lsp file-name)
  (let* ((content-alist (lsp-content-alist lsp))
         (content-alist (alist-delete file-name content-alist))
         (imports       (lsp-imports lsp))
         (file-name     (or (lsp:resolve-file lsp file-name)
                            file-name))
         (dir           (dirname file-name))
         (imports       (delete-duplicates (cons dir imports)))
         (lsp           (set-field lsp (lsp-imports) imports))
         (content-alist (catch #t
                          (lambda _
                            (with-error-to-port (%make-void-port "w")
                              (lambda _
                                (file+import-content-alist
                                 file-name
                                 #:imports imports
                                 #:content-alist content-alist))))
                          (const '())))
         (lsp           (set-field lsp (lsp-content-alist) content-alist)))
    (lsp:update-parse lsp file-name)))

(define (lsp:set-text lsp file-name text)
  (let* ((content-alist (lsp-content-alist lsp))
         (content-alist (alist-delete file-name content-alist))
         (content-alist (acons file-name text content-alist))
         (lsp           (set-field lsp (lsp-content-alist) content-alist))
         (parse-alist   (lsp-parse-alist lsp))
         (parse-alist   (alist-delete file-name parse-alist))
         (lsp           (set-field lsp (lsp-parse-alist) parse-alist)))
    (if (not text) lsp
        (lsp:update-parse lsp file-name))))

(define* (lsp:params->file-name params)
  "Return file-name from PARAMS."
  (let* ((uri-string (rpc:params:get-uri-string params))
         (position   (rpc:params:get-position params)))
    (lsp:uri-string->file-name uri-string)))

(define* (lsp:params->location params)
  "Return location object from PARAMS."
  (let ((file-name  (lsp:params->file-name params))
        (position   (rpc:params:get-position params)))
    (lsp:file-name+position->location file-name position)))

(define* (lsp:word-at-point lsp point)
  "Return word at POINT."
  (let* ((file-name  (location-file point))
         (lsp text   (lsp:get-text lsp file-name))
         (offset     (location->offset point text))
         (offset     (if (zero? offset) 0 (1- offset)))
         (count      (string-length text))
         (word-set   (char-set-adjoin char-set:letter+digit #\_ #\.))
         (char       (string-ref text offset)))
    (and (char-set-contains? word-set char)
         (let ((start (let loop ((pos offset))
                        (if (zero? pos) pos
                            (let* ((next (1- pos))
                                   (char (string-ref text next)))
                              (if (not (char-set-contains? word-set char)) pos
                                  (loop next))))))
               (end (let loop ((pos offset))
                      (if (= pos (1- count)) pos
                          (let* ((next (1+ pos))
                                 (char (string-ref text next)))
                            (if (not (char-set-contains? word-set char)) pos
                                (loop next)))))))
           (substring text start (1+ end))))))

(define* (lsp:cater-for-caret lsp point)
  "In case POINT is the location of a caret instead of a cursor, move
POINT one position left if POINT is on punctuation to the right of a
word."
  (let* ((file-name (location-file point))
         (lsp text  (lsp:get-text lsp file-name))
         (offset    (location->offset point text)))
    (log-debug "lsp:cater-for-caret: ~a:~a (~a)\n" (location-line point) (location-column point) offset)
    (if (zero? offset) point
        (let* ((word-set (char-set-adjoin char-set:letter+digit #\_))
               (char     (string-ref text offset))
               (previous (string-ref text (1- offset))))
          (cond ((char-set-contains? word-set char)
                 point)
                ((char-set-contains? word-set previous)
                 (log-debug "lsp:cater-for-caret: ~a\n" (1- offset))
                 (make-location file-name
                                (location-line point)
                                (1- (location-column point))))
                (else
                 point))))))

(define (lsp:init-port! port)
  (setvbuf port 'none)
  (set-port-encoding! port "ISO-8859-1")
  (when (mingw?)
    (set-port-binary/text-mode! port O_BINARY)))


;;;
;;; Message handlers.
;;;

(define (lsp:check-syntax lsp params)
  (let* ((file-name   (lsp:params->file-name params))
         (lsp text    (lsp:get-text lsp file-name))
         (parse-tree
          errors      (string->parse-tree+errors text
                                                 #:file-name file-name
                                                 #:content-alist (lsp-content-alist lsp)))
         (errors      (append (lsp:import-errors lsp file-name) errors))
         (diagnostics (lsp:errors->diagnostics lsp file-name errors)))
    (values lsp diagnostics)))

(define (lsp:complete lsp params)
  (let* ((point    (lsp:params->location params))
         (lsp
          complete (lsp:completions lsp point))
         (complete (or complete '()))
         (word     (false-if-exception*
                    (lsp:word-at-point lsp point)
                    #:backtrace? (%debug?)
                    #:warning "lsp:word-at-point: ~a\n" point))
         (complete (if (not word) complete
                       (filter (cute string-prefix? word <>) complete)))
         ;; XXX Apparently LSP (or the implementations of emacs-lsp-mode
         ;; and VSCodium derivatives) assign semantics to the `.'
         ;; character.  Completion results must not include words that
         ;; include a dot.
         (prefix   (and word (string-rindex word #\.)))
         (complete (if (not prefix) complete
                           (map (cute substring <> (1+ prefix))
                                complete)))
         (result   (map lsp:completion->result complete)))
    (log-debug "lsp:goto-def result=~a\n" result)
    (values lsp result)))

(define (lsp:goto-def lsp params)
  (let* ((point (lsp:params->location params))
         (point (lsp:cater-for-caret lsp point))
         (lsp
          def   (lsp:lookup-definition lsp point))
         (result (if (not def) '()
                     (lsp:location->result lsp def))))
    (log-debug "lsp:goto-def result=~a\n" result)
    (values lsp result)))

(define* (lsp:initialize lsp params)
  (let ((capabilities
         `((completionProvider        . ((resolveProvider . #t)))
           (definitionProvider        . #t)
           (documentHighlightProvider . #f)
           (documentSymbolProvider    . #f)
           (hoverProvider             . #f)
           (referencesProvider        . #f)
           (textDocumentSync          . 1)
           (workspaceSymbolProvider   . #f))))
    (values lsp `((capabilities . ,capabilities)))))

(define (lsp:open-file lsp params)
  (let* ((file-name (lsp:params->file-name params))
         (text      (rpc:params:get-text params))
         (lsp       (lsp:set-text lsp file-name text))
         (imports   (lsp-imports lsp))
         (file-name (or (lsp:resolve-file lsp file-name)
                        file-name))
         (dir       (dirname file-name))
         (imports   (delete-duplicates (cons dir imports)))
         (lsp       (set-field lsp (lsp-imports) imports)))
    (log-debug "lsp:open-file ~a\n" file-name)
    (values lsp #f)))

(define* (lsp:update-text lsp params)
  (let* ((file-name  (lsp:params->file-name params))
         (text       (rpc:params:get-content-changes params))
         (lsp        (lsp:set-text lsp file-name text)))
    (log-debug "lsp:update-text ~a\n" file-name)
    (values lsp #f)))


;;;
;;; Message dispatching.
;;;

(define %lsp:handler-alist
  `(("initialize"              . ,lsp:initialize)
    ("textDocument/completion" . ,lsp:complete)
    ("textDocument/definition" . ,lsp:goto-def)
    ("textDocument/didChange"  . ,(lambda (lsp . args)
                                    (let ((lsp (apply lsp:update-text
                                                      lsp args)))
                                      (apply lsp:check-syntax lsp args))))
    ("textDocument/didOpen"    . ,(lambda (lsp . args)
                                    (let ((lsp x (apply lsp:open-file lsp args)))
                                      (apply lsp:check-syntax lsp args))))
    ("textDocument/didSave"    . ,lsp:check-syntax)))

(define %lsp:singleton-alist
  `(("textDocument/didChange"  . ,#t)))

(define* (lsp:dispatch-message lsp message #:key (queue '()))
  "Parse MESSAGE for LSP method and dispatch it, or skip it according to
%LSP:SINGLETON-ALIST and QUEUE.  QUEUE.  Return two values: LSP, and a
response or #f."
  (let* ((id         (rpc:get-id message))
         (method     (rpc:get-method message))
         (params     (rpc:get-params message))
         (handler    (assoc-ref %lsp:handler-alist method))
         (singleton? (assoc-ref %lsp:singleton-alist method))
         (methods    (map rpc:get-method queue))
         (cancel     (filter (compose (cute equal? <> "$/cancelRequest")
                                      rpc:get-method)
                             queue))
         (skip?      (or (and id
                              (find (compose (cute equal? <> id)
                                             rpc:params:get-id
                                             rpc:get-params)
                                    cancel))
                         (and singleton?
                              (find (cute equal? <> method) methods)))))
    (log-verbose "method[~a]: ~a\n" id method)
    (log-debug "message: ~a\n" message)
    (log-debug "params: ~a\n" params)
    (if (or (not handler) skip?) (values lsp #f)
        (let ((lsp result (handler lsp params)))
          (log-debug "reply[~a]: ~a\n" id result)
          (match result
            (#f (values lsp #f))
            ((('method . method) rest ...)
             (values lsp (lsp:make-message result)))
            (_
             (values lsp (lsp:make-reply-message message result))))))))

(define* (lsp:handle-message lsp message #:key (imports '()) (queue '()))
  "Handle MESSAGE in LSP context, (re)creating LSP context using IMPORTS,
and return two values: LSP, and a response or #f."
  (let* ((root-uri-string (rpc:get-root-uri-string message))
         (lsp (if (or (not root-uri-string)
                      (equal? root-uri-string
                              (lsp-root-uri-string lsp)))
                  lsp
                  (make-lsp root-uri-string #:imports imports))))
    (lsp:dispatch-message lsp message #:queue queue)))


;;;
;;; Entry point.
;;;

(define* (lsp:stdio #:key
                    (input-port (current-input-port))
                    (output-port (current-output-port))
                    debug?
                    file-name
                    (imports '()))
  "Read Language Server message from INPUT-PORT and write response to
OUTPUT-PORT until INPUT-PORT is drained."
  (define (service-lsp lsp)
    (when (eof-object? (lookahead-u8 input-port))
      (exit EXIT_SUCCESS))
    (let loop ((lsp lsp) (queue '()))
      (let ((messages (unfold (disjoin
                               port-closed?
                               (negate char-ready?)
                               (compose eof-object? lookahead-u8))
                              rpc:read-message
                              identity
                              input-port)))
        (log-verbose "\n")
        (log-debug "messages[~a]: ~a\n" (length messages)
                   (map rpc:get-method messages))
        (match (append queue messages)
          (()
           lsp)
          ((message queue ...)
           (let ((lsp response (lsp:handle-message lsp message
                                                   #:queue queue
                                                   #:imports imports)))
             (when response
               (display response output-port)
               (force-output output-port))
             (loop lsp queue)))))))
  (lsp:init-port! input-port)
  (lsp:init-port! output-port)
  (parameterize ((%debug? debug?))
    (let loop ((lsp (make-lsp #f)))
      (unless (or (port-closed? input-port)
                  (eof-object? (lookahead-u8 input-port)))
        (loop (service-lsp lsp))))))
