// dzn-runtime -- Dezyne runtime library
//
// Copyright © 2015, 2016, 2017, 2019, 2020, 2021, 2022 Rutger van Beusekom <rutger@dezyne.org>
// Copyright © 2017, 2018, 2019, 2020, 2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
//
// This file is part of dzn-runtime.
//
// dzn-runtime is free software: you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// dzn-runtime is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with dzn-runtime.  If not, see <http://www.gnu.org/licenses/>.
//
// Commentary:
//
// Code:

#ifndef DZN_CONTAINER_HH
#define DZN_CONTAINER_HH

#include <dzn/locator.hh>
#include <dzn/runtime.hh>
#include <dzn/pump.hh>

#include <algorithm>
#include <functional>
#include <iostream>
#include <iterator>
#include <map>
#include <queue>
#include <sstream>
#include <string>

namespace dzn
{
  template <typename System, typename Function>
  struct container
  {
    dzn::meta meta;
    dzn::locator dzn_locator;
    dzn::runtime dzn_rt;
    System system;

    std::map<std::string, Function> lookup;
    std::queue<std::string> trail;
    std::mutex mutex;
    std::condition_variable condition;
    dzn::pump pump;

    friend std::ostream& operator << (std::ostream& os, container<System,Function>&)
    {
      return os;
    }

    container(bool flush, dzn::locator&& l = dzn::locator{})
    : meta{"<external>","container",0,0,{},{&system.dzn_meta},{[this]{system.check_bindings();}}}
    , dzn_locator(std::forward<dzn::locator>(l))
    , dzn_rt()
    , system(dzn_locator.set(dzn_rt).set(pump))
    , pump()
    {
      dzn_locator.get<illegal_handler>().illegal = []{std::clog << "illegal" << std::endl; std::exit(0);};
      dzn_rt.performs_flush(this) = flush;
      system.dzn_meta.name = "sut";
    }
    ~container()
    {
      std::unique_lock<std::mutex> lock(mutex);
      condition.wait(lock, [this]{return trail.empty();});
      dzn::pump* p = system.dzn_locator.template try_get<dzn::pump>(); // only shells have a pump
      //resolve the race condition between the shell pump dtor and the container pump dtor
      if(p && p != &pump) pump([p] {p->stop();});
      pump.wait();
    }
    void perform(const std::string& str)
    {
      if (std::count(str.begin(), str.end(), '.') > 1
          || str == "<defer>")
        return;

      auto it = lookup.find(str);
      if(it != lookup.end())
        pump(it->second);

      std::unique_lock<std::mutex> lock(mutex);
      trail.push(str);
      condition.notify_one();
    }
    void operator()(std::map<std::string, Function>&& lookup)
    {
      this->lookup = std::move(lookup);
      pump.pause();
      std::string str;
      while(std::getline (std::cin, str))
      {
        if(str.find("<flush>") != std::string::npos
           || str == "<defer>")
          pump.flush();
        perform(str);
      }
      pump.resume();
    }
    void match(const std::string& perform)
    {
      std::string expect = trail_expect();
      if(expect != perform)
        throw std::runtime_error("unmatched expectation: trail expects: \"" + expect
                                 + "\" behavior expects: \"" + perform + "\"");
    }
    std::string trail_expect()
    {
      std::unique_lock<std::mutex> lock(mutex);
      condition.wait_for(lock, std::chrono::seconds(10),
                         [this]{return trail.size();});
      std::string expect = trail.front();
      trail.pop();
      if (trail.empty())
        condition.notify_one();
      return expect;
    }
  };
}

#endif //DZN_CONTAINER_HH
//version: 2.16.0
