;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (ide pipe)
  #:use-module (ice-9 popen)
  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)
  #:use-module (ide shell-util)
  #:export (blocking-pipe?
            pipeline
            pipeline*
            pipeline->string
            pipeline*->string
            set-non-blocking!
            set-unbuffered!))

(define piped-process (@@ (ice-9 popen) piped-process))

(define (pipe->fdes)
  (let ((p (pipe)))
   (cons (port->fdes (car p))
         (port->fdes (cdr p)))))

(define (pipeline commands)
  "Execute a pipeline of COMMANDS, feeding INPUT to the first command.
If the first command is a procedure, run that and take its output as
input for the pipeline, instead of INPUT.  Each command is a list of a
program and its arguments as strings.  Return three values: the input
port to write to the pipeline, the output port to read from the
pipeline, and the pids from all COMMANDS."
  (define (command pipe proc previous)
    (match previous
      ((pfrom . pids)
       (let* ((to pfrom)
	      (from pipe))
         (cond ((procedure? proc)
                (throw 'error "procedure inside pipeline" commands))
               (else
                (let ((pid (piped-process (car proc) (cdr proc) from to)))
	          (cons from (cons pid pids)))))))))
  (let* ((to (pipe->fdes))
         (pipes (map (lambda _ (pipe->fdes)) commands))
	 (pipeline (fold command `(,to) pipes commands)))
    (match pipeline
      ((from . pids)
       (values (fdes->outport (cdr to)) (fdes->inport (car from)) pids)))))

(define* (pipeline* commands)
  "Execute a pipeline of COMMANDS, feeding INPUT to the first command.
If the first command is a procedure, run that and take its output as
input for the pipeline, instead of INPUT.  Each command is a list of a
program and its arguments as strings.  Return four values: the input
port to write to the pipeline, the output port to read from the
pipeline, the error port to read from the pipeline, and the pids from
all COMMANDS."
  (match (pipe)
    ((error-input-port . error-output-port)
     (move->fdes error-output-port 2)
     (let ((output-port input-port pids
                        (with-error-to-port error-output-port
                          (cute pipeline commands))))
       (close error-output-port)
       (let ((error (read-string error-input-port)))
         (values output-port input-port error-input-port pids))))))

(define* (pipeline->string commands #:key (input ""))
  "Execute a pipeline of COMMANDS, feeding INPUT to the first command.
If the first command is a procedure, run that and take its output as
input for the pipeline, instead of INPUT.  Each command is a list of a
program and its arguments as strings.  Return three values: the output
that the pipeline produced as a string, and a sum of the output stati of
the pipeline COMMANDS."
  (define (pipeline-with-input commands input)
    (let ((output-port input-port pids (pipeline commands)))
      (display input output-port)
      (close output-port)
      (let ((stdout (read-string input-port)))
        (false-if-exception (close input-port))
        (values stdout (apply + (map (compose status:exit-val cdr waitpid) pids))))))
  (match commands
    (((? procedure? procedure) commands ...)
     (unless (string-null? input)
       (format (current-error-port) "pipeline*->string: warning: ignoring input: ~a\n" input))
     (let ((input (with-output-to-string procedure)))
       (pipeline-with-input commands input)))
    (_
     (pipeline-with-input commands input))))

(define* (pipeline*->string commands #:key (input ""))
  "Execute a pipeline of COMMANDS, feeding INPUT to the first command.
If the first command is a procedure, run that and take its output as
input for the pipeline, instead of INPUT.  Each command is a list of a
program and its arguments as strings.  Return three values: the output
that the pipeline produced as a string, a sum of the output stati of the
pipeline COMMANDS, and the error output that the pipeline procuded as a
string."
  (match (pipe)
    ((error-input-port . error-output-port)
     (move->fdes error-output-port 2)
     (let ((output status
                   (with-error-to-port error-output-port
                     (cut pipeline->string commands #:input input))))
       (close error-output-port)
       (let ((error (read-string error-input-port)))
         (values output status error))))))

(define (blocking-pipe?)
  (mingw?))

(define (set-non-blocking! port)
  (unless (blocking-pipe?)
    (let ((flags (fcntl port F_GETFL)))
      (fcntl port F_SETFL (logior O_NONBLOCK flags))))
  port)

(define (set-unbuffered! port)
  (setvbuf port 'none)
  port)
