;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands browse)
  #:use-module (ice-9 getopt-long)
  #:use-module (srfi srfi-26)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide shell-util)
  #:use-module (ide util)
  #:export (parse-opts
            main))

(define (parse-opts args)
  (let* ((option-spec
          '((help (single-char #\h))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f)))
    (when help?
        (format #t "\
Usage: ide browse [OPTION]... [VIEW]
Open VIEW in a browser.

  -h, --help             display this help and exit

Views:
  system state trace     default: system

  ~a
  ~a
  ~a
"
                  (get-url "system")
                  (get-url "state")
                  (get-url "trace"))
        (exit  EXIT_SUCCESS))
    options))

(define* (browse url #:key debug? verbose?)
  (let* ((chrome-args `("--disable-translate" "--temp-profile" "--window-size"
                        ,(string-append "--app=" url)))
         (browsers (if (mingw?) `(("cmd" "/c" "start" ,url)
                                  ("cmd" "/c" "start" "chrome" ,@chrome-args))
                       `(("chromium-browser" ,@chrome-args)
                         ("chromium" ,@chrome-args)
                         ("icecat" ,url)
                         ("iceweasel" ,url)
                         ("firefox" ,url)
                         ("mozilla" ,url))))
         (void (%make-void-port "w")))
    (when (or debug? verbose?)
      (format (current-error-port) "ide browse: opening ~s\n" url))
    (let loop ((browsers browsers))
      (unless (null? browsers)
        (unless (zero? (if debug? (apply system* (car browsers))
                           (with-output-to-port void
                             (cute with-error-to-port void
                                   (cute apply system* (car browsers))))))
          (loop (cdr browsers)))))))

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (files (option-ref options '() '()))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (type (if (pair? files) (car files) "system"))
         (url (if (or (string-prefix? "file:" type)
                      (string-prefix? "http:" type)
                      (string-prefix? "https:" type)) type
                      (get-url type))))
    (when verbose?
      (format (current-error-port) "~a\n" url))
    (when (string-null? url)
      (format (current-error-port) "URL is empty\n")
      (exit 2))
    (browse url #:debug? debug? #:verbose? verbose?)))
