;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021,2022 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands system)
  #:use-module (ice-9 getopt-long)
  #:use-module (ide commands daemon)
  #:use-module (ide commander)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide util))

(define (parse-opts args)
  (let* ((option-spec
          '((help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))
            (wfc)))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide system [OPTION]... [FILE]...
Make a system view available.

  -h, --help             display this help and exit
  -I, --import=DIR+      add directory DIR to import path
  -m, --model=MODEL      explore model MODEL

View:
  ~a
" (get-url "system"))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (result EXIT_SUCCESS))

    (define (parse-handler message)
      (let ((parse (parse-message message)))
        (cond
         ((parse? parse)
          (let* ((data (parse-data parse))
                 (status (assoc-ref data "status"))
                 (stderr (assoc-ref data "stderr")))
            (when stderr
              (display stderr (current-error-port)))
            (cond
             ((zero? status)
              (when (or debug? verbose?)
                (format #t "system: no errors found\n"))
              (when (isatty? (current-error-port))
                (format (current-error-port) "See result: ~a\n" (get-url "system"))))
             (else
              (set! result EXIT_FAILURE)))))
         (else
          (format (current-error-port)
                  "ide system: error: received: ~s\n"
                  message)
          (set! result EXIT_FAILURE)))
        message))

    (daemon:start #:debug? debug? #:port %ide-port)
    (let* ((files (option-ref options '() #f))
           (file-name (car files))
           (imports (multi-opt options 'import))
           (model (option-ref options 'model #f))
           (debug? (ide:command-line:get 'debug))
           (wfc? (ide:command-line:get 'wfc))
           (input (piped-file->stream file-name #:debug? debug? #:imports imports))
           (command (make-command "parse" `("dzn"
                                            ,@(if wfc? '() '("--skip-wfc"))
                                            "graph"
                                            "--backend=system"
                                            "--locations"
                                            ,@(if model `("--model" ,model) '())
                                            "--format=json" "-")
                                  #:input input
                                  #:model model))
           (commander (make-commander #:debug? debug? #:verbose? verbose?))
           (result-handler
            (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                     parse-handler))
           (result
            (+ (run-command commander command
                            #:result-handler result-handler
                            #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
               result)))
      (exit result))))
