;;; Dezyne --- Dezyne command line tools
;;; Copyright © 2015, 2016, 2017, 2018 Rutger van Beusekom <rutger@dezyne.org>
;;; Copyright © 2018, 2019 Rob Wieringa <rob@rma.wieringa@gmail.com>
;;; Copyright © 2016 Henk Katerberg <hank@mudball.nl>
;;; Copyright © 2015, 2016, 2017, 2018, 2019, 2022 Jan Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.
;;;
;;; Commentary:
;;;
;;; Code:

(define-module (dzn code v2_8-c++)
  #:use-module (ice-9 curried-definitions)
  #:use-module (ice-9 match)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)

  #:use-module (dzn misc)
  #:use-module (dzn command-line)

  #:use-module ((oop goops) #:renamer (lambda (x) (if (member x '(<port> <foreign>)) (symbol-append 'goops: x) x)))
  #:use-module (dzn goops)
  #:use-module (dzn config)

  #:use-module (dzn ast)
  #:use-module (dzn code)
  #:use-module (dzn code-util)
  #:use-module (dzn code dzn)
  #:use-module (dzn code c++)
  #:use-module (dzn templates))

;;;
;;; Compatibility for v2.8.0.
;;;
(define asd? #f)
(define ast:provided ast:provides-port*)
(define ast:required ast:requires-port*)
(define om:events ast:event*)
(define om:name ast:name)
(define om:in? ast:in?)
(define om:out? ast:out?)
(define om:ports ast:port*)


;;;
;;; Version 2.8 c++ code generator.
;;;
(define-method (.name (o <scope.name>))
  (ast:name o))

(define-method (.scope (o <scope.name>))
  (ast:scope o))

(define-method (om:scope+name o)
 (map string->symbol (ast:full-name o)))

(define-method (code:scope+name o)
  (om:scope+name o))

(define-method (code:scope+name (o <root>))
  (code:file-name o))

(define-method (code:scope+name (o <event>))
  ((compose code:scope+name .signature) o))

(define-method (code:scope+name (o <extern>))
  (list (.value o)))

(define-method (code:scope+name (o <int>))
  '(int))

(define-method (code:scope+name (o <signature>))
  ((compose code:scope+name .type) o))

(define-method (code:scope+name (o <trigger>))
  ((compose code:scope+name .event) o))

(define-method (code:scope+name (o <enum-field>))
  (append ((compose code:scope+name .type) o) (list (.field o))))

(define-method (code:scope+name (o <binding>))
  ((compose code:scope+name .type (cute .instance (parent o <model>) <>) injected-instance-name) o))

(define (om:scope o)
  (drop-right (code:scope+name o) 1))

(define om:variables ast:variable*)

(define (injected-binding? binding)
  (or (equal? "*" (.port.name (.left binding)))
      (equal? "*" (.port.name (.right binding)))))

(define (injected-binding binding)
  (cond ((equal? "*" (.port.name (.left binding))) (.right binding))
        ((equal? "*" (.port.name (.right binding))) (.left binding))
        (else #f)))

(define (injected-bindings model)
  (filter injected-binding? ((compose .elements .bindings) model)))

(define (injected-instance-name binding)
  (or (.instance.name (.left binding)) (.instance.name (.right binding))))

(define-method (c++:name (o <binding>))
  (injected-instance-name o))

(define-method (c++:function-type (o <type>))
  o)

(define-method (c++:function-type (o <trigger>))
  ((compose c++:function-type .signature .event) o))

(define-method (c++:function-type (o <signature>))
  ((compose c++:function-type .type) o))

(define-method (c++:function-type (o <function>))
  ((compose c++:function-type .signature) o))

(define-method (c++:argument_n (o <trigger>))
  (map
   (lambda (f i) (clone f #:name (string-append "_"  (number->string i))))
   (code:formals o)
   (iota (length (code:formals o)) 1 1)))

(define-method (c++:optional-type (o <trigger>))
  (let ((type (ast:type o)))
    (if (is-a? type <void>) '() type)))

(define-method (c++:void-in-triggers (o <component-model>))
  (filter
   (lambda (t) (is-a? ((compose .type .signature .event) t) <void>))
   (append (ast:provided-in-triggers o) (ast:required-out-triggers o))))

(define-templates-macro define-templates v2_8-c++)
(include-from-path "dzn/templates/dzn.scm")
(include-from-path "dzn/templates/code.scm")
(include-from-path "dzn/templates/c++.scm")
(include-from-path "dzn/templates/v2_8-c++.scm")

(define c++:dzn-locator (@@ (dzn code c++) c++:dzn-locator))


;;;
;;; Entry point.
;;;

(define* (ast-> root #:key (dir ".") model)
  "Entry point."

  (code-util:foreign-conflict? root)

  (let ((root (code:om root)))
    (let ((generator (code-util:indenter (cute x:header root)))
          (file-name (code-util:root-file-name root dir ".hh")))
      (code-util:dump root generator #:file-name file-name))

    (when (code-util:generate-source? root)
      (let ((generator (code-util:indenter (cute x:source root)))
            (file-name (code-util:root-file-name root dir ".cc")))
        (code-util:dump root generator #:file-name file-name)))

    (when model
      (let ((model (ast:get-model root model)))
        (when (is-a? model <component-model>)
          (let ((generator (code-util:indenter (cute x:main model)))
                (file-name (code-util:file-name "main" dir ".cc")))
            (code-util:dump root generator #:file-name file-name)))))))
