;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2019, 2020 Jan Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2019 Rob Wieringa <rob.wieringa@verum.com>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.
;;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commander)
  #:use-module (srfi srfi-9 gnu)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 match)
  #:use-module (oop goops)
  #:use-module (8sync)
  #:use-module (8sync systems websocket client)
  #:use-module (ide config)
  #:use-module (ide util)
  #:use-module (dzn runtime)
  #:use-module (command command)
  #:use-module (transport transport)
  #:use-module (transport Websocket)
  #:export (bye?
            bye-data

            error?
            error-command
            error-status
            error-stdout
            error-stderr

            hello?
            hello-status
            hello-stdout
            hello-stderr
            hello-connections

            parse?
            parse-data

            simulate?
            simulate-data

            make-command
            make-commander
            make-command-handler
            make-finalizer
            make-hello-browse-handler
            make-result-handler
            parse-message
            run-command))

(define-actor <decoupler> (<actor>)
  ((decouple decouple)))

(define-method (decouple (o <decoupler>) message thunk)
  (thunk))

(define* (make-command name argv #:key input model trace)
  `(("name" . ,name)
    ("argv" . ,argv)
    ,@(if input `(("input" . ,input)) '())
    ,@(if model `(("model" . ,model)) '())
    ,@(if trace `(("trace" . ,trace)) '())))

(define* (make-commander #:key debug? verbose?)
  (let* ((locator (make <dzn:locator>))
         (locator (if debug? locator
                      (dzn:set! locator (const #t) 'trace)))
         (locator (dzn:set! locator (if (or debug? verbose?) log-verbose (const #t)) 'log))
         (locator (dzn:set! locator (if (or debug? verbose?) log-debug (const #t)) 'log-debug))
         (runtime (make <dzn:runtime>))
         (locator (dzn:set! locator runtime))
         (hive (make-hive))
         (locator (dzn:set! locator hive))
         (decoupler (bootstrap-actor hive <decoupler>))
         (decoupler ((@@ (8sync actors) hive-resolve-local-actor) hive decoupler))
         (locator (dzn:set! locator decoupler)))
    (make <command:Commander> #:locator locator #:name "commander")))

(define (make-command-handler commander command)
  (lambda (message)
    (let ((decoupler (actor-id (dzn:get (.locator commander) <decoupler>))))
      (<- decoupler 'decouple
          (lambda _
            (let ((cmd `(("command" . ,command))))
              (action commander .control .in .command cmd)))))))

(define* (make-finalizer commander #:key debug? verbose?)
  (lambda (message)
    (force-output)
    (force-output (current-error-port))
    (let* ((ws (.websocket (.transprt (.daemon commander))))
           (hive (dzn:get (.locator commander) <hive>))
           (clients (slot-ref ws 'clients)))
      (false-if-exception
        (map (compose self-destruct
                      websocket-close
                      (cute (@@ (8sync actors) hive-resolve-local-actor) hive <>))
             clients))
      message)))

(define* (make-result-handler command #:key debug? verbose?)
  (let ((name (match command
                ((("name" . name) rest ...) name)
                (_ "ide"))))
    (lambda (message)
      (when debug?
        (format (current-error-port) "received: ~s\n" message))
      (when (or debug? verbose?)
        (format (current-error-port) "\ninitialization done\n")
        (force-output (current-error-port))
        (let ((url (get-url name)))
          (when (and (string? url) (not (string-null? url)))
            (format #t "~a\n" url))))
      message)))

(define-immutable-record-type <bye>
  (make-bye data)
  bye?
  (data bye-data))

(define-immutable-record-type <hello>
  (make-hello status stdout stderr connections)
  hello?
  (status hello-status)
  (stdout hello-stdout)
  (stderr hello-stderr)
  (connections hello-connections))

(define-immutable-record-type <parse>
  (make-parse data)
  parse?
  (data parse-data))

(define-immutable-record-type <simulate>
  (make-simulate data)
  simulate?
  (data simulate-data))

(define-immutable-record-type <error>
  (make-error command status stdout stderr)
  error?
  (command error-command)
  (status error-status)
  (stdout error-stdout)
  (stderr error-stderr))

(define (parse-message message)
  (match message
    ((("bye" . bye))
     (make-bye bye))
    ((("hello" alist ...))
     (make-hello (assoc-ref alist "status")
                 (assoc-ref alist "stdout")
                 (assoc-ref alist "stderr")
                 (assoc-ref alist "connections")))
    ((("parse" . (and (? string?) data)))
     (make-parse data))
    ;; XXX FIXME: instead of (<command> "err" "<error>"), send full
    ;; message.  First, commands/verify and commands/simulate must be
    ;; refactored to use commander.
    ((("parse" "err" (and (? string? stderr))))
     (make-error "parse " 1 "" stderr))
    ((("simulate" . (and (? string?) data)))
     (make-simulate data))
    ;; XXX FIXME: instead of (<command> "err" "<error>"), send full
    ;; message.  First, commands/verify and commands/simulate must be
    ;; refactored to use commander.
    ((("simulate" "err" (and (? string? stderr))))
     (make-error "simulate " 1 "" stderr))
    (_ #f)))

(define* (make-hello-browse-handler type #:key debug? verbose?)
  (lambda (message)
    (let ((hello (parse-message message)))
      (cond ((hello? hello)
             (let ((connections (hello-connections hello)))
               (when debug?
                 (format (current-error-port)
                         "connections: ~s\n" connections))
               (unless (member type connections)
                 (when debug?
                   (format (current-error-port)
                           "launching browser ~s\n"
                           type))
                 (system* "ide" "browse" type))))
            (else
             (format (current-error-port)
                     "browse handler: received: ~s\n"
                     message))))))

(define* (run-command commander command  #:key result-handler ide-port debug? verbose?)
  (let* ((hive (dzn:get (.locator commander) <hive>))
         (decoupler (dzn:get (.locator commander) <decoupler>))
         (decoupler (actor-id (dzn:get (.locator commander) <decoupler>)))
         (finalizer (make-finalizer commander #:debug? debug? #:verbose? verbose?))
         (result-handler (or result-handler
                             (compose finalizer
                                      (make-result-handler command #:debug? debug? #:verbose? verbose?))))
         (name (match command
                 ((("name" . name) rest ...) name)
                 (_ "ide")))
         (setup-handler (make-command-handler commander command))
         (result EXIT_SUCCESS))

    (define (error-handler message)
      (format (current-error-port) "ide ~a: error: ~a\n" name message)
      (set! result EXIT_FAILURE)
      message)

    (set! (.set_up (.out (.control commander))) setup-handler)
    (set! (.result (.out (.control commander))) result-handler)
    (set! (.error (.out (.control commander))) (compose finalizer error-handler))

    (run-hive hive (list (bootstrap-message
                          hive decoupler 'decouple
                          (lambda _
                            (action commander .control .in .setup
                                    (format #f "ws://127.0.0.1:~a" ide-port))))))
    result))
