;;; Dezyne-IDE --- An IDE for Dezyne
;;; Copyright © 2020 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; Dezyne-IDE is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with Dezyne-IDE.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:
;;;
;;; Dezyne Language Server for the Language Server Protocol
;;; https://microsoft.github.io/language-server-protocol
;;;
;;; Code:

(define-module (ide commands lsp)
  #:use-module (ice-9 getopt-long)
  #:use-module (ide command-line)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide commands daemon)
  #:use-module (ide lsp))

(define (parse-opts args)
  (let* ((option-spec
          '((help         (single-char #\h))
            (import       (single-char #\I) (value #t))
            (relay-daemon (single-char #\r))))
         (options (getopt-long args option-spec))
         (help?   (option-ref options 'help #f))
         (files   (option-ref options '() '()))
         (usage?  (and (not help?) (pair? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide lsp [OPTION]...
Run a Language Protocol Server for Dezyne over stdio.

  -h, --help                      display this help and exit
  -I, --import=DIR+               add DIR to import path
  -r, --relay-daemon              relay editor messages from daemon
")
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (main args)
  (daemon-config-set!)
  (let* ((options     (parse-opts args))
         (debug?      (ide:command-line:get 'debug))
         (relay?      (option-ref options 'relay-daemon #f))
         (editor-port (and relay? %editor-port))
         (imports     (multi-opt options 'import))
         (files       (option-ref options '() '()))
         (file-name   (and (pair? files) (car files))))
    (when relay?
      (daemon:start #:debug? debug? #:port %ide-port))
    (lsp:stdio #:debug? debug? #:editor-port editor-port #:file-name file-name #:imports imports)))
