;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2019, 2020 Jan Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2019 Rob Wieringa <rob.wieringa@verum.com>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.
;;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands system)
  #:use-module (ice-9 getopt-long)
  #:use-module (gaiag parse)
  #:use-module (ide commands daemon)
  #:use-module (ide commander)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide util))

(define (parse-opts args)
  (let* ((option-spec
          '((help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide system [OPTION]... [FILE]...
Make a system view available.

  -h, --help             display this help and exit
  -I, --import=DIR+      add directory DIR to import path
  -m, --model=MODEL      explore model MODEL

View:
  ~a
" (get-url "system"))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (result EXIT_SUCCESS))

    (define (parse-handler message)
      (log-debug  "received: ~s\n" message)
      (let ((parse (parse-message message)))
        (cond ((parse? parse)
               (when (or debug? verbose?)
                 (format #t "system: no errors found\n"))
               (when verbose?
                 (format (current-error-port) "See result: ~a\n" (get-url "system"))))
              ((error? parse)
               (display (error-stderr parse) (current-error-port))
               (set! result EXIT_FAILURE))
              (else
               (format (current-error-port)
                       "ide parse: error: received: ~s\n"
                       message)
               (set! result EXIT_FAILURE)))
        message))

    (daemon:start #:debug? debug? #:port %ide-port)
    (let* ((files (option-ref options '() #f))
           (file-name (car files))
           (imports (multi-opt options 'import))
           (model (option-ref options 'model #f))
           (no-browser? (ide:command-line:get 'no-browser #f))
           (peg? (ide:command-line:get 'peg #f))
           (debug? (ide:command-line:get 'debug))
           (input (file->stream file-name #:debug? debug? #:imports imports))
           (command (make-command "parse" `("dzn"
                                            ,@(if peg? '("--peg") '())
                                            "code"
                                            "--locations"
                                            "--language=json" "-")
                                  #:input input))
           (commander (make-commander #:debug? debug? #:verbose? verbose?))
           (result-handler
            (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                     parse-handler))
           (result
            (+ (run-command commander command
                            #:result-handler result-handler
                            #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
               result))
           (result
            (if (or no-browser? (not (zero? result))) result
                (let* ((command (make-command "hello" '("dzn" "hello")))
                       (commander (make-commander #:debug? debug? #:verbose? verbose?))
                       (result-handler
                        (compose (make-finalizer commander
                                                 #:debug? debug? #:verbose? verbose?)
                                 (make-hello-browse-handler
                                  "system"
                                  #:debug? debug? #:verbose? verbose?))))
                  (+ (run-command commander command
                                  #:result-handler result-handler
                                  #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
                     result)))))
      (exit result))))
