(define-module (ide pipe)
  #:use-module (ice-9 popen)
  #:use-module (srfi srfi-1)
  #:use-module (gaiag shell-util)
  #:export (blocking-pipe?
            pipe->fdes
            pipeline*
            non-blocking!
            unbuffered!))

(eval-when (expand load eval)
  (load-extension (string-append "libguile-" (effective-version))
                  "scm_init_popen"))

(define (pipe->fdes)
  (let ((p (pipe)))
   (cons (port->fdes (car p))
         (port->fdes (cdr p)))))

(define (pipeline* commands)
  "Execute a pipeline of @var(commands) -- where each command is a
list of a program and its arguments as strings or a lambda to be
executed by the forked guile process -- returning an input port to the
end of the pipeline, an output port to the beginning of the pipeline
and a list of PIDs of the processes executing the @var(commands)."
  (let* ((to (pipe->fdes))
         (pipes (map (lambda _ (pipe->fdes)) commands))
	 (pipeline (fold (lambda (from proc prev)
                           (let* ((to (car prev))
                                  (pids (cdr prev))
                                  (pid (piped-process (car proc)
                                                      (cdr proc)
                                                      from
                                                      to)))
                             (cons from (cons pid pids))))
                         `(,to)
                         pipes
                         commands))
	 (from (car pipeline))
	 (pids (cdr pipeline)))
    (values (fdes->inport (car from)) (fdes->outport (cdr to)) pids)))

(define (blocking-pipe?)
  (mingw?))

(define (non-blocking! port)
  (unless (blocking-pipe?)
    (let ((flags (fcntl port F_GETFL)))
      (fcntl port F_SETFL (logior O_NONBLOCK flags))))
  port)

(define (unbuffered! port)
  (setvbuf port 'none)
  port)
