(define-module (transport libtransport)
  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)
  #:use-module (oop goops)
  #:use-module (json)
  #:use-module (gaiag misc)
  #:use-module (gaiag commands trace)
  #:duplicates (merge-generics)
  #:export (libtransport:serialize
            libtransport:unserialize))

(define parser-read-char (@@ (json parser) parser-read-char))
(define parser-peek-char (@@ (json parser) parser-peek-char))
(define read-pair (@@ (json parser) read-pair))

(define (read-object parser)
  (let loop ((c (parser-peek-char parser))
             (alist '()))
    (case c
      ;; Skip whitespaces
      ((#\ht #\vt #\lf #\cr #\sp)
       (parser-read-char parser)
       (loop (parser-peek-char parser) alist))
      ;; end of object
      ((#\})
       (parser-read-char parser)
       (reverse alist))
      ;; Read one pair and continue
      ((#\")
       (let ((pair (read-pair parser)))
         (loop (parser-peek-char parser) (cons pair alist))))
      ;; Skip comma and read more pairs
      ((#\,)
       (parser-read-char parser)
       (loop (parser-peek-char parser) alist))
      ;; invalid object
      (else (throw 'json-invalid parser)))))

(module-define! (resolve-module '(json parser)) 'read-object read-object)

(define (libtransport:serialize o)
  ;; (format (current-error-port) "libtransport:serialize o=~s\n" o)
  (catch 'json-invalid
    (lambda _ (scm->json-string o))
    (lambda (key . sexp)
      (format (current-error-port) "json-invalid: sexp=~s\n" sexp)
      "\"\"")))

(define (libtransport:unserialize o)
  ;; (format (current-error-port) "libtransport:unserialize o=~s\n" o)
  (catch 'json-invalid
    (lambda _
      (json-string->alist-scm o))
    (lambda (key parser)
      (let ((data (read-string (json-parser-port parser))))
        (if (string? data) data
            (begin
              (format (current-error-port) "json-invalid: parser=~s\n" data)
              '()))))))
