/*
 * Copyright (C) 2020,2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

let SDColors = {
  state: { fill: '#FEFECE', stroke: '#A80036' },
  transition: { fill: '#FFFFFF', stroke: '#AAAAAA' },
  state2transition: { stroke: '#A80036' },
  transition2state: { stroke: '#106588' },
};

class SDNode extends Viz {
  constructor(data) {
    super(data); // {id}
    this.klass = 'SDNode';
    this.id = data.id;
    this.node = new Node(data.id);
  }

  setDiagram(diagram) {
    this.diagram = diagram;
    this.node.setGraph(diagram.graph);
  }

  setGraph(graph) {
    this.graph = graph;
    this.node.setGraph(graph);
  }

  drag(x, y) {
    let bnd = this.bounds();
    this.node.pos.x = x + bnd.width/2;
    this.node.pos.y = y + bnd.height/2;
    this.refresh();
    this.diagram.graph.restart(); // reset time
  }

  refresh() {
    let bnd = this.bounds();
    this.move(this.node.pos.x - bnd.width/2, this.node.pos.y - bnd.height/2);
  }
}

class State extends SDNode {
  constructor(data) {
    super(data); // {id, state}
    this.klass = 'State';
    this.node.mass = 4; // TODO: fine-tune
    this.node.charge = 4;
  }

  initViz() {
    // TODO: string manupulation, prefer JSON format
    let states = this.data.state.replace(/,\n/g,',').split('\n');
    let pairs = states.map(state => {
      let inst = (state[0] == '[') ? '' : state.substr(0,state.indexOf('='));
      let val = (state[0] == '[') ? state : state.substr(state.indexOf('=')+1);
      val = val.substr(1,val.length-2);
      let vals = (val.length==0) ? [] : val.split(',');
      return [inst,vals];
    });

    let triples = pairs.map(pair => {
      let inst = pair[0];
      let variables = pair[1].map(vv => vv.split('=')[0]);
      let values = pair[1].map(vv => vv.split('=')[1]);
      return [inst, variables.join('\n'), values.join('\n')];
    });
    // TODO: make filtering empty states an option
    triples = triples.filter(triple => triple[1] != '');
    
    let content = triples.map(tr => tr.map(elt => new Text(elt)));
    
    let state = new Table(content);
    let box = new RoundedBoundingBox(state);
    box.padding = 5;
    box.color = SDColors.state.fill;
    box.strokeColor = SDColors.state.stroke;
    box.strokeWeight = 1;
    box.round = 10;
    this.setViz(box);
    this.refresh();
    this.node.radius = Math.sqrt(this.bounds().width*this.bounds().width + this.bounds().height*this.bounds().height)/2;
    this.node.charge = this.node.radius/8; // TODO: fine-tune
  }

}

class InitialState extends State {
  constructor(data) {
    super(data); // {id, state}
    this.klass = 'InitialState';
  }

  initViz() {
    let dot = new Circle(this.p);
    dot.radius = 15;
    dot.color = 'black';
    this.setViz(dot);
    this.node.radius = dot.radius;
    this.refresh();
  }
}

class Transition extends SDNode {
  constructor(data) {
    super(data); // {id, trigger, action, from, to}
    this.klass = 'Transition';
    this.node.mass = 2; // TODO: fine-tune
    this.node.charge = 2;
  }

  initViz() {
    let trigger = new Text(this.data.trigger);
    let action = new Text(this.data.action);
    let transition = new Table([[trigger], [action]]);
    
    let box = new RoundedBoundingBox(transition);
    box.padding = 5;
    box.color = SDColors.transition.fill;
    box.strokeColor = SDColors.transition.stroke;
    box.strokeWeight = 0;
    box.shadowed = false;
    box.round = 10;
    this.setViz(box);
    this.refresh();
    this.node.radius = Math.sqrt(this.bounds().width*this.bounds().width + this.bounds().height*this.bounds().height)/2;
    this.node.charge = this.node.radius/30; // TODO: fine-tune
  }

  hover(on) {
    this.viz.strokeColor = SDColors.transition.stroke;
    this.viz.strokeWeight = on ? 1 : 0;
  }
}

class Waypoint extends Viz {
  constructor(data) {
    super(data); // {id, radius, color}
    this.klass = 'Waypoint';
    this.id = data.id;
    this.node = new Node(data.id);
    this.node.mass = .3; // TODO: fine-tune
    this.node.charge = .3; // TODO: fine-tune
  }

  setDiagram(diagram) {
    this.diagram = diagram;
    this.node.setGraph(diagram.graph);
  }

  setGraph(graph) {
    this.graph = graph;
    this.node.setGraph(graph);
  }

  initViz() {
    let dot = new Circle(this.p);
    dot.radius = this.data.radius;
    dot.color = this.data.color;
    this.setViz(dot);
    this.refresh();
    this.node.radius = this.data.radius;
  }

  refresh() {
    let bnd = this.bounds();
    this.move(this.node.pos.x - bnd.width/2, this.node.pos.y - bnd.height/2);
  }
}

class Segment extends Viz {
  constructor(data) {
    super(data); // {from, to, color, index, total}
    this.klass = 'Segment';
    this.fromNode = data.from;
    this.toNode = data.to;
    this.edge = new Edge(data.from.id, data.to.id);
    this.edge.length = 1;
    this.edge.stiffness = 12;
  }

  setDiagram(diagram) {
    this.diagram = diagram;
    this.edge.setGraph(diagram.graph);
  }

  setGraph(graph) {
    this.graph = graph;
    this.edge.setGraph(graph);
  }

  initViz() {
    let arrow = new DirectedRectLine(this.fromNode.x + this.fromNode.width/2,
                                     this.fromNode.y + this.fromNode.height/2,
                                     this.toNode.x + this.toNode.width/2,
                                     this.toNode.y + this.toNode.height/2);
    let t0 = arrow.thicknessStart;
    let t1 = arrow.thicknessEnd;
    arrow.thicknessStart = (t0*(this.data.total-this.data.index) + t1*this.data.index)/this.data.total;
    arrow.thicknessEnd = (t0*(this.data.total-this.data.index-1) + t1*(this.data.index+1))/this.data.total;
    arrow.color = this.data.color;
    this.setViz(arrow);
  }

  refresh() {
    this.viz.refreshPoints(this.fromNode.x + this.fromNode.width/2,
                           this.fromNode.y + this.fromNode.height/2,
                           this.toNode.x + this.toNode.width/2,
                           this.toNode.y + this.toNode.height/2);
  }

  show(p) {
    this.refresh();
    this.viz.show(p);
  }
}

class Connection extends Viz {
  constructor(data) {
    super(data); // {from, to, color, nrPoints}
    this.klass = 'Connection';
    this.segments = []; // [{Edge, SDNode, SDNode, Viz}]
    this.points = []; // [Waypoint]
    this.constructBetween(data.from, data.to, data.nrPoints);
  }

  constructBetween(fromNode, toNode, nrPoints) {
    let prevNode = fromNode;
    for (let i = 0; i < nrPoints; i++) {
      let wpt = new Waypoint({id: '' + fromNode.id + '.' + i + '.' + toNode.id,
                              radius: 3*(nrPoints-i)/(nrPoints+1),
                              color: this.data.color});
      this.points.push(wpt);
      this.segments.push(new Segment({from: prevNode, to: wpt, color: this.data.color, index: i, total: nrPoints+1}));
      prevNode = wpt;
    }
    this.segments.push(new Segment({from: prevNode, to: toNode, color: this.data.color, index: nrPoints, total: nrPoints+1}));
    this.segments.forEach(con => con.edge.length /= (nrPoints+1));
  }

  setDiagram(diagram) {
    this.diagram = diagram;
    this.segments.forEach(seg => seg.setDiagram(diagram));
    this.points.forEach(point => point.setDiagram(diagram));
  }

  setGraph(graph) {
    this.graph = graph;
    this.segments.forEach(seg => seg.setGraph(graph));
    this.points.forEach(point => point.setGraph(graph));
  }

  initViz() {
    let vizList = [];
    this.points.forEach(point => {
      point.initViz();
      vizList.push(point);
    });
    this.segments.forEach(seg => {
      seg.initViz();
      vizList.push(seg);
    });
    this.setViz(new Group(vizList));
  }

  refresh() {
    this.points.forEach(pt => pt.refresh());
    this.segments.forEach(seg => seg.refresh());
  }

  show(p) {
    this.refresh();
    this.viz.show(p);
  }
}

class StateDiagram extends Viz {
  constructor(data) {
    super(data); // {states, transitions}
    this.klass = 'StateDiagram';
    this.states = data.states.map(state => (state.id == '*') ? new InitialState(state) : new State(state));

    this.transitions = [];
    this.connections = [];
    data.transitions.forEach((trans, i) => {
      let fromState = this.states.find(s => s.id == trans.from);
      let toState = this.states.find(s => s.id == trans.to);
      if (fromState.klass == 'InitialState') {
        this.connections.push(new Connection({from: fromState, to: toState, color: SDColors.transition2state.stroke, nrPoints: 0}));
      } else {
        let node = new Transition({id: 't'+i,
                                   trigger: trans.trigger,
                                   action: trans.action,
                                   from: trans.from,
                                   to: trans.to,
                                   location: trans.location});
        this.transitions.push(node);
        let nrp = (trans.from == trans.to) ? 1 : 0;
        this.connections.push(new Connection({from: fromState, to: node, color: SDColors.state2transition.stroke, nrPoints: nrp}));
        this.connections.push(new Connection({from: node, to: toState, color: SDColors.transition2state.stroke, nrPoints: nrp}));
      }
    });
    this.SDNodes = this.states.concat(this.transitions);
    let nodes = this.SDNodes.map(n => n.node);
    this.connections.map(con => con.points).forEach(points => {
      nodes = nodes.concat(points.map(pt => pt.node));
    })

    let edges = [];
    this.connections.map(con => con.segments).forEach(segs => {
      edges = edges.concat(segs.map(seg => seg.edge));
    })

    this.graph = new Graph(nodes, edges);
    this.SDNodes.forEach(n => n.setDiagram(this));
    this.connections.forEach(con => con.setDiagram(this));
    this.frame;
    this.energyPerNode = 0;
  }

  initViz() {
    this.SDNodes.forEach(n => n.initViz());
    this.connections.forEach(con => con.initViz());
    this.group = new Group(this.connections.concat(this.SDNodes));
    this.setViz(this.group);
  }

  lookupSDNode(id) {
    return this.SDNodes.find(s => (s.data.id == id));
  }

  refresh() {
    this.energyPerNode = this.graph.update();
    this.viz.refresh();
  }

  copyLayout(oldDiagram)
  {
    this.states.forEach(st => {
      let ost = oldDiagram.states.find(ost => st.id == ost.id);
      st.node.pos.x = ost.node.pos.x * 2;
      st.node.pos.y = ost.node.pos.y * 2;
      st.node.charge = st.node.radius/30; // TODO: fine-tune
      st.refresh();
    });
    this.transitions.forEach(tr => {
      let fromState = this.states.find(s => s.id == tr.data.from);
      let toState = this.states.find(s => s.id == tr.data.to);
      if (fromState) {
        tr.node.pos.x = (fromState.node.pos.x + toState.node.pos.x)/2;
        tr.node.pos.y = (fromState.node.pos.y + toState.node.pos.y)/2;
      }
      tr.refresh();
    });
    this.connections.forEach(con => {
      // for now trust that there is at most one Waypoint
      if (con.points.length > 0) {
        con.points[0].node.pos.x = (con.data.from.node.pos.x + con.data.to.node.pos.x)/2;
        con.points[0].node.pos.y = (con.data.from.node.pos.y + con.data.to.node.pos.y)/2;
        con.points[0].refresh();
      }
      con.segments.forEach(seg => {
        seg.edge.length = (seg.fromNode.node.radius + seg.toNode.node.radius)/2;
        seg.edge.stiffness = 12;
      });
    });
  }
}

class StateDiagramSimple extends Viz {
  constructor(data) {
    super(data); // {states, transitions}
    this.klass = 'StateDiagramSimple';
    this.states = data.states.map(state => (state.id == '*') ? new InitialState(state) : new State(state));

    this.transitions = [];
    this.connections = [];
    data.transitions.forEach((trans, i) => {
      let fromState = this.states.find(s => s.id == trans.from);
      let toState = this.states.find(s => s.id == trans.to);
      this.connections.push(new Connection({from: fromState, to: toState, color: 'black', nrPoints: 0}));
    });
    let nodes = this.states.map(n => n.node);
    this.connections.map(con => con.points).forEach(points => {
      nodes = nodes.concat(points.map(pt => pt.node));
    })

    let edges = [];
    this.connections.map(con => con.segments).forEach(segs => {
      edges = edges.concat(segs.map(seg => seg.edge));
    })

    this.graph = new Graph(nodes, edges);
    this.states.forEach(n => n.setDiagram(this));
    this.connections.forEach(con => con.setDiagram(this));
    this.frame;
    this.energyPerNode = 0;
  }

  initViz() {
    this.states.forEach(n => n.initViz());
    this.connections.forEach(con => con.initViz());
    this.group = new Group(this.connections.concat(this.states));
    this.setViz(this.group);
  }

  refresh() {
    this.energyPerNode = this.graph.update();
    this.viz.refresh();
  }
}
