// Dezyne-IDE --- An IDE for Dezyne
//
// Copyright © 2019,2020 Rob Wieringa <Rob.Wieringa@verum.com>
// Copyright © 2019 Johri van Eerd <johri.van.eerd@verum.com>
//
// This file is part of Dezyne-IDE.
//
// Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
// All rights reserved.

function trace2seq(trace_lseligible) {

  var trace = trace_lseligible.seqdiag;
  var eligibles = trace_lseligible.lseligible || '';

  function role(inst) {
    var foo = lanes.filter(function(o) {return o.key==inst});
    return foo && foo.length && foo[0].role || 'component';
  }

  var type2msg = Object();
  type2msg['deadlock'] = 'Deadlock';
  type2msg['livelock'] = 'Livelock';
  type2msg['deterministic'] = 'Guard is non-deterministic';
  type2msg['illegal'] = 'Action is illegal';
  type2msg['compliance'] = 'Interface and component mismatch';

  function get_state(instance) {
    var o = trace.find(function(o) { return o.instance == instance; });
    return (o) ? o.state : [];
  }

  function mergeerror(m, msg) {
    var errorMsg = msg.message;
    if (errorMsg in type2msg) {
      errorMsg = type2msg[errorMsg];
    }
    errorMsg = errorMsg.replace(/,/g, ', ');
    if (m.length == 0) {
      var message = {
        kind: 'error',
        from: lanes[1].key,
        fromindex:1,
        to: lanes[1].key,
        toindex:1,
        time:1,
        text: '',
        state:[],
        error: errorMsg
      };
      m.push(message);
    }
    else {
      m[m.length-1].error = errorMsg;
    }
  }

  function isinterface(m) {
    for(var i = 0; i < m.length; ++i) {
      if(m[i].kind == 'TraceType') return m[i].type == 'interface';
    }
    return false;
  }

  function isunconstrained(m) {
    for(var i = 0; i < m.length; ++i) {
      if(m[i].kind == 'TraceType') return m[i].type == 'unconstrained';
    }
    return false;
  }

  var initial_state = Object();
  trace
    .filter(function(o) {return o.kind == 'Initialize';})
    .forEach(function(o) {
      initial_state[o.instance] = o.state;
    });

  var cumulative_state = initial_state;
  trace.forEach(function(o) {
    if (o.instance && o.state) {
      cumulative_state[o.instance] = o.state;
      o.cumulative_state = copy_state(cumulative_state);
    }
  });


  var errormessage = (trace.length > 0 && trace[trace.length-1].kind == 'Error') ? trace[trace.length-1] : null;

  if ((errormessage && errormessage.message && errormessage.message.indexOf('not performed')!=-1) && trace.length > 1) {
    trace[trace.length-2].missing = true;
  }

  var intfclient = 'Client';

  var lanes =
      trace
      .filter(function(o) {return o.kind == 'Initialize';})
      .map(function(o) {
        return {'key': o.instance,'role': o.role};
      });
  if (isinterface(trace)) {
    lanes.unshift({'key': intfclient, 'role': 'provided'});
  }
  lanes = lanes || [];

  var foo = lanes.filter(function(o) {return o.role=='component'});
  var component = foo && foo.length && foo[0].key;

  function is_component_step(step) {return role(step.instance) == 'component';}
  function is_required_port_step(step) {return /required/i.test(role(step.instance));}
  function is_required_port_send(step) {return step.kind == 'ActionStatement' && is_required_port_step(step);}
  function is_external_port_step(step) {return /external/i.test(role(step.instance));}

  function is_arrow(step) {
    if (step.missing) return true;
    if (isinterface(trace)) return step.synchronization;

    // For component traces arrow draw is triggered by component
    // step. (Port steps only used for program text location.)
    if (step.synchronization) return is_component_step(step);
    if (step.kind == 'Dequeue') return is_component_step(step);
    if (is_required_port_send(step)) return is_external_port_step(step);
    return false
  }

  function is_builtin(step) {
    return (step.synchronization && step.synchronization == 'illegal');
  }

  function is_program_step(step) {
    return(step.kind != 'TraceType' && step.kind != 'Initialize' && step.kind != 'Error');
  }

  var time = 0.0;
  var time_tick = 1.4;
  trace = trace.map(function(o, i, a) {
    // Add time to all program steps in trace
    // Time moves forward on synchronization step completion.

    if (is_program_step(o)) {
      o.time = time;
      // ThreadExit and Block inherit time from last synchronization step on its lifeline.
      if (o.kind == 'ThreadExit' || o.kind == 'Block') {
        for (var idx = i; idx >= 0; --idx) {
          if (a[idx].instance == o.instance &&
              (a[idx].synchronization || a[idx].kind == 'ThreadEnter')) {
            o.time = a[idx].time;
            break;
          }
        }
      }
      // Port synchronization steps on arrows pointing from
      // component to required port inherit time from the related
      // component step. (I.e. required port ThreadEnter.)
      if (o.kind == 'ThreadEnter' && /required/i.test(role(o.instance))) {
        for (var idx = i-1; idx >= 0; --idx) {
          if (a[idx].synchronization) {
            o.time = a[idx].time;
            break;
          }
        }
      }
      // Port synchronization steps on arrows pointing from
      // component to provided port inherit time from the related
      // component step.
      if (o.synctype && o.synctype != 'call' && /provided/i.test(role(o.instance)) && !o.missing) {
        for (var idx = i-1; idx >= 0; --idx) {
          if (a[idx].synchronization) {
            o.time = a[idx].time;
            break;
          }
        }
      }
      // Required port ThreadEnter for tau-triggered
      // notifications will inherit time from the first
      // notification sent by that thread context.
      if (o.synctype && o.synctype == 'notification' && /required/i.test(role(o.instance))) {
        port = o.instance;
        for (var idx = i-1; idx >= 0; --idx) {
          if (a[idx].instance == port) {
            if (a[idx].kind == 'ThreadEnter' && a[idx].synctype != 'call') {
              a[idx].time = time;
              break;
            }
            if (a[idx].synchronization)
              // o is not the first notification sent by this call context
              break;
          }
        }
      }

      if (is_arrow(o)) time += time_tick;
    }
    return o;
  });

  function copy_state(state) {
    var result = {};
    for (var instance in state) {
      result[instance] = state[instance];
    }
    return result;
  }

  var messages =
      trace
      .filter(function(o) {
        return (o.synchronization ||
                o.kind == 'Dequeue' ||
                (o.kind == 'ActionStatement' && is_required_port_step(o)));
      })
      .map(function(o, i, a) {
        if (!is_arrow(o)) return null; // => skip steps not used to draw arrow

        var from = null;
        var to = null;
        var fromindex = null;
        var toindex = null;
        var kind = null;
        var text = null;

        var missing = false;
        if (o.missing) {
          from = component;
          to = o.instance;
          fromindex = o.selection[0].index;
          toindex = o.selection[0].index;

          kind = o.synctype;
          text = o.synchronization;
          missing = true;
        } else if (isinterface(trace)) {
          function is_call(step) {return step.synctype == 'call';}

          from = is_call(o) ? intfclient : o.instance;
          to = is_call(o) ? o.instance : intfclient;
          fromindex = o.selection[0].index;
          toindex = o.selection[0].index;

          kind = o.synctype;
          text = o.synchronization;
        } else {
          function short_msg(step) {return step.synchronization.replace(/.*[.]/g, '');}
          function get_port(step) {return step.synchronization.replace(/[.].*/, '')};
          function is_into_component(step) {
            if (is_builtin(step)) return false;
            if (/provided/i.test(role(get_port(step)))) return step.synctype == 'call';
            if (/required/i.test(role(get_port(step)))) return step.synctype != 'call';
            return false;
          }
          // Synchronization steps appear pair-wise: once for the
          // port and once for the component. Synchronization step
          // pair order = (from, to). Dequeue steps appear once.
          if (o.kind == 'Dequeue') {
            from = o.instance;
            to = o.instance;
            fromindex = o.selection[0].index;
            toindex = o.selection[0].index;

            kind = 'call';
            text = o.event;
          } else if (is_required_port_send(o)) {
            from = o.instance;
            to = o.instance;
            fromindex = o.selection[0].index;
            toindex = o.selection[0].index;

            kind = 'call';
            text = o.event;
          }
          else {
            if (is_component_step(o)) {
              from = is_builtin(o) ? o.instance : is_into_component(o) ? get_port(o) : o.instance;
              to = is_builtin(o) ? o.instance : is_into_component(o) ? o.instance : get_port(o);
              if (isunconstrained(trace)) {
                fromindex = o.selection[0].index;
                toindex = o.selection[0].index;
              } else {
                fromindex = (is_builtin(o) ? o.selection[0].index :
                             is_into_component(o) ?
                             (a[i-1].kind == 'Receive' && !is_external_port_step(a[i-1])
                              ? a[i-2].selection[0].index // ActionStatement
                              : a[i-1].selection[0].index) :
                             o.selection[0].index);
                toindex = (is_builtin(o) ? o.selection[0].index :
                           is_into_component(o) ? o.selection[0].index :
                           i+1 < a.length ? a[i+1].selection[0].index :
                           o.selection[0].index);
              }
              kind = o.synctype;
              text = short_msg(o);
            }
            else {
              function is_call(step) {return step.synctype == 'call';}

              from = is_call(o) ? intfclient : o.instance;
              to = is_call(o) ? o.instance : intfclient;
              fromindex = o.selection[0].index;
              toindex = o.selection[0].index;

              kind = o.synctype;
              text = o.synchronization;
            }
          }
        }
        return {
          kind:kind,
          from:from,
          fromindex:fromindex,
          to:to,
          toindex:toindex,
          text:text,
          time:o.time,
          state: o.cumulative_state,
          missing:missing,
        };
      })
      .filter(function(o){return o;}); // remove 'null' entries


  if (errormessage) mergeerror(messages, errormessage);


  var activation_bars =
      trace.filter(function(o){return (o.kind == 'ThreadEnter' ||
                                       o.kind == 'ThreadExit' ||
                                       o.kind == 'Block' ||
                                       o.kind == 'Release' ||
                                       o.kind == 'Dequeue' ||
                                       o.synchronization)})
      .map(function(o, i, a) {
        // Add one activation bar for each ThreadEnter except for
        // provided port tau. Duration until ThreadExit or Block.
        // Each blocked thread Release/ThreadExit pair in current
        // thread scope will be merged into this activation bar.
        if (o.kind == 'ThreadEnter' &&
            !(o.event == 'tau' && /provided/i.test(role(o.instance)))) {
          function active_duration() {
            nPendingThreadExit = 1;
            for (var idx = i; idx < a.length; ++idx) {
              if (a[idx].instance == o.instance) {
                duration = a[idx].time - o.time;
                if (a[idx].kind == 'Block') return duration;
                if (a[idx].kind == 'Release') ++nPendingThreadExit;
                if (a[idx].kind == 'ThreadExit') {
                  if (nPendingThreadExit == 1) return duration;
                  else --nPendingThreadExit;
                }
              }
            }
            return time - o.time;
          }

          if (active_duration() < time_tick/2) return null;

          return {category:'activity',
                  group:o.instance,
                  start:o.time,
                  duration:active_duration(),
                  thread:/provided/i.test(role(o.instance)) ? 'blocked' : 'active'};
        }

        // Add one blocked bar for each component call to required
        // port with duration until its return.
        if (o.synctype && o.synctype == 'call' && role(o.instance) == 'component') {
          function block_duration() {
            for (var idx = i+1; idx < a.length; ++idx) {
              if (a[idx].synctype && a[idx].synctype == 'return' &&
                  a[idx].instance == o.instance) {
                return a[idx].time - o.time;
              }
            }
            return time - o.time;
          }
          return {category:'activity',
                  group:o.instance,
                  start:o.time,
                  duration:block_duration(),
                  thread:'blocked'};
        }

        return null;
      })
      .filter(function(o){return o;}); // remove 'null' entries


  var lane_nr = 0;
  var lifelines = lanes
      .map(function(o) {
        var nr = lane_nr; ++lane_nr;
        return {category: 'lifeline',
                key: o.key,
                isGroup: true,
                loc: '',
                lane_nr: nr,
                duration: time,
                start: 0};
      });


  // take care: headers use final (cumulative) state.
  var lane_nr = 0;
  var headers = lanes
      .map(function(o) {
        var nr = lane_nr; ++lane_nr;
        return {category: 'header',
                key: 'H'+o.key,
                group: o.key,
                text: o.key,
                loc: '',
                lane_nr: nr,
                role: o.role,
                state: cumulative_state[o.key]};
      });

  var eligibleperport = [];

  buttons = [];
  var model = component || lanes[1].key;
  buttons.push({category:'eligible',
                key:'--back--',
                group:model,
                name:'back',
                time:1
               });
  eligibleperport[model] = 1;

  eligibles.split('\n')
    .filter(function(eligible){ return eligible; })
    .forEach(function(eligible) {
      if (eligible.indexOf('.') == -1) {
        var event = eligible;
        var port = lanes[1].key;
      } else {
        var event = eligible.split('.').pop();
        var port = eligible.split('.')[0];
      }

      var time = (eligibleperport[port] || 0) + 1;
      eligibleperport[port] = time;

      buttons.push({category:'eligible',
                    key:eligible,
                    group:port,
                    name:event,
                    time:time
                   });
    });

  console.log('---------------------------------------');
  console.log('lifelines = %j', lifelines);
  console.log('headers = %j', headers);
  console.log('messages = %j', messages);
  console.log('activation_bars = %j', activation_bars);
  console.log('buttons = %j', buttons);
  console.log('---------------------------------------');

  return JSON.stringify({nodeDataArray: [].concat(lifelines, headers, activation_bars, buttons), linkDataArray: messages}, null, 2);
}
