;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2019,2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2019,2020 Rob Wieringa <Rob.Wieringa@verum.com>
;;; Copyright © 2019,2020 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;; Copyright © 2020,2021 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (daemon Data)
  #:use-module (oop goops)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 match)
  #:use-module (8sync)
  #:use-module (dzn runtime)
  #:use-module (dzn decoupler)
  #:use-module (ide util)
  #:use-module (ide disable-imports)
  #:use-module (transport libtransport)
  #:use-module (transport libenvelope)
  #:use-module (daemon daemon_interface)
  #:duplicates (merge-generics)
  #:export (<daemon:Data>
            .storedata))

(define-class <daemon:Data> (<dzn:component>)
  (storedata #:accessor .storedata #:init-form (make <daemon:Idata>) #:init-keyword #:storedata)

  (hello-command #:accessor .hello-command #:init-value #f)
  (parse-command #:accessor .parse-command #:init-value #f)
  (simulate-command #:accessor .simulate-command #:init-value #f)
  (state-command #:accessor .state-command #:init-value #f)
  (verify-command #:accessor .verify-command #:init-value #f)

  (request-data #:accessor .request-data #:init-value (list))
  (selection-data #:accessor .selection-data #:init-value #f)

  (hello-data #:accessor .hello-data #:init-value #f)
  (parse-data #:accessor .parse-data #:init-value #f)
  (verify-data #:accessor .verify-data #:init-value #f)
  (simulate-data #:accessor .simulate-data #:init-value #f)
  (state-data #:accessor .state-data #:init-value #f)
  (backintrace-data #:accessor .backintrace-data #:init-value #f)
  (replaytrace-data #:accessor .replaytrace-data #:init-value #f))

(define-method (initialize (o <daemon:Data>) args)
  (next-method)
  (set! (.storedata o)
        (make <daemon:Idata>
          #:in (make <daemon:Idata.in>
                 #:name "storedata"
                 #:self o
                 #:store_command (lambda args (call-in o (lambda _ (apply storedata-store_command (cons o args))) `(,(.storedata o) store_command)))
                 #:store_result (lambda args (call-in o (lambda _ (apply storedata-store_result (cons o args))) `(,(.storedata o) store_result)))
                 #:request (lambda args (call-in o (lambda _ (apply storedata-request (cons o args))) `(,(.storedata o) request)))
                 #:get_extend_trace_data (lambda args (call-in o (lambda _ (apply storedata-get_extend_trace_data (cons o args))) `(,(.storedata o) get_extend_trace_data)))
                 #:get_selection (lambda args (call-in o (lambda _ (apply storedata-get_selection (cons o args))) `(,(.storedata o) get_selection)))
                 #:get_location (lambda args (call-in o (lambda _ (apply storedata-get_location (cons o args))) `(,(.storedata o) get_location)))
                 #:get_lts_command_selection (lambda args (call-in o (lambda _ (apply storedata-get_lts_command_selection (cons o args))) `(,(.storedata o) get_lts_command_selection)))
                 #:get_command_backintrace (lambda args (call-in o (lambda _ (apply storedata-get_command_backintrace (cons o args))) `(,(.storedata o) get_command_backintrace)))
                 #:get_command_simulate_backintrace (lambda args (call-in o (lambda _ (apply storedata-get_command_simulate_backintrace (cons o args))) `(,(.storedata o) get_command_simulate_backintrace)))
                 #:get_command_replaytrace (lambda args (call-in o (lambda _ (apply storedata-get_command_replaytrace (cons o args))) `(,(.storedata o) get_command_replaytrace)))
                 #:get_command_simulate_replaytrace (lambda args (call-in o (lambda _ (apply storedata-get_command_simulate_replaytrace (cons o args))) `(,(.storedata o) get_command_simulate_replaytrace)))
                 #:get_command_blackbox_step (lambda args (call-in o (lambda _ (apply storedata-get_command_blackbox_step (cons o args))) `(,(.storedata o) get_command_blackbox_step)))
                 #:get_command_blackbox_state (lambda args (call-in o (lambda _ (apply storedata-get_command_blackbox_state (cons o args))) `(,(.storedata o) get_command_blackbox_state)))
                 #:valid_p (lambda args (call-in o (lambda _ (apply storedata-valid_p (cons o args))) `(,(.storedata o) valid_p)))
                 #:req (lambda args (call-in o (lambda _ (apply storedata-req (cons o args))) `(,(.storedata o) req)))
                 #:reduce (lambda args (call-in o (lambda _ (apply storedata-reduce (cons o args))) `(,(.storedata o) reduce)))
                 #:reset (lambda args (call-in o (lambda _ (apply storedata-reset (cons o args))) `(,(.storedata o) reset))))
          #:out (make <daemon:Idata.out>))))


(define (json-get json key) ;; REMOVEME
  (let ((key (if (string? key) (string->symbol key) key)))
    (or (assq-ref json key)
        (assoc-ref json (symbol->string key)))))

(define-method (storedata-store_command (o <daemon:Data>) command)
  (let ((log (dzn:get (.locator o) <procedure> 'log))
        (log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Data: store_command: command: ~s\n" command)
    (let* ((command-data (libenvelope:data command))
           (type (assoc-ref command-data "name")))
      (log-debug "Data: store_command: type: ~s\n" type)
      (log "command: ~a\n" type)
      (case (string->symbol type)
        ((bye) (set! (.hello-command o) command-data))
        ((hello) (set! (.hello-command o) command-data))
        ((parse) (set! (.parse-command o) command-data))
        ((simulate) (set! (.simulate-command o) command-data))
        ((state) (set! (.state-command o) command-data))
        ((verify) (set! (.verify-command o) command-data))
        (else (log-error "Data: store_command: no such command: ~s\n" type))))))

(define-method (storedata-store_result (o <daemon:Data>) data out-stored)
  (define (symbol< a b)
    (string< (symbol->string a) (symbol->string b)))
  (let ((log (dzn:get (.locator o) <procedure> 'log))
        (log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Data: store_result: data: ~s\n" data)
    (let* ((label (libenvelope:label data))
           (command-data (libenvelope:data data))
           (status (json-get command-data 'status))
           (foo (log-debug "Data: store_result: status: ~s\n" status))
           (out (json-get command-data 'stdout))
           (out (and out (not (string-null? out)) out))
           (err (json-get command-data 'stderr))
           (err (and err (not (string-null? err)) err))
           (json (if (not (eq? status 0)) (list "err" err) out) ;;(and out (libtransport:unserialize out))
                 )
           (json (if (eq? label 'hello)
                     (let* ((connections (.request-data o))
                            (connections (sort connections symbol<))
                            (result `((stdout . ,json)
                                      (connections ,@connections))))
                       (set! (.request-data o) (list))
                       result)
                     json)))
      (log "result: ~a\n" label)
      (log-debug "Data: store_result: json: ~s\n" json)
      (when err
        (log-debug "Data: store_result: error: ~a\n" err))
      (case label
        ((bye) (set! (.hello-data o) json))
        ((hello) (set! (.hello-data o) json))
        ((parse) (set! (.parse-data o) json))
        ((verify) (set! (.verify-data o) json))
        ((simulate) (set! (.simulate-data o) json))
        ((state) (set! (.state-data o) json))
        ((backintrace) (set! (.backintrace-data o) (or (and (pair? json) json) out "")))
        ((replaytrace) (set! (.replaytrace-data o) (or (and (pair? json) json) out "")))
        (else (log-error "Data: store_result: no such label: ~s\n" label)))
      (let ((result (libenvelope:wrap label json)))
        (set-cdr! out-stored result)))))

(define-method (storedata-request (o <daemon:Data>) notification out-data)
  (let ((log (dzn:get (.locator o) <procedure> 'log))
        (log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Data.in.request notification: ~s\n" notification)
    (let* ((label (assoc-ref notification "label"))
           (label (if (string? label) (string->symbol label) label))
           (data (case label
                   ((system) (.parse-data o))
                   ((verify) (.verify-data o))
                   ((state) (.state-data o))
                   ((trace) (.simulate-data o))
                   (else (log-error "Data.in.request: no such label: ~s\n" label))))
           (result (libenvelope:wrap label data))
           (origin (assoc-ref notification "origin"))
           (origin (and=> (memq origin '(state system trace)) car)))
      (when origin
        (set! (.request-data o) (cons origin (.request-data o))))
      (set-cdr! out-data result))))

(define-method (storedata-get_extend_trace_data (o <daemon:Data>) selection data)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_extend_trace_data ~s\n" selection)))

(define-method (storedata-get_selection (o <daemon:Data>) out-selection)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_selection\n")
    (set-cdr! out-selection (.selection-data o))
    (.selection-data o)))

(define (seqdiag:seqdiag-selection->location seqdiag-selection working-directory)
  (let* ((file-name (assoc-ref seqdiag-selection "file"))
         (line (assoc-ref seqdiag-selection "line"))
         (column (assoc-ref seqdiag-selection "column"))
         (end (assoc-ref seqdiag-selection "end"))
         (end-line (and end (assoc-ref end "line")))
         (end-column (and end (assoc-ref end "column"))))
    (and file-name
         `(("working-directory" . ,working-directory)
           ("file-name" . ,file-name)
           ("line" . ,line)
           ("column" . ,column)
           ,@(if end-line `(("end-line" . ,end-line)) '())
           ,@(if end-column `(("end-column" . ,end-column)) '())))))

(define-method (seqdiag:index->location (o <daemon:Data>) index)
  (let* ((data (libtransport:unserialize (.simulate-data o)))
         (working-directory (assoc-ref data "working-directory"))
         (seqdiag-location (list-ref (assoc-ref data "seqdiag") index))
         (seqdiag-selections (assoc-ref seqdiag-location "selection")))
    (and (pair? seqdiag-selections)
         (seqdiag:seqdiag-selection->location (car seqdiag-selections) working-directory))))

(define-method (storedata-get_location (o <daemon:Data>) selection out-location)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_location ~s\n" selection)
    (match selection
      (((("index" . (? number? index))))
       (let ((location (seqdiag:index->location o index)))
         (and location
              (set-cdr! out-location location))))
      ((("selection" . (location)))
       (set-cdr! out-location location))
      (_
       (set-cdr! out-location '())))))

(define-method (storedata-get_lts_command_selection (o <daemon:Data>) selection command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_lts_command_selection ~s\n" selection)))

(define (make-command name model trace input)
  (let ((argv (append (if model `("-m" ,model) `()) `("-T" ,trace "-"))))
    `(("command" . (("name" . ,name)
                    ("model" . ,model)
                    ("trace" . ,trace)
                    ("input" . ,input)
                    ("argv" . (,name ,@argv)))))))

(define-method (storedata-get_command_backintrace (o <daemon:Data>) out-command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_command_backintrace\n")
    (let* ((simulate (.simulate-command o))
           (command (assoc-ref simulate "command"))
           (command (or command simulate)) ;; FIXME
           (model (assoc-ref command "model"))
           (trace (assoc-ref command "trace"))
           (input (assoc-ref command "input"))
           (command (make-command "backintrace" model trace (disable-imports input))))
      (log-debug "storedata-get_command_backintrace backintrace= ~s\n" command)
      (set-cdr! out-command command))))

(define drop-right (@ (srfi srfi-1) drop-right))

(define (split trace) ;; trace is either separated by ',' or '\n'
  (let* ((trace (string-trim-both trace))
         (trace (if (or (equal? trace "\n") (equal? trace ",")) "" trace))
         (trace (if (string-index trace #\,) (string-split trace #\,) (string-split trace #\newline))))
    (cond ((null? trace) trace)
          ((equal? (last trace) "") (drop-right trace 1))
          (else trace))))

(define (join trace)
    (if (null? trace) "," (string-join (append trace (list "")) ",")))

(define (extend trace element)
  (let* ((trace (split trace))
         (trace (append trace (list element))))
    (join trace)))

(define (drop-last trace)
  (let* ((trace (split trace))
         (trace (if (null? trace) trace (drop-right trace 1))))
    (join trace)))

(define-method (storedata-get_command_replaytrace (o <daemon:Data>) eligible out-command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_command_replaytrace eligible=~s\n" eligible)
    (let* ((simulate (.simulate-command o))
           (command (assoc-ref simulate "command"))
           (command (or command simulate)) ;; FIXME
           (model (assoc-ref command "model"))
           (trace (assoc-ref command "trace"))
           (input (assoc-ref command "input"))
           (trace (extend trace eligible))
           (command (make-command "replaytrace" model trace (disable-imports input))))
      (log-debug "storedata-get_command_replaytrace replaytrace= ~s\n" command)
      (set-cdr! out-command command))))

(define-method (storedata-get_command_simulate_backintrace (o <daemon:Data>) out-command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_command_simulate_backintrace\n")
    (let* ((simulate (.simulate-command o))
           (command (assoc-ref simulate "command"))
           (command (or command simulate))
           (model (assoc-ref command "model"))
           (input (assoc-ref command "input"))
           (trace (.backintrace-data o))
           (trace (join (split trace)))
           (command (make-command "simulate" model trace (disable-imports input))))
      (log-debug "storedata-get_command_simulate_backintrace command= ~s\n" command)
      (set! (.simulate-command o) command)
      (set-cdr! out-command command))))

(define-method (storedata-get_command_simulate_replaytrace (o <daemon:Data>) out-command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_command_simulate_replaytrace\n")
    (let* ((simulate (.simulate-command o))
           (command (assoc-ref simulate "command"))
           (command (or command simulate)) ;; FIXME
           (model (assoc-ref command "model"))
           (input (assoc-ref command "input"))
           (trace (.replaytrace-data o))
           (trace (join (split trace)))
           (command (make-command "simulate" model trace (disable-imports input))))
      (log-debug "storedata-get_command_simulate_replaytrace command= ~s\n" command)
      (set! (.simulate-command o) command)
      (set-cdr! out-command command))))

(define-method (storedata-get_command_blackbox_step (o <daemon:Data>) instance command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_command_blackbox_step ~s\n" instance)))

(define-method (storedata-get_command_blackbox_state (o <daemon:Data>) command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-get_command_blackbox_state ~s\n" command)))

(define-method (storedata-valid_p (o <daemon:Data>) data)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-valid_p ~s\n" data)
    (and data
         (not (equal? data "")))))

(define-method (storedata-req (o <daemon:Data>) data selection)
  (format (current-error-port) "storedata-req\n")
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug))
        (decoupler (actor-id (dzn:get (.locator o) <decoupler>))))
    (log-debug "storedata-req ~s ~s\n" data selection)
    (<- decoupler 'decouple
        (lambda _
          (action o .storedata .out .ack data selection)))))

(define-method (storedata-reduce (o <daemon:Data>) blockindex data selection)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-reduce ~s ~s\n" blockindex data)))

(define-method (storedata-reset (o <daemon:Data>))
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "storedata-reset")))
