;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2020 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands simulate)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 rdelim)
  #:use-module (oop goops)
  #:use-module (dzn ast)
  #:use-module (dzn goops)
  #:use-module (dzn parse)
  #:use-module (ide shell-util)
  #:use-module (ide commander)
  #:use-module (ide commands daemon)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide util)
  #:use-module (ide command-line)
  #:export (sequence->error
            handle-simulate-result))

(define (parse-opts args)
  (let* ((option-spec
          '((help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))
            (server (value #t))
            (trace (single-char #\t) (value #t))
            (trace-file (single-char #\T) (value #t))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide simulate [OPTION]... [FILE]...
Run simulate and make a trace view available.

  -h, --help             display this help and exit
  -I, --import=DIR+      add directory DIR to import path
  -m, --model=MODEL      explore model MODEL
  -t, --trace=TRACE      use named event trace TRACE
  -T, --trace-file=FILE  read named event trace from FILE

View:
  ~a
" (get-url "trace"))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (sequence->error sequence)
  (let* ((ecneuqes (reverse sequence))
         (error (car ecneuqes))
         (message (assoc-ref error "message"))
         (location (find (lambda (e)
                           (and=> (assoc-ref (cdr e) "selection")
                                  (compose (cute assoc-ref <> "file") car)))
                         ecneuqes))
         (location (and=> (assoc-ref location "selection") car))
         (dzn-file-name (assoc-ref location "file"))
         (line (assoc-ref location "line"))
         (column (assoc-ref location "column"))
         (index (assoc-ref location "index")))
    (and message (format #f "~a:~a:~a:i~a: ~a\n" dzn-file-name line column index message))))

(define (seqdiag:get-model sexp)
  (let* ((inits (filter (compose (cut equal? <> "Initialize") (cut assoc-ref <> "kind"))
                        sexp))
         (model-inits (if (= (length inits) 1) inits
                         (filter (compose (cut equal? <> "component") (cut assoc-ref <> "role"))
                                 inits))))
    (assoc-ref (car model-inits) "instance")))

(define (seqdiag:sequence->trail sequence model)
  (let* ((steps (filter
                 (conjoin (compose (cut equal? <> model) (cut assoc-ref <> "instance"))
                          (disjoin (cut assoc-ref <> "synchronization")
                                   (conjoin (compose (cut equal? <> "Error")
                                                     (cut assoc-ref <> "kind"))
                                            (compose (cut equal? <> "illegal")
                                                     (cut assoc-ref <> "message")))))
                        sequence)))
    (map (disjoin (cut assoc-ref <> "synchronization")
                  (cut assoc-ref <> "message"))
         steps)))

(define* (handle-simulate-result message #:key debug? verbose?)
  (let* ((simulate (json-string->alist-scm (simulate-data message)))
         (seqdiag (and simulate (assoc-ref simulate "seqdiag")))
         (eligible (or (and simulate (assoc-ref simulate "lseligible")) ""))
         (eligible (string-trim-right eligible))
         (eligible (string-split eligible #\newline))
         (model (seqdiag:get-model seqdiag))
         (trail (seqdiag:sequence->trail seqdiag model))
         (error (and seqdiag (sequence->error seqdiag))))
    (when debug?
      (format (current-error-port) "simulate: ~a\n" simulate)
      (format (current-error-port) "seqdiag: ~a\n" seqdiag)
      (format (current-error-port) "eligible: ~a\n" eligible))
    (when (or debug? verbose?)
      (format #t "trail: ~a\n" (string-join trail ","))
      (format #t "eligible: ~a\n" (string-join eligible ",")))
    error))

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (files (option-ref options '() #f))
         (file-name (car files))
         (imports (multi-opt options 'import))
         (input (piped-file->stream file-name #:debug? debug? #:imports imports))
         (model (or (option-ref options 'model #f)
                    (let ((ast (false-if-exception (string->ast input))))
                      (if (not ast) (exit EXIT_OTHER_FAILURE)
                          (let* ((models (filter (negate (is? <system>))
                                                 (ast:model* ast)))
                                 (ast (clone ast #:elements models)))
                            (and=> (ast:get-model ast) ast:dotted-name))))))
         (result EXIT_SUCCESS))

    (define (simulate-handler message)
      (log-debug  "received: ~s\n" message)
      (let ((message (parse-message message)))
        (cond ((simulate? message)
               (let ((error (handle-simulate-result message #:debug? debug? #:verbose? verbose?)))
                 (when error
                   (set! result EXIT_FAILURE)
                   (format (current-error-port) "~a" error))
                 (when (isatty? (current-error-port))
                   (format (current-error-port) "See result: ~a\n" (get-url "trace")))))
              ((error? message)
               (display (error-stderr message) (current-error-port))
               (set! result EXIT_FAILURE))
              (else
               (format (current-error-port)
                       "ide simulate: error: received: ~s\n"
                       message)
               (set! result EXIT_FAILURE)))
        message))

    (unless model
      (format (current-error-port) "no model to simulate\n")
      (exit EXIT_OTHER_FAILURE))

    (daemon:start #:debug? debug? #:port %ide-port)
    (let* ((trace-file (option-ref options 'trace-file #f))
           (trace (if trace-file (with-input-from-file trace-file read-string) ""))
           (trace (option-ref options 'trace trace))
           (trace (string-map (lambda (c) (if (eq? c #\newline) #\, c)) trace))
           (trace (if (mingw?) (string-append trace ",") trace))
           (command (make-command "simulate" `("simulate"
                                               ,@(if model `("-m" ,model) '())
                                               "-T" ,trace
                                               "-")
                                  #:input input
                                  #:model model
                                  #:trace trace))
           (commander (make-commander #:debug? debug? #:verbose? verbose?))
           (result-handler
            (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                     simulate-handler))
           (result
            (+ (run-command commander command
                            #:result-handler result-handler
                            #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
               result)))
      (exit result))))
