;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2021 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;; Copyright © 2021 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;; Commentary:
;;
;; Dezyne Language Server for the Language Server Protocol
;; https://microsoft.github.io/language-server-protocol

;;; Code:

(define-module (ide lsp)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-9 gnu)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)
  #:use-module (ice-9 binary-ports)
  #:use-module (ice-9 iconv)
  #:use-module (ice-9 match)
  #:use-module (web http)
  #:use-module (web uri)
  #:use-module (web socket client)

  #:use-module (dzn parse)
  #:use-module (dzn parse peg)
  #:use-module (dzn parse complete)
  #:use-module (dzn parse lookup)
  #:use-module (dzn parse tree)
  #:use-module (dzn parse util)
  #:use-module (ide shell-util)

  #:use-module (ide config)
  #:use-module (ide json)
  #:use-module (ide util)
  #:use-module (ide commands daemon)
  #:export (lsp:stdio))

;;;
;;; RPC messages.
;;;

(define (rpc:get-id jsonrpc)
  (assoc-ref jsonrpc "id"))

(define (rpc:get-jsonrpc jsonrpc)
  (assoc-ref jsonrpc "jsonrpc"))

(define (rpc:get-method jsonrpc)
  (assoc-ref jsonrpc "method"))

(define (rpc:get-params jsonrpc)
  (assoc-ref jsonrpc "params"))

(define (rpc:get-root-uri-string jsonrpc)
  (assoc-ref jsonrpc "rootUri"))

(define (rpc:get-root-uri-string jsonrpc)
  (and=> (assoc-ref jsonrpc "params")
         rpc:params:get-root-uri-string))

(define (rpc:params:get-content-changes params)
  (and=> (assoc-ref params "contentChanges")
         (compose (cute assoc-ref <> "text") car)))

(define (rpc:params:get-root-uri-string params)
  (assoc-ref params "rootUri"))

(define (rpc:params:get-uri-string params)
  (and=> (assoc-ref params "textDocument")
         (cute assoc-ref <>  "uri")))

(define (rpc:params:get-position params)
  (assoc-ref params "position"))

(define (rcp:make-message sexp)
  "Return an JSON RPC message with SEXP."
  (let* ((body    (scm->json-string sexp))
         (text    (string-append body "\r\n"))
         (size    (string-length text)))
    (format #f "Content-Length: ~a\r\n\r\n~a" size text)))

(define* (rpc:read-message #:optional (input-port (current-input-port)))
  (let* ((headers (read-headers input-port))
         (size    (assoc-ref headers 'content-length)))
    (when size
      (let* ((bv (get-bytevector-n input-port size))
             (json-data (bytevector->string bv "utf-8")))
        (json-string->alist-scm json-data)))))


;;;
;;; Parse errors.   XXX Split to util/peg;
;;;

;;; XXX TODO use error-/error: prefix?
;;; Use one <error> type?
(define-immutable-record-type <missing>
  (make-missing location expectations)
  missing?
  (location     missing-location)
  (expectations missing-expectations))

(define-immutable-record-type <skipped>
  (make-skipped location skipped location-end)
  skipped?
  (location     skipped-location)
  (skipped      skipped-skipped)
  (location-end skipped-location-end))

(define-immutable-record-type <verify>
  (make-verify location verify location-end)
  verify?
  (location     verify-location)
  (verify       verify-verify)
  (location-end verify-location-end))

(define (error:location error)
  (match error
    ((or ($ <missing> location)
         ($ <skipped> location)
         ($ <verify>  location))
     location)))

(define (error:location-end error)
  (match error
    (($ <missing> location)
     location)
    (($ <skipped> location skipped location-end)
     location-end)
    (($ <verify> location verify location-end)
     location-end)))

(define* (error:message error #:key without-location?)
  (match error
    (($ <missing> location missing)
     (let* ((type (error:type error))
            (location (location->string location))
            (message (format #f "missing: ~a" missing)))
       (if without-location? message
           (format #f "~a:~a: ~a" location type message))))
    (($ <skipped> location skipped)
     (let* ((type (error:type error))
            (location (location->string location))
            (message (format #f "skipped: ~a" skipped)))
       (if without-location? message
           ;; XXX TODO: location range
           (format #f "~a:~a: ~a" location type message))))
    (($ <verify> location verify)
     (let* ((type (error:type error))
            (location (location->string location))
            (message verify))
       (if without-location? message
           ;; XXX TODO: location range
           (format #f "~a:~a: ~a" location type message))))))

(define (error:type error)
  (match error
    ((or ($ <missing>) ($ <skipped>) ($ <verify>))
     'error)))

(define (error:source error)
  (match error
    ((or ($ <missing>) ($ <skipped>))
     "dzn parse")
    (($ <verify>)
     "dzn verify")))

(define (peg:file-name+offset+error->error file-name offset error text)
  (let ((location (file-offset->location file-name offset text)))
    (match error
      (('missing missing)
       (make-missing location missing))
      (('skipped skipped)
       (let* ((offset-end   (+ offset (string-length skipped)))
              (location-end (file-offset->location file-name offset-end text)))
         (make-skipped location skipped location-end))))))

(define* (peg:collect-errors text #:key (file-name "-"))
  (let ((errors '()))
    (define (add-error! offset string error)
      (let ((error (peg:file-name+offset+error->error file-name offset error text)))
        (set! errors (cons error errors))))
    (parameterize ((%peg:fall-back? #t)
                   (%peg:locations? #t)
                   (%peg:skip? peg:skip-parse)
                   (%peg:error add-error!))
      (peg:parse text))
    errors))


;;;
;;; LSP library.
;;;

(define-immutable-record-type <lsp>
  (make-lsp- root-uri-string imports content-alist parse-alist)
  lsp?
  (root-uri-string lsp-root-uri-string) ;string
  (imports         lsp-imports)         ;list of strings
  (content-alist   lsp-content-alist)   ;alist ((file-name . content) ...)
  (parse-alist     lsp-parse-alist))    ;alist ((file-name . parse-tree) ...)

(define* (make-lsp uri-string #:key (imports '()) (content-alist '())
                   (parse-alist '()))
  (make-lsp- uri-string imports content-alist parse-alist))

(define* (lsp:resolve-file lsp file-name #:key (imports '()))
  (let ((imports (append imports (lsp-imports lsp))))
    (and=> (search-path imports file-name) canonicalize-path)))

(define (lsp:get-text lsp file-name)
  (let ((file-name (lsp:resolve-file lsp file-name)))
    (or (assoc-ref (lsp-content-alist lsp) file-name) "")))

(define (lsp:get-parse-tree lsp file-name)
  (let* ((parse-alist (lsp-parse-alist lsp))
         (file-name (lsp:resolve-file lsp file-name)))
    (or (assoc-ref parse-alist file-name)
        (lsp:get-parse-tree (lsp:set-file lsp file-name) file-name))))

(define (lsp:file-name+position->location file-name position)
  (let ((line   (assoc-ref position "line"))
        (column (assoc-ref position "character")))
    (make-location file-name (1+ line) column)))

(define (lsp:file-name+offset->location lsp file-name offset)
  (let ((text (lsp:get-text lsp file-name)))
    (file-offset->location file-name offset text)))

(define* (lsp:uri-string->file-name uri-string)
  (let* ((uri       (string->uri uri-string))
         (host      (uri-host uri))
         (path      (uri-path uri))
         (parts     (split-and-decode-uri-path path))
         (file-name (string-join parts "/"))
         (file-name (if (or (string-prefix? "/" file-name)
                            ;; XXX for tests: file://localhost/some/file/name
                            (equal? host "localhost")
                            (mingw?)) file-name
                            (string-append "/" file-name))))
    (canonicalize-path file-name)))

(define* (lsp:file-name->uri-string lsp file-name)
  (let* ((file-name (lsp:resolve-file lsp file-name))
         (parts     (string-split file-name #\/)))
    (string-append "file://" (if (mingw?) "/" "")
                   (encode-and-join-uri-path parts))))

(define* (lsp:location->result lsp location #:key (location-end location))
  (let* ((file-name  (location-file location))
         (uri-string (lsp:file-name->uri-string lsp file-name))
         (line       (1- (location-line location)))
         (column     (location-column location))
         (end-line   (1- (location-line location-end)))
         (end-column (location-column location-end)))
    `((uri   . ,uri-string)
      (range . ((start . ((line      . ,line)
                          (character . ,column)))
                (end   . ((line      . ,end-line)
                          (character . ,end-column))))))))

(define (lsp:error->diagnostic lsp error)
  (define (type->severity type)
    (case type
      ((error) 1)
      ((warn) 2)
      ((info) 3)
      ((hint) 4)))
  (let* ((message  (error:message error #:without-location? #t))
         (location (error:location error))
         (end      (error:location-end error))
         (type     (error:type error))
         (severity (type->severity type)))
    `(,@(lsp:location->result lsp location #:location-end end)
      (severity . ,severity)
      (source   . ,(error:source error))
      (message  . ,message))))

(define (lsp:errors->diagnostics lsp file-name errors)
  (let* ((diagnostics (map (cute lsp:error->diagnostic lsp <>) errors))
         (diagnostics (delete-duplicates diagnostics))
         (uri-string  (lsp:file-name->uri-string lsp file-name)))
    `((method . "textDocument/publishDiagnostics")
      (params . ((uri         . ,uri-string)
                 (diagnostics . ,diagnostics))))))

(define (lsp:completion->result completion)
  `(("label" . ,completion) ("kind" . 11)))

(define (lsp:make-message data)
  "Return an LSP message with DATA."
  (let ((message `((jsonrpc . "2.0")
                   ,@data)))
    (rcp:make-message message)))

(define (lsp:make-reply-message request result)
  "Return an LSP relpy message for REQUEST with RESULT."
  (let* ((id      (rpc:get-id request))
         (jsonrpc (rpc:get-jsonrpc request))
         (message `((jsonrpc . ,jsonrpc)
                    (id      . ,id)
                    (result  . ,result))))
    (rcp:make-message message)))

(define (lsp:context lsp location)
  (let* ((file-name (location-file location))
         (text      (lsp:get-text lsp file-name))
         (offset    (location->offset location text))
         (tree      (lsp:get-parse-tree lsp file-name)))
    (complete:context tree offset)))

(define (lsp:completions lsp location)
  (let* ((context   (lsp:context lsp location))
         (token     (.tree context))
         (file-name (location-file location))
         (text      (lsp:get-text lsp file-name))
         (offset    (location->offset location text)))
    (complete token context offset
              #:file-name->parse-tree
              (cute lsp:get-parse-tree lsp <>)
              #:resolve-file
              (lambda* (file-name #:key (imports '()))
                (lsp:resolve-file lsp file-name #:imports imports)))))

(define (lsp:lookup-definition lsp location)
  (let* ((context     (lsp:context lsp location))
         (token       (.tree context))
         (file-name   (location-file location)))
    (lookup-location token context
                     #:file-name file-name
                     #:file-name->text
                     (cute lsp:get-text lsp <>)
                     #:file-name->parse-tree
                     (cute lsp:get-parse-tree lsp <>)
                     #:resolve-file
                     (lambda* (file-name #:key (imports '()))
                       (lsp:resolve-file lsp file-name #:imports imports)))))

(define (lsp:update-parse lsp file-name)
  (let* ((content-alist (lsp-content-alist lsp))
         (parse-alist   (lsp-parse-alist lsp))
         (parse-alist   (alist-delete file-name parse-alist))
         (content       (lsp:get-text lsp file-name))
         (parse-tree    (catch #t
                          (lambda _
                            (with-error-to-port (%make-void-port "w")
                              (lambda _
                                (parameterize ((%peg:fall-back? #t))
                                  (string->parse-tree
                                   content
                                   #:file-name file-name
                                   #:content-alist content-alist)))))
                          (const '())))
         (parse-alist   (acons file-name parse-tree parse-alist)))
    (set-field lsp (lsp-parse-alist) parse-alist)))

(define (lsp:set-file lsp file-name)
  (let* ((content-alist (lsp-content-alist lsp))
         (content-alist (alist-delete file-name content-alist))
         (imports       (lsp-imports lsp))
         (file-name     (lsp:resolve-file lsp file-name))
         (dir           (dirname file-name))
         (imports       (delete-duplicates (cons dir imports)))
         (lsp           (set-field lsp (lsp-imports) imports))
         (content-alist (catch #t
                          (lambda _
                            (with-error-to-port (%make-void-port "w")
                              (lambda _
                                (file+import-content-alist
                                 file-name
                                 #:imports imports
                                 #:content-alist content-alist))))
                          (const '())))
         (lsp           (set-field lsp (lsp-content-alist) content-alist)))
    (lsp:update-parse lsp file-name)))

(define (lsp:set-text lsp file-name text)
  (let* ((content-alist (lsp-content-alist lsp))
         (content-alist (alist-delete file-name content-alist))
         (content-alist (acons file-name text content-alist))
         (parse-alist   (lsp-parse-alist lsp))
         (parse-alist   (alist-delete file-name parse-alist))
         (lsp           (set-field lsp (lsp-content-alist) content-alist)))
    (lsp:update-parse lsp file-name)))

(define* (lsp:params->file-name params)
  "Return file-name from PARAMS."
  (let* ((uri-string (rpc:params:get-uri-string params))
         (position   (rpc:params:get-position params)))
    (lsp:uri-string->file-name uri-string)))

(define* (lsp:params->location params)
  "Return location object from PARAMS."
  (let ((file-name  (lsp:params->file-name params))
        (position   (rpc:params:get-position params)))
    (lsp:file-name+position->location file-name position)))

(define* (lsp:word-at-point lsp point)
  "Return word at POINT."
  (let* ((file-name  (location-file point))
         (text       (lsp:get-text lsp file-name))
         (offset     (location->offset point text))
         (offset     (if (zero? offset) 0 (1- offset)))
         (count      (string-length text))
         (word-set   (char-set-adjoin char-set:letter+digit #\_ #\.))
         (char       (string-ref text offset)))
    (and (char-set-contains? word-set char)
         (let ((start (let loop ((pos offset))
                        (if (zero? pos) pos
                            (let* ((next (1- pos))
                                   (char (string-ref text next)))
                              (if (not (char-set-contains? word-set char)) pos
                                  (loop next))))))
               (end (let loop ((pos offset))
                      (if (= pos (1- count)) pos
                          (let* ((next (1+ pos))
                                 (char (string-ref text next)))
                            (if (not (char-set-contains? word-set char)) pos
                                (loop next)))))))
           (substring text start (1+ end))))))

(define (lsp:init-port! port)
  (setvbuf port 'none)
  (set-port-encoding! port "ISO-8859-1")
  (when (mingw?)
    (set-mode port O_BINARY)))


;;;
;;; Message handlers.
;;;

(define* (lsp:check-syntax lsp params #:key debug?)
  (let* ((file-name   (lsp:params->file-name params))
         (text        (lsp:get-text lsp file-name))
         (errors      (peg:collect-errors text #:file-name file-name))
         (diagnostics (lsp:errors->diagnostics lsp file-name errors)))
    (values lsp diagnostics)))

(define* (lsp:complete lsp params #:key debug?)
  (let* ((point    (lsp:params->location params))
         (complete (or (false-if-exception (lsp:completions lsp point)) '()))
         (word     (false-if-exception (lsp:word-at-point lsp point)))
         (complete (if (not word) complete
                       (filter (cute string-prefix? word <>) complete)))
         ;; XXX Apparently LSP (or the implementations of emacs-lsp-mode
         ;; and VSCodium derivatives) assign semantics to the `.'
         ;; character.  Completion results must not include words that
         ;; include a dot.
         (prefix   (and word (string-rindex word #\.)))
         (complete (if (not prefix) complete
                           (map (cute substring <> (1+ prefix))
                                complete)))
         (result   (map lsp:completion->result complete)))
    (when debug?
      (log-debug "lsp:goto-def result=~a\n" result))
    (values lsp result)))

(define* (lsp:goto-def lsp params #:key debug?)
  (let* ((point (lsp:params->location params))
         (def   (false-if-exception (lsp:lookup-definition lsp point)))
         (result (if (not def) '()
                     (lsp:location->result lsp def))))
    (when debug?
      (log-debug "lsp:goto-def result=~a\n" result))
    (values lsp result)))

(define* (lsp:initialize lsp params #:key debug?)
  (let ((capabilities
         `((completionProvider        . ((resolveProvider . #t)))
           (definitionProvider        . #t)
           (documentHighlightProvider . #f)
           (documentSymbolProvider    . #f)
           (hoverProvider             . #f)
           (referencesProvider        . #f)
           (textDocumentSync          . 1)
           (workspaceSymbolProvider   . #f))))
    (values lsp `((capabilities . ,capabilities)))))

(define* (lsp:open-file lsp params #:key debug?)
  (let* ((file-name (lsp:params->file-name params))
         (lsp       (lsp:set-file lsp file-name)))
    (when debug?
      (log-debug "lsp:open-file ~a\n" file-name))
    (values lsp #f)))

(define* (lsp:update-text lsp params #:key debug?)
  (let* ((file-name  (lsp:params->file-name params))
         (text       (rpc:params:get-content-changes params))
         (lsp        (lsp:set-text lsp file-name text)))
    (when debug?
      (log-debug "lsp:update-text ~a\n" file-name))
    (values lsp #f)))


;;;
;;; Message dispatching.
;;;

(define %lsp:handler-alist
  `(("initialize"              . ,lsp:initialize)
    ("textDocument/completion" . ,lsp:complete)
    ("textDocument/definition" . ,lsp:goto-def)
    ("textDocument/didChange"  . ,(lambda (lsp . args)
                                    (let ((lsp (apply lsp:update-text
                                                      lsp args)))
                                      (apply lsp:check-syntax lsp args))))
    ("textDocument/didOpen"    . ,lsp:open-file)
    ("textDocument/didSave"    . ,lsp:check-syntax)))

(define* (lsp:dispatch-message lsp message #:key debug?)
  "Parse MESSAGE for LSP method and dispatch it.  Return two values:
LSP, and a response or #f."
  (let* ((id      (rpc:get-id message))
         (method  (rpc:get-method message))
         (params  (rpc:get-params message))
         (handler (assoc-ref %lsp:handler-alist method)))
    (when debug?
      (log-debug "message[~a]: ~a\n" id message)
      (log-debug "method: ~a\n" method)
      (log-debug "params: ~a\n" params))
    (if (not handler) (values lsp #f)
        (let ((lsp result (handler lsp params #:debug? debug?)))
          (when debug?
            (log-debug "reply[~a]: ~a\n" id result))
          (match result
            (#f (values lsp #f))
            ((('method . method) rest ...)
             (values lsp (lsp:make-message result)))
            (_
             (values lsp (lsp:make-reply-message message result))))))))

(define* (lsp:handle-message lsp message #:key debug? (imports '()))
  "Handle MESSAGE in LSP context, (re)creating LSP context using IMPORTS,
and return two values: LSP, and a response or #f."
  (let* ((root-uri-string (rpc:get-root-uri-string message))
         (lsp (if (or (not root-uri-string)
                      (equal? root-uri-string
                              (lsp-root-uri-string lsp)))
                  lsp
                  (make-lsp root-uri-string #:imports imports))))
    (lsp:dispatch-message lsp message #:debug? debug?)))


;;;
;;; Daemon messages for editor.
;;;

(define* (daemon:read-message websocket)
  "Read ide daemon message from WEBSOCKET."
  (let ((json-data (websocket-receive websocket)))
    (json-string->alist-scm json-data)))

(define (daemon:go-to->error go-to)
  (let* ((working-directory (assoc-ref go-to "working-directory"))
         (file-name         (assoc-ref go-to "file-name"))
         (file-name         (if (or (absolute-file-name? file-name) (not working-directory)) file-name
                                (string-append working-directory "/" file-name)))
         (file-name         (canonicalize-path file-name))
         (line              (assoc-ref go-to "line"))
         (column            (assoc-ref go-to "column"))
         (location          (make-location file-name line column))
         (end-line          (assoc-ref go-to "end-line"))
         (end-column        (assoc-ref go-to "end-column"))
         (location-end      (make-location file-name end-line end-column)))
    (make-verify location "verify" location-end)))

(define (lsp:daemon-handle-go-to lsp go-to)
  (let* ((error     (daemon:go-to->error go-to))
         (file-name (location-file (error:location error))))
    (lsp:errors->diagnostics lsp file-name (list error))))

(define* (lsp:handle-daemon-message lsp message #:key debug? (imports '()))
  "Handle MESSAGE in LSP context and return two values: LSP, and a
response or #f."
  (let* ((response (match message
                     ((("go_to" . go-to))
                      (lsp:daemon-handle-go-to lsp go-to))
                     (_ #f)))
         (message (and response (lsp:make-message response))))
    (values lsp message)))

;;;
;;; Entry point.
;;;

(define* (lsp:stdio #:key
                    (input-port (current-input-port))
                    (output-port (current-output-port))
                    (editor-port %editor-port)
                    debug?
                    file-name
                    (imports '()))
  "Read Language Server message from INPUT-PORT and write response to
OUTPUT-PORT until INPUT-PORT is drained."
  (lsp:init-port! input-port)
  (lsp:init-port! output-port)
  (let* ((daemon-listening? (and editor-port
                                 (daemon:listening? editor-port)))
         (editor-url        (format #f "ws://127.0.0.1:~a" editor-port))
         (daemon-websocket  (and daemon-listening?
                                 (make-websocket editor-url)))
         (daemon-socket     (and daemon-websocket
                                 (websocket-socket daemon-websocket))))
    (define (service-daemon lsp)
      (let ((message (daemon:read-message daemon-websocket)))
        (lsp:handle-daemon-message lsp message
                                   #:debug? debug?
                                   #:imports imports)))
    (define (service-lsp lsp)
      (when (eof-object? (lookahead-u8 input-port))
        (exit EXIT_SUCCESS))
      (let ((message (rpc:read-message input-port)))
        (if (not message) (values lsp #f)
            (lsp:handle-message lsp message
                                #:debug? debug?
                                #:imports imports))))
    (define (servicable-ports)
      (if (not daemon-socket) input-port
          (match (select `(,daemon-socket ,input-port) '() '())
            (((ports ...) () ()) ports))))
    (let loop ((lsp (make-lsp #f)))
      (unless (port-closed? input-port)
        (let ((lsp response
                   (if (not daemon-socket) (service-lsp lsp)
                       (let ((ports (servicable-ports)))
                         (cond
                          ((memq daemon-socket ports)
                           (unless (eof-object? (lookahead-u8 daemon-socket))
                             ;; XXX TODO: maybe exit when daemon exits?
                             ;; reconnect??
                             (service-daemon lsp)))
                          ((memq input-port ports)
                           (service-lsp lsp))
                          (else (values lsp #f)))))))
          (when response
            (display response output-port)
            (force-output output-port))
          (loop lsp))))))
