/*
 * Copyright (C) 2020,2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

let globals = {
  spacing: 100,
  hpadding: 5,
  hsep: 15,
  vpadding: 5,
  barwidth: 10,
  verticalPerTime: 25,
  verticalStart: 25
};

class Header extends Viz {
  constructor(data) { 
    super(data); // {text, role, states, lifeline}
    this.klass = 'Header';
    this.instance = data.instance;
    this.role = data.role;
    this.states = data.states;
    this.stateEntry = null;
    this.lifeline = data.lifeline;
  }

  initViz() {
    let color =
        (this.role == 'component'
         || this.role == 'interface') ? '#AEE8A0' : '#FFFC80';

    let text = this.instance.replace(/.*\./,'');
    let btext = new Text(text);
    btext.bold = true;
    btext.refreshMe(); // always needed after font changes!
    let rows = [[btext]]; // one rows, one column
    if (this.states && this.states.length > 0) {
      let index = this.lifeline.index;
      let stateEntry = this.states[0].find(entry => entry.index == index);
      if (stateEntry) {
        let w = this.calcDimensions(index);
        // init table with empty text, but take nrLines into account
        let text = this.emptyLines(w.nrLines);
        let names = new Text(text);
        let values = new Text(text);
        names.setWidth(w.namesWidth);
        values.setWidth(w.valuesWidth);
        this.stateTable = new Table([[names, values]]); // one row, two columns
        this.stateTable.hpad = 0;
        this.stateTable.vpad = 0;
        this.stateTable.refreshMe();
        rows.push([this.stateTable]); // two rows, one column
      }
    } 
    this.table = new Table(rows);
    this.table.centered = true;
    this.table.refreshMe();
    let box = new RoundedBoundingBox(this.table);
    box.padding = 0;
    box.round = 10;
    box.color = color;
    this.refreshMe();
    this.setViz(box);
  }

  calcDimensions(index) {
    let result = {namesWidth: 0, valuesWidth: 0, nrLines: 0};
    this.states.forEach(state => {
      let stateEntry = state.find(entry => entry.index == index);
      if (stateEntry) {
        result.nrLines = Math.max(result.nrLines, stateEntry.state.length);
        let names = stateEntry.state.map(entry => entry.name).join('\n');
        let values = stateEntry.state.map(entry => entry.value).join('\n');
        let text = new Text(names);
        result.namesWidth = Math.max(result.namesWidth, text.width);
        text = new Text(values);
        result.valuesWidth = Math.max(result.valuesWidth, text.width);
      }
    });
    return result;
  }

  emptyLines(nr) {
    let text = '';
    for (let i = 2; i <= nr; i++) {
      text = text + '\n';
    }
    return text;
  }
  
  refreshMe() {
    if (this.stateEntry) {
      // cope with changes in stateEntry
      let names = this.stateEntry.map(entry => entry.name).join('\n');
      let values = this.stateEntry.map(entry => entry.value).join('\n');
      let content = this.stateTable.content[0];
      content[0].text = names;
      content[0].refreshMe();
      content[1].text = values;
      content[1].refreshMe();
      this.stateTable.refreshMe();
    } 
    this.table.refreshMe();
    this.viz.refreshMe();
  }

  midOffset() {
    return this.width/2;
  }

  setState(stateEntry) {
    if (stateEntry) {
      this.stateEntry = stateEntry;
      this.refreshMe();
    }
  }
}

class HeaderGroup extends Viz {
  constructor(data) {
    super(data); // {name, elements, foreign, lifelineGroup, states}
    this.klass = 'HeaderGroup';
    this.name = data.name;
    this.elements = this.lifelinesToHeaders(data.elements);
    this.lifelineGroup = data.lifelineGroup;
    this.states = data.states;
    this.foreign = data.foreign;
  }

  lifelinesToHeaders(elements) {
    return elements.map(element => element.isA('Lifeline') ? element.header : element.header);
  }

  firstHeader() {
    let frst = this.elements[0];
    return frst.isA('Header') ? frst : frst.firstHeader();
  }

  lastHeader() {
    let lst = this.elements[this.elements.length-1];
    return lst.isA('Header') ? lst : lst.lastHeader();
  }

  firstDepth() {
    let frst = this.elements[0];
    return frst.isA('Header') ? 1 : frst.firstDepth() + 1;
  }
  
  maxDepth() {
    let max = 0;
    this.elements.forEach(element => {
      let d = element.isA('HeaderGroup') ? element.maxDepth() : 0;
      max = Math.max(max, d);
    });
    return max + 1;
  }

  initViz(light) {
    //let text = this.text.replaceAll(/\./g,'.\n');
    let text = this.name.replace(/.*\./,'');

    let name = new Text(text);
    this.elements.forEach(element => element.initViz(!light));

    let px = 0;
    let mxprev = 0;
    this.elements.forEach(element => {
      if (element.isA('Header')) {
        let mdist = (px + element.midOffset()) - mxprev;
        mdist = Math.max(mdist, globals.spacing);
        px = mxprev + mdist - element.midOffset();
        element.move(px, 0);
        mxprev = element.x + element.midOffset();
        px += element.width + globals.hsep;
      } else {
        let maxd = element.maxDepth();
        let frst = element.firstHeader();
        let mdist = (px + frst.midOffset()) - mxprev;
        mdist = Math.max(mdist, globals.spacing);
        px = mxprev + mdist - frst.midOffset();
        element.move(px, - maxd*(globals.vpadding*2 + name.height));
        let lst = element.lastHeader();
        mxprev = lst.x + lst.midOffset();
        px += element.width + globals.hsep;
      }
    });
    let eframe = new Frame(this.elements);
    name.hCenter(eframe.width/2);
    eframe.shift(0, name.height + globals.vpadding);
    let frame = new Frame([name, eframe]);
    let bbox = new BoundingBox(frame)
    bbox.padding = globals.hpadding;
    bbox.color = this.foreign ? '#E5FFE5' : light ? '#C9FFC9' : '#C0F7BC'; 
    bbox.shadowed = false;
    bbox.strokeWeight = .1;
    if (this.name == "") {
      bbox.color = 'white';
      bbox.shadowed = false;
      bbox.strokeWeight = 0;
    }
    this.setViz(bbox);
  }

  midOffset() {
    return this.elements[0].midOffset() + globals.hpadding;
  }
}

class Activity extends Viz {
  constructor(data) {
    super(data); // {key, time, location, lifeline}
    this.key = data.key;
    this.time = data.time;
    this.location = data.location;
    this.lifeline = data.lifeline;
    this.klass = 'Activity';
  }

  initViz() {
    this.setViz(new Dot());
  }
}

class AnyBar extends Viz {
  constructor(data) {
    super(data); // {startTime, endTime, lifeline}
    this.klass = 'AnyBar';
    this.startTime = data.startTime;
    this.endTime = data.endTime;
    this.lifeline = data.lifeline;
  }

  initViz() {
    this.bar = new Box();
    this.bar.width = globals.barwidth;
    this.barExtend = 4;
    this.bar.height = this.timeToY(this.endTime) - this.timeToY(this.startTime) + 2*this.barExtend;
    this.bar.color = 'white';
    this.setViz(this.bar);
  }

  timeToY(t) {
    return globals.verticalStart + t * globals.verticalPerTime - this.barExtend + 17; // TODO: why 17?
  }
}

class BlockedBar extends AnyBar {
  constructor(data) {
    super(data); // {startTime, endTime, lifeline}
    this.klass = 'BlockedBar';
  }

  initViz() {
    super.initViz();
    this.bar.color = '#DDDDDD';
    this.refreshMe();
  }
}

class ActivityBar extends AnyBar {
  constructor(data) {
    super(data); // {startTime, endTime, blocked, lifeline}
    this.klass = 'ActivityBar';
    this.blocked = data.blocked.map(bl => new BlockedBar({...bl, lifeline: data.lifeline}));
  }

  initViz() {
    super.initViz();
    this.bar.color = '#FE938C';
    this.blocked.forEach(bl => {
      bl.initViz();
      // relative positioning:
      bl.move(0, (bl.startTime - this.startTime)*globals.verticalPerTime);
    });
    let frame = new Frame([this.bar].concat(this.blocked));
    this.setViz(frame);
  }
}

class Eligible extends Viz {
  constructor(data) {
    super(data); // {text, type, illegal}
    this.klass = 'Eligible';
    this.text = data.text;
    this.type = data.type;
    this.illegal = data.illegal;
  }

  initViz() {
    //let text = this.text.replaceAll(/\./g,'.\n');
    let text = this.text.replace(/.*\./,'');
    this.button = new Button(text, this.buttonHandleEligible, this);
    this.button.color = this.illegal ? '#EEEEEE'
      : this.text == '<back>' ? '#5BC0EB' // '#6FFFE9'
      : '#B9E5F7';
    this.button.strokeWeight = 1;
    this.button.strokeColor = this.illegal ? '#CCCCCC' : '#888888';
    this.button.padding = 4;

    this.setViz(this.button); // TODO
  }

  show(p) {
    this.viz.show(p);
  }

  buttonHandleEligible() {
    // take care: this.isA('Button')
    this.manager.handleEligible()
  }
  
  handleEligible() {
    console.log('PRESSED ' + this.text);
  }
  
}

class Lifeline extends Viz {
  constructor(data) {
    super(data); // {index, length, header, activities, labels, states, bars}
    this.klass = 'Lifeline';
    this.index = data.index;
    this.header = new Header({...data.header, states: data.states, lifeline: this});
    this.activities = this.data.activities.map(activity => new Activity({...activity, lifeline: this}));
    this.bars = this.data.bars.map(bar => new ActivityBar({...bar, lifeline: this}));
    this.eligibles = this.data.labels.map(eligible => new Eligible(eligible));
  }

  initViz() {
    let eligibleVertical = 30;

    let len = globals.verticalStart + this.data.length*globals.verticalPerTime;
    let line = new VDottedLine(len);

    // header is done separately!

    this.activities.forEach(activity => {
      activity.initViz();
      activity.move(0, globals.verticalStart + activity.time*globals.verticalPerTime);
    });

    this.bars.forEach(bar => {
      bar.initViz();
      bar.move(-bar.width/2, bar.timeToY(bar.startTime));
    });

    this.eligibles.forEach((eligible,i) => {
      eligible.initViz();
      eligible.move(0, len + i*eligibleVertical);
      eligible.hCenter(0);
    });
    let frame = new Frame([line].concat(this.activities, this.bars, this.eligibles));
    this.setViz(frame);
  }

  midOffset() {
    return this.width/2;
  }

  headerOffset() {
    return this.header.midOffset() - this.midOffset();
  }

  findActivity(key) {
    return this.activities.find(activity => activity.key == key);
  }

  setState(stateEntry) {
    if (stateEntry)
      this.header.setState(stateEntry)
  }

  highlight(on) {
    this.viz.content[0].highlight(on);
  }
}

class LifelineGroup extends Viz {
  constructor(data) {
    super(data); // {name, lifelines, states}
    this.klass = 'LifelineGroup';
    this.name = data.name;
    this.states = data.states;
    this.elements = this.partition(data.lifelines);
    this.foreign = this.isForeign(this.elements);
    if (this.foreign) {
      this.elements = this.elements.filter(element =>
        !(element.header.isA('Header') && element.header.role == 'foreign'));
    }
    this.header = new HeaderGroup({name: this.name, elements: this.elements,
                                   foreign: this.foreign,
                                   lifelineGroup: this, states: this.states});
  }

  isForeign(elements) {
    return elements.find(element =>
        element.header.isA('Header') && element.header.role == 'foreign');
  }

  partition(lifelines) {
    let prefix = function(ll) {
      let str = ll.header.instance;
      str = (this.name == '') ? str : str.replace(this.name + '.', '');
      return str.replace(/\..*/,'');
    }.bind(this);
    // step1: 
    let groups = [];
    let group = null;
    lifelines.forEach(ll => {
      if (group) {
        if (prefix(group[0]) == prefix(ll)) {
          group.push(ll);
        } else {
          groups.push(group);
          group = [ll];
        }
      } else {
        group = [ll];
      }
    });
    if (group) {
      groups.push(group);
    }

    let elements = groups.map(group => {
      if (group.length == 1) {
        let lldata = group[0];
        return new Lifeline({...lldata, states: this.states});
      } else {
        let p = prefix(group[0]);
        let name = (this.name == '') ? p : this.name + '.' + p;
        return new LifelineGroup({name: name, lifelines: group, states: this.states});
      }
    });
    return elements;
  }

  initViz() {
    this.elements.forEach(element => element.initViz());
    this.alignElements();
    let frame = new Frame(this.elements);
    this.setViz(frame);
  }

  alignElements() {
    this.elements.forEach(element => {
      if (element.isA('Lifeline')) {
        element.move(element.header.x + element.header.midOffset() - element.midOffset(), 0);
      } else {
        element.move(element.header.x + element.header.midOffset() - element.midOffset(), 0);
      }
    });
  }

  midOffset() {
    return this.elements[0].midOffset();
  }

  headerOffset() {
    return this.header.midOffset() - this.midOffset();
  }

  firstLifeline() {
    let frst = this.elements[0];
    return frst.isA('Lifeline') ? frst : frst.firstLifeline();
  }

  findActivity(key) {
    return this.elements
      .map(element => element.findActivity(key))
      .find(act => act != null);
  }

  findLifeline(index) {
    return this.elements
      .map(element => {
        if (element.isA('Lifeline')) {
          return (element.index == index) ? element : null;
        } else {
          return element.findLifeline(index);
        }
      })
      .find(ll => ll != null);
  }

  setState(state) {
    if (state) {
      this.elements.forEach(element => {
        if (element.isA('Lifeline')) {
          let entry = state.find(st => st.index == element.index);
          if (entry)
            element.setState(entry.state);
        } else {
          element.setState(state);
        }
      });
    }
  }
}

class Event extends Viz {
  constructor(data) {
    super(data); // {text, from, to, type, index}
    this.klass = 'Event';
    this.from;
    this.to;
    this.type = data.type;
    this.seqdiag;
    this.index = data.index;
  }

  initViz(seqdiag) {
    this.seqdiag = seqdiag;
    this.from = seqdiag.findActivity(this.data.from);
    this.to = seqdiag.findActivity(this.data.to);
    let arrow = new HArrow(0, 0, 0);
    if (this.type == 'return') arrow.dotted = true;
    let text = new Text(this.data.text);
    let content = [arrow, text];
    if (this.type == 'error') {
      let img = new Alert();
      let msg = new Text(this.data.messages.join ('\n'));
      content = content.concat([img, msg]);
    }
    let frame = new Frame(content); // (0,0) located
    this.setViz(frame);
    this.refresh();
  }

  refresh() {
    let frame = this.viz;
    let arrow = frame.content[0];
    let text = frame.content[1];

    let fromBounds = this.from.relativeBounds(this.seqdiag);
    let toBounds = this.to ? this.to.relativeBounds(this.seqdiag) : {x: fromBounds.x + 100};
    let height = fromBounds.y;
    // compensate for activity bar with:
    let left;
    if (fromBounds.x == toBounds.x) {
      left = toBounds.x + globals.barwidth/2;
      arrow.refreshMe(20, 0, 0);
    } else if (fromBounds.x < toBounds.x) {
      left = fromBounds.x + globals.barwidth/2;
      arrow.refreshMe(0, toBounds.x-fromBounds.x-globals.barwidth, 0);
    } else {
      left = toBounds.x + globals.barwidth/2;
      arrow.refreshMe(fromBounds.x-toBounds.x-globals.barwidth, 0, 0);
    }
    text.refresh();
    let text_bnd = text.bounds();
    if (fromBounds.x == toBounds.x) text.move(0, 0);
    else text.hCenter((fromBounds.x+toBounds.x)/2-left);
    text.shift(0, -(text_bnd.height+5));
    if (this.type == 'error') {
      let img = frame.content[2];
      let imgbnd = img.bounds();
      img.hCenter((fromBounds.x+toBounds.x)/2-left);
      img.shift(0, 5);
      let msg = frame.content[3];
      let msgbnd = msg.bounds();
      msg.hCenter((fromBounds.x+toBounds.x)/2-left);
      msg.shift(0, imgbnd.height + 15);
    }
    frame.refreshMe();
    this.move(left, height);
  }

  highlight(on) {
    this.viz.content[0].highlight(on);
  }

}

class Body extends Viz {
  constructor(data) {
    super(data); // {lifelines, events, states}
    this.klass = 'Body';
    // add first state to lifelines
    this.lifelineGroup = new LifelineGroup({name: '', lifelines: this.data.lifelines, states: this.data.states});
    this.events = this.data.events.map(d => new Event(d));
    this.header = this.lifelineGroup.header;
  }

  initViz() {
    this.lifelineGroup.initViz();
    this.events.forEach(event => event.initViz(this)); // need global info!
    let body = new Frame([this.lifelineGroup].concat(this.events));
    body.shift(this.headerOffset(), this.header.height);
    this.setViz(body);
  }

  headerOffset() {
    return this.lifelineGroup.headerOffset();
  }
  
  findActivity(key) {
    return this.lifelineGroup.findActivity(key);
  }

  findLifeline(index) {
    return this.lifelineGroup.findLifeline(index);
  }

  setState(state) {
    if (state)
      this.lifelineGroup.setState(state);
  }
}

class SequenceDiagram extends Viz {
  constructor(data) {
    super(data); // {lifelines, events, states}
    this.klass = 'SequenceDiagram';
    // restore lifeline index field:
    data.lifelines.forEach( (ll,index) => ll.index = index);
    // restore lifeline length field:
    let maxTime = 0;
    data.lifelines.forEach( ll => {
      ll.activities.forEach(act => maxTime = Math.max(maxTime, act.time));
    });
    maxTime += 1;
    // compensate lifeline length for error event:
    if (data.events && data.events.length > 0 && data.events[data.events.length-1].type == 'error') {
      maxTime += 2; // TODO: is 2 ok?
    }
    data.lifelines.forEach(ll => ll.length = maxTime+1);
    // add index to events:
    data.events.forEach( (event, index) => event.index = index);
    // add lifeline index to states:
    if (data.states) {
      data.states.forEach(state => {
        state.forEach(stateEntry => {
          let ll = data.lifelines.find(ll => ll.header.instance == stateEntry.instance);
          stateEntry.index = ll ? ll.index : 0;
        });
      });
    }
    // add activity bars:
    addBars(data);
    
    this.body = new Body(data);
    this.header = this.body.header;
    this.states = data.states;
  }

  initViz() {
    this.header.initViz();
    this.body.initViz();
    this.setViz(new Frame([this.header,this.body]));
  }

  findActivity(key) {
    return this.body.findActivity(key);
  }

  setActive(stateIndex) {
    if (this.states && stateIndex < this.states.length)
      this.body.setState(this.states[stateIndex]);
  }
}
