/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

class SequenceDiagramP5 {

  /*
   * interface to the outside world, through functions
   *   in.draw(data)
   *     show the data as a diagram
   *   in.dimensions(px, py, width, height)
   *     set the location (px, py) and size (with, height) of the
   *     diagram
   *   out.selected(location)
   *     process the selected node, using its location
   *
   *   out functions can be (re)defined by the outside world.
   */

  constructor(parent) {
    this.in = {
      draw: function (data) {
        this.sketch.data = data;
        if (this.sketch.set_up) {
          this.sketch.initDiagram(data);
          this.sketch.redraw();
        }
      }.bind(this),
      dimensions: function (px, py, width, height) {
        this.sketch.dimensions(px, py, width, height);
      }.bind(this),
    };
    this.out = {
      event: function (event) {
        console.log('event: %j', event);
      }
      ,
      selected: function (location) {
        console.log('selected location: %j', location);
      }
    };

    // default dimensions: whole window
    this.px = 0;
    this.py = 0;
    this.fixedSize = false;
    this.width = window.innerWidth;
    this.height = window.innerHeight;

    // interface with sketch:
    this.sketch = new p5(this.sequenceSketch, parent);
    this.sketch.in = this.in;
    this.sketch.out = this.out;
    this.sketch.myPx = this.px;
    this.sketch.myPy = this.py;
    this.sketch.fixedSize = this.fixedSize;
    this.sketch.myWidth = this.width;
    this.sketch.myHeight = this.height;
  };

  sequenceSketch(p) {
    // interface with surrounding StateDiagramP5 object:
    p.in = null;
    p.out = null;
    p.myPx = null;
    p.myPy = null;
    p.fixedSize = false;
    p.myWidth = null;
    p.myHeight = null;

    p.dimensions = function(px, py, width, height) {
      p.myPx = px;
      p.myPy = py;
      p.myWidth = width;
      p.myHeight = height;
      p.fixedSize = true;
      if (p.set_up) {
        world.positionCanvas(px, py);
        world.resizeCanvas(width, height);
        p.redraw();
      }
    };

    // initDiagram is accessed outside sketch
    // do not call before p5 setup!
    p.initDiagram = function(data) {
      try {
        p.data = data;
        seqdiag = new SequenceDiagram(data);
        seqdiag.initViz();
        seqdiag.body.shift(10,0);
        seqdiag.header.shift(10,0);
        shiftToBottom();
        activeEventIndex = seqdiag.body.events.length-1;
        highlightActiveEvent();
        p.cursor(p.ARROW);
        p.noLoop();
      } catch(e) {
        seqdiag = null;
        message = new BoundingBox(new Text(''));
        message.padding = 30;
        message.content.size = 30;
        message.color = '#F37361';
        setMessage(message, 'invalid input: see console [F12] for details');
        console.log('%j: %j', e, data);
        p.noLoop();
      }
    };

    function setMessage(diagram, msg) {
      diagram.content.text = msg;
      diagram.content.refresh();
      diagram.center(p.windowWidth/2, p.windowHeight/2);
      diagram.refresh();
    }

    
    let world = null;
    let seqdiag = null;
    let message = null;
    
    p.setup = function() {
      let px = p.myPx;
      let py = p.myPy;
      let w = p.fixedSize ? p.myWidth : p.windowWidth;
      let h = p.fixedSize ? p.myHeight : p.windowHeight;
      world = new World(p, w, h);
      if (px != null && py != null) world.positionCanvas(px, py);
      p.set_up = true;
      p.cursor(p.WAIT);
      if (p.data) {
        p.initDiagram(p.data);
      }
      p.noLoop();
    };

    p.draw = function() {
      p.background(255);
      world.set();
      if (seqdiag) {
        seqdiag.body.show(p);
        // always at top of screen:
        let w0 = world.canvasToWorld(0,0);
        seqdiag.header.move(seqdiag.header.x, w0.y);
        seqdiag.header.show(p);
        if (focusAnimating()) {
          p.loop();
        } else {
          p.noLoop();
          checkRepeat(handleKey);
        }
      } else if (message) {
        message.show(p);
        p.noLoop();
      }
    };

    function mouseInCanvas() {
      if (!p.fixedSize) return true; // canvas spans whole window
      if (p.myWidth && !(0 <= p.mouseX && p.mouseX <= p.myWidth)) return false;
      if (p.myHeight && !(0 <= p.mouseY && p.mouseY <= p.myHeight)) return false;
      return true;
    }

    p.windowResized = function() {
      if (!p.fixedSize)
        world.resizeCanvas(p.windowWidth, p.windowHeight);
    };

    function selectionOfKlass(sel, klass) {
      let obj = sel.find(s => s.isA(klass));
      return obj;
    }
    function selection(sel) {
      // take care: header stuff 'hides' lifeline stuff
      return selectionOfKlass(sel, 'Header')
        || selectionOfKlass(sel, 'HeaderGroup')
        || selectionOfKlass(sel, 'Button')
        || selectionOfKlass(sel, 'Event')
        || selectionOfKlass(sel, 'Lifeline');
    }

    let highlightedObj = null;
    function highlight(obj) {
      if (highlightedObj) {
        highlightedObj.highlight(false);
      }
      highlightedObj = obj;
      highlightedObj.highlight(true);
      p.redraw();
    }

    function resetHighlight() {
      if (highlightedObj) {
        highlightedObj.highlight(false);
        highlightedObj = null;
      }
    }

    let activeEventIndex = 0;
    let highlightedEvent = null;
    let activeLifelineIndex = -1;
    let highlightedLifeline = null;

    let dragging = false;
    let draggingOffsetX = 0;
    let draggingOffsetY = 0;
    
    function lastEvent() {
      let nrEvents = seqdiag.body.events.length;
      return (nrEvents > 0) ? seqdiag.body.events[nrEvents-1] : null;
    }

    function hasMatchingLifeline(activity, lifeline) {
      return activity && lifeline && activity.lifeline.index == lifeline.index;
    }
    
    function selectNextActiveEvent(down) {
      function matches(eventIndex, lifeline) {
        if (!lifeline) return true;
        let event = seqdiag.body.events[eventIndex];
        return hasMatchingLifeline(event.from, lifeline) || hasMatchingLifeline(event.to, lifeline);
      }
      let nrEvents = seqdiag.body.events.length;
      if (down) {
        if (activeEventIndex < nrEvents-1)
          activeEventIndex++;
        while (activeEventIndex < nrEvents-1 && !matches(activeEventIndex, highlightedLifeline))
          activeEventIndex++;
      } else {
        if (activeEventIndex >= 0)
          activeEventIndex--;
        while (activeEventIndex >= 0 && !matches(activeEventIndex, highlightedLifeline))
          activeEventIndex--;
      }
      highlightActiveEvent();
      p.redraw();
    }
    
    function highlightActiveEvent() {
      if (highlightedEvent) {
        highlightedEvent.highlight(false);
      }
      if (activeEventIndex >= 0) {
        highlightedEvent = seqdiag.body.events[activeEventIndex];
        highlightedEvent.highlight(true);
        focus(highlightedEvent);
        let fromHighlighted = hasMatchingLifeline(highlightedEvent.from, highlightedLifeline);
        let activity = fromHighlighted || !highlightedEvent.to ? highlightedEvent.from : highlightedEvent.to;
        let location = activity.location;
        if (location) p.out.selected({...location,
                                      'working-directory': p.data['working-directory']});
      } else {
        highlightedEvent = null;
        // do not change focus upon deselect
        // focus(null);
      }
      let index = Math.min(activeEventIndex+1, seqdiag.states.length-1);
      seqdiag.setActive(index);
    }

    function resetHighlightEvent() {
      activeEventIndex = -1;
      highlightActiveEvent();
    }

    function setActiveLifeline(index) {
      // deselect by second activate:
      activeLifelineIndex = (activeLifelineIndex == index) ? -1 : index;

      if (highlightedLifeline) {
        highlightedLifeline.highlight(false);
      }
      if (activeLifelineIndex >= 0) {
        highlightedLifeline = seqdiag.body.findLifeline(activeLifelineIndex);
        highlightedLifeline.highlight(true);
        p.redraw();
      } else {
        highlightedLifeline = null;
      }
    }

    function focus(obj) {
      // obj isA Event or null 
      let topLeft = world.canvasToWorld(0, 0);
      let bottomRight = world.canvasToWorld(world.canvas.width, world.canvas.height);
      let oldX = 0;
      let oldY = 0;
      let newX = 0;
      let newY = 0;
      let margin = 10/world.scale;
      let bodybnd = seqdiag.body.bounds();
      let headerbnd = seqdiag.header.bounds();
      if (obj) {
        let objbnd = obj.relativeBounds(seqdiag.body.viz);
        // extend object bounds with the width of both headers:
        function extend(hdr) {
          let bnd = hdr.relativeBounds(seqdiag.header.viz);
          if (headerbnd.x + bnd.x < bodybnd.x + objbnd.x) {
            let diff = (bodybnd.x + objbnd.x) - (headerbnd.x + bnd.x);
            objbnd.x -= diff;
            objbnd.width += diff;
          }
          if (headerbnd.x + bnd.x + bnd.width > bodybnd.x + objbnd.x + objbnd.width) {
            let diff = (headerbnd.x + bnd.x + bnd.width) - (bodybnd.x + objbnd.x + objbnd.width);
            objbnd.width += diff;
          }
        }
        if (obj.from) {
          let hdr = obj.from.lifeline.header;
          extend(hdr);
        }
        if (obj.to) {
          let hdr = obj.to.lifeline.header;
          extend(hdr);
        }
        // absolute coordinates obj top:
        oldX = bodybnd.x + objbnd.x;
        newX = (objbnd.width > (bottomRight.x - topLeft.x)) ? (bottomRight.x - topLeft.x)/2 - objbnd.width/2
          : (oldX < topLeft.x) ? topLeft.x + margin
          : (oldX + objbnd.width > bottomRight.x) ? bottomRight.x - objbnd.width - margin
          : oldX;
        oldY = bodybnd.y + objbnd.y;
        newY = (oldY < topLeft.y + headerbnd.height) ? headerbnd.y + headerbnd.height + margin
          : (oldY + objbnd.height > bottomRight.y) ? bottomRight.y - objbnd.height - margin
          : oldY;
        if (obj.index == seqdiag.body.events.length-1) {
          // scroll down the bottom of body, to show the eligibles
          oldY = bodybnd.y + bodybnd.height;
          newY = (oldY > bottomRight.y) ? bottomRight.y - margin : oldY;
        }
      } else {
        // assure right side header is left-aligned to view,
        oldX = headerbnd.x;
        newX = (oldX + headerbnd.width > bottomRight.x) ? bottomRight.x - headerbnd.width - margin
          : oldX;
        newX = (newX < topLeft.x) ? topLeft.x + margin
          : newX;
        // scroll up to show body top
        oldY = bodybnd.y;
        newY = (oldY < topLeft.y + headerbnd.height) ? headerbnd.y + headerbnd.height + margin
          : oldY;
      }
      if (newX != oldX || newY != oldY) {
        startFocusAnimation(newX - oldX, newY - oldY);
      }
    }

    function shiftToBottom() {
      let bottomRight = world.canvasToWorld(world.canvas.width, world.canvas.height);
      let margin = 10/world.scale;
      let bodybnd = seqdiag.body.bounds();
      // scroll up to show body bottom
      let oldY = bodybnd.y + bodybnd.height;
      if (oldY > bottomRight.y) {
        let newY = bottomRight.y - margin;
        seqdiag.body.shift(0, newY - oldY);
      }
    }
    
    let nrFocusFrames = 10;
    let focusShift = {x:0, y:0};
    let focusFrame = nrFocusFrames;

    function startFocusAnimation(dx, dy) {
      focusShift = {x: 1/nrFocusFrames*dx, y: 1/nrFocusFrames*dy};
      seqdiag.body.shift(focusShift.x, focusShift.y);
      seqdiag.header.shift(focusShift.x, 0);
      focusFrame = 1;
    }

    function focusAnimating() {
      if (focusFrame == nrFocusFrames) return false;
      seqdiag.body.shift(focusShift.x, focusShift.y);
      seqdiag.header.shift(focusShift.x, 0);
      focusFrame++;
      return true;
    }

    function limitVertical() {
      // limit vertical scrolling
      let bbnd = seqdiag.body.bounds();
      let hbnd = seqdiag.header.bounds();
      let bnd = {x: bbnd.x, y: bbnd.y-hbnd.height, width: bbnd.width, height: bbnd.height+hbnd.height};
      // allow some space at the bottom
      bnd.height += 15;
      world.limitVertical(bnd);
    }

    p.keyPressed = function() {
      if (!mouseInCanvas()) return true;
      if (!seqdiag) return false;
      startTimeout();
      handleKey();
    };

    p.keyReleased = function() {
      if (!mouseInCanvas()) return true;
      stopTimeout();
      return false;
    }

    function handleKey() {
      // use p5 'key' variable for ASCII keys
      // use p5 'keyCode' variable for non-ASCII keys
      if (p.keyCode == p.DOWN_ARROW) {
        selectNextActiveEvent(true); // down
      } else if (p.keyCode == p.UP_ARROW) {
        selectNextActiveEvent(false); // up
      } else if (p.key === '-') {
        if (p.keyIsDown(p.CONTROL)) {
          world.zoomAround(0, 0, world.zoomOutFactor);
          p.redraw();
        } 
      } else if (p.key === '+' || p.key == '=') {
        if (p.keyIsDown(p.CONTROL)) {
          world.zoomAround(0, 0, world.zoomInFactor);
          p.redraw();
        } 
      } else if (p.key === '0') {
        if (p.keyIsDown(p.CONTROL)) {
          world.scale = 1;
          world.fit(seqdiag.header.bounds());
          limitVertical();
          p.redraw();
        } 
      } else if (p.key === '1') {
        if (p.keyIsDown(p.CONTROL)) {
          world.scale = 1;
          world.zoomAround(0, 0, 1);
          limitVertical();
          p.redraw();
        } 
      } else {
        // default behaviour for unbound keys:
        return true;
      }
      // suppress default behaviour:
      return false;
    };

    let timeout = {
      delay: 250, // const
      handler: null,
      start: 0,
      wait: 0,
      running: false
    }
    
    function startTimeout() {
      timeout.running = true;
      timeout.wait = timeout.delay * 2; // initial delay is larger
      timeout.start = Date.now();
    };

    function stopTimeout() {
      timeout.running = false;
      if (timeout.handler != null) {
        clearTimeout(timeout.handler);
        timeout.handler = null;
      }
    }

    function checkRepeat(fn) {
      if (timeout.running && timeout.handler == null) {
        let now = Date.now();
        let delta = now - timeout.start;
        timeout.handler = setTimeout(() => {
          // restart timer:
          timeout.start = Date.now();
          timeout.wait = timeout.delay;
          timeout.handler = null;
          fn();
        }, timeout.wait - delta);
      }
    }

    p.mousePressed = function(e) {
      if (!mouseInCanvas()) return;
      if (!seqdiag) return;
      let wpt = world.mousePoint();
      // take care: handle body and header separately
      if (e.ctrlKey) {
        draggingOffsetX = wpt.x - seqdiag.body.x;
        draggingOffsetY = wpt.y - seqdiag.body.y;
      } else {
        let sel = seqdiag.body.objectsAt(wpt.x, wpt.y).concat(seqdiag.header.objectsAt(wpt.x, wpt.y));
        let obj = selection(sel);
        if (obj) {
          if (obj.klass == 'Button') {
            highlight(obj);
            p.redraw();
          }
        }
      }
    };

    p.mouseDragged = function(e) {
      if (!mouseInCanvas()) return;
      if (!seqdiag) return;
      if (e.ctrlKey) {
        dragging = true;
        let wpt = world.mousePoint();
        let margin = 10/world.scale;
        let bodybnd = seqdiag.body.bounds();
        let headerbnd = seqdiag.header.bounds();
        seqdiag.body.move(wpt.x-draggingOffsetX, wpt.y-draggingOffsetY);
        seqdiag.header.move(wpt.x-draggingOffsetX+headerbnd.x-bodybnd.x, headerbnd.y);

        // limit vertical dragging for body:
        let topLeft = world.canvasToWorld(0, 0);
        let bottomRight = world.canvasToWorld(world.canvas.width, world.canvas.height);
        // too high? move down to fit bottom:
        bodybnd = seqdiag.body.bounds();
        if (bodybnd.y + bodybnd.height < bottomRight.y)
          seqdiag.body.move(bodybnd.x, bottomRight.y - bodybnd.height - margin);
        // too low? move up to fit top:
        bodybnd = seqdiag.body.bounds();
        if (bodybnd.y > headerbnd.y + headerbnd.height)
          seqdiag.body.move(bodybnd.x, headerbnd.y + headerbnd.height);
        p.redraw();
      } else dragging = false;
    };

    p.mouseReleased = function(e) {
      if (dragging) {
        dragging = false;
      } else {
        if (!mouseInCanvas()) return;
        if (!seqdiag) return;
        let wpt = world.mousePoint();
        // take care: handle body and header separately
        let sel = seqdiag.body.objectsAt(wpt.x, wpt.y).concat(seqdiag.header.objectsAt(wpt.x, wpt.y));
        console.log ('sel=%j', sel);
        resetHighlight();
        let obj = selection(sel);
        if (obj) {
          console.log ('klas=%j', obj.klass);
          if (obj.klass == 'Button') {
            obj.callback();
            p.cursor(p.WAIT);
            p.out.event (obj.manager.text);
          } else if (obj.klass == 'Event') {
            activeEventIndex = obj.index;
            highlightActiveEvent();
          } else if (obj.klass == 'Header') {
            setActiveLifeline(obj.lifeline.index);
          } else if (obj.klass == 'HeaderGroup') {
            // noop
          } else if (obj.klass == 'Lifeline') {
            setActiveLifeline(obj.index);
          } 
        } else {
          resetHighlightEvent();
        }
        p.redraw();
      }
    };

    p.mouseWheel = function(e) {
      if (!mouseInCanvas()) return;
      if (!seqdiag) return;
      // HACK: override world.mouseWheel to force zooming around y=0
      if (e.ctrlKey) {
        let zoom = (e.deltaY < 0) ? world.zoomInFactor : world.zoomOutFactor;
        world.zoomAround(p.mouseX, 0, zoom);
      } else {
        world.mouseWheel(e);
      }
      limitVertical();
      p.redraw();
    };
  };
};

