/*
 * Copyright (C) 2020,2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

/***********************************************************
 * Building blocks are base for all visualisation.
 ***********************************************************/

/*
 * When DEBUG is on, additional visualisation is presented.
 * currently: each Frame will show a red border
 */
let DEBUG = false;

/*
 * BuildingBlock: the abstract base class for all building blocks.
 * elements:
 *   klass: string
 *     type identification
 *   parent: the containing BuildingBLock (if any)
 *   x, y:  float
 *     location (default (0,0) which is top-left)
 *   width, height: float
 *     dimensions (default (0, 0))
 * functions:
 *   isA: string --> bool
 *     returns true if the BuildingBlock is of the given klass
 *   bounds: nil --> {float, float, float, float, float, float}
 *     returns current {x, y, with, height, scaleX, scaleY}
 *   relativeBounds: BuildingBlock --> {float, float, float, float, float, float}
 *     returns current {x, y, with, height, scaleX, scaleY} relative to a given parent
 *   refreshMe: nil -> nil
 *     recalculates dimensions (if relevant)
 *   refresh: nil -> nil
 *     refreshes its sub blocks (if any) and recalculates dimensions (if relevant)
 *   move: (float, float) --> nil
 *     sets the location, and moves its sub blocks (if any)
 *   hCenter: float --> nil
 *     horizontally center around given x
 *   center: (float, float) --> nil
 *     horizontally and vertically center around given (x, y)
 *   shift: (float, float) --> nil
 *     relative move over (dz, dy)
 *   show: nil --> nil
 *     the actual representation in p5js terms
 *   atPos: (float, float) --> bool
 *     returns true if the given (x, y) is within bounds
 *   objectsAt: (float, float) --> [BuildingBlock]
 *     if atPos(x, y) returns itself and all sub blocks that are atPos(x, y)
 *   highlight: bool --> nil
 *     set or reset highlighted
 */
class BuildingBlock {
  constructor() {
    this.klass = 'BuildingBlock';
    this.parent = null;
    this.x = 0;
    this.y = 0;
    this.width = 0;
    this.height = 0;
    this.scaleX = 1;
    this.scaleY = 1;
    this.highlighted = false;
    this.highlightColor = '#55AAFF';
  }

  isA(str) {
    return this.klass == str;
  }

  setParent(parent) {
    this.parent = parent;
  }

  bounds() {
    return {x: this.x, y: this.y,
            width: this.width*this.scaleX, height: this.height*this.scaleY,
            scaleX: this.scaleX, scaleY: this.scaleY};
  }

  relativeBounds(parent) {
    let bnd = this.bounds();
    if (this.parent && this.parent != parent) {
      let rbnd = this.parent.relativeBounds(parent);
      return {x: rbnd.x + bnd.x*rbnd.scaleX, y: rbnd.y + bnd.y*rbnd.scaleY,
              width: bnd.width*bnd.scaleX*rbnd.scaleX, height: bnd.height*bnd.scaleY*rbnd.scaleY,
              scaleX: bnd.scaleX*rbnd.scaleX, scaleY: bnd.scaleY*rbnd.scaleY};
    } else {
      return bnd;
    }
  }

  refresh() {
    this.refreshMe();
  }
  
  refreshMe() {
    // noop
  }
  
  move(x, y) {
    this.x = x;
    this.y = y;
  }

  hCenter(x) {
    let bnd = this.bounds();
    this.move(x-bnd.width/2, this.y);
  }

  center(x, y) {
    let bnd = this.bounds();
    this.move(x-bnd.width/2, y-bnd.height/2);
  }

  shift(dx, dy) {
    let oldx = this.x;
    let oldy = this.y;
    this.move(oldx+dx, oldy+dy);
  }

  scale(scaleX, scaleY) {
    this.scaleX = scaleX;
    this.scaleY = scaleY;
  }

  show(p) {
    // noop
  }

  atPos(x, y) {
    // check bounding box, allow some margin
    let eps = 2;
    return this.x - eps <= x && x <= this.x + this.width*this.scaleX + eps &&
      this.y - eps <= y && y <= this.y + this.height*this.scaleY + eps;
  }

  objectsAt(x, y) {
    return this.atPos(x, y) ? [this] : [];
  }

  highlight(on) {
    this.highlighted = on;
  }
}

/*
 * Box()
 *   a rectangle, default 1x1, colored black.
 *   currently its border look is fixed
 */
class Box extends BuildingBlock {
  constructor() {
    super();
    this.klass = 'Box';
    this.width = 1;
    this.height = 1;
    this.color = 'black';
    this.strokeColor = 'black';
    this.strokeWeight = .5;
  }
  
  show(p) {
    let shape = function() {
      p.rect(0, 0, this.width, this.height);
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    p.fill(this.color);
    p.noStroke();
    shape();
    p.noFill();
    if (this.highlighted) {
      p.stroke(this.highlightColor);
      p.strokeWeight(this.strokeWeight+2);
      shape();
    }
    p.stroke(0);
    p.strokeWeight(this.strokeWeight);
    shape();
    p.pop();
  }
}

/*
 * DTriangle()
 *   a downward triangle
 *   currently its border look is fixed
 */
class DTriangle extends BuildingBlock {
  constructor() {
    super();
    this.klass = 'DTriangle';
    this.width = 1;
    this.height = 1;
    this.color = 'black';
    this.strokeColor = 'black';
    this.strokeWeight = .5;
  }

  show(p) {
    let shape = function() {
      p.triangle(0, 0, this.width, 0, this.width/2, this.height);
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    p.noStroke();
    p.fill(this.color);
    shape();
    p.noFill();
    if (this.highlighted) {
      p.stroke(this.highlightColor);
      p.strokeWeight(this.strokeWeight+2);
      shape();
    }
    p.stroke(0);
    p.strokeWeight(this.strokeWeight);
    shape();
    p.pop();
  }
}

/*
 * Text(text)
 *   a (left-aligned) representation of given text (string)
 *   currently its looks (font, size, and color) are fixed
 *   its width is calculated using p5js primitives
 */
class Text extends BuildingBlock {
  constructor(text) {
    super();
    this.klass = 'Text';
    this.text = text;
    this.font = 'sans-serif'; // TODO
    this.size = 15; // TODO
    this.bold = false;
    this.fixedWidth = false;
    this.canvas = document.createElement("canvas");
    this.width = this.vWidth();
    this.height = this.vHeight();
  }

  setWidth(w) {
    this.fixedWidth = true;
    this.width = w;
  }

  vWidth() {
    let width = 0;
    // (c) https://www.w3docs.com/snippets/javascript/how-to-calculate-text-width-with-javascript.html
    let context = this.canvas.getContext("2d");
    context.font = (this.bold ? 'bold ' : '') + this.size + 'px ' + this.font;
    this.text.split(/\n/).forEach(line => {
      let metrics = context.measureText(line);
      width = Math.max(width, metrics.width);
    });
    return width;
  }

  vHeight() {
    let lines = this.text.split(/\n/);
    return this.size * lines.length;
  }
  
  refreshMe() {
    if (!this.fixedWidth) this.width = this.vWidth();
    this.height = this.vHeight();
  }
  
  show(p) {
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    p.textFont(this.font);
    p.textStyle(this.bold ? p.BOLD : p.NORMAL);
    p.textAlign(p.LEFT, p.BOTTOM);
    p.fill(0);
    p.strokeWeight(0);
    p.textSize(this.size);
    p.textLeading(this.size);
    p.text(this.text, 0, this.height);
    p.pop();
  }
}

/*
 * VDottedLine(len)
 *   a vertical dotted line of given length (float)
 *   currently its looks are fixed
 */
class VDottedLine extends BuildingBlock {
  constructor(len) {
    super();
    this.klass = 'VDottedLine';
    this.width = 0;
    this.height = len;
    this.color = 'black';
    this.weight = .5;
  }

  show(p) {
    let shape = function() {
      p.drawingContext.setLineDash([5, 5]);
      p.line(0, 0, 0, this.height);
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    p.noFill();
    if (this.highlighted) {
      p.stroke(this.highlightColor);
      p.strokeWeight(this.weight+2);
      shape();
    }
    p.strokeWeight(this.weight);
    p.stroke(126);
    shape();
    p.pop();
  }
}

/*
 * Dot()
 *   an (almost invisible) dot
 *   can be used as anchor point
 */
class Dot extends BuildingBlock {
  constructor() {
    super();
    this.klass = 'Dot';
    this.visible = true;
  }

  show(p) {
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    if (this.visible) {
      p.strokeWeight(.5);
      p.point(0, 0);
    }
    p.pop();
  }
}

/*
 * RectLine(fx, fy, tx, ty)
 *  a line from (fx, fy) to (tx, ty).
 */
class RectLine extends BuildingBlock {
  constructor(fx, fy, tx, ty) {
    super();
    this.klass = 'RectLine';
    this.color = 'black';
    this.weight = 1;
    this.refreshMe(fx, fy, tx, ty);
  }

  refreshMe(fx, fy, tx, ty) {
    this.fx = fx;
    this.fy = fy;
    this.tx = tx;
    this.ty = ty;
    this.x = Math.min(fx,tx);
    this.y = Math.min(fy,ty);
    this.width = Math.abs(fx-tx);
    this.height = Math.abs(fy-ty);
  }
         
  show(p) {
    let shape = function() {
      p.strokeCap(p.SQUARE);
      p.beginShape();
      p.vertex(this.fx-this.x, this.fy-this.y);
      p.vertex(this.tx-this.x, this.ty-this.y);
      p.endShape();
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    p.noFill();
    if (this.highlighted) {
      p.strokeWeight(this.weight+2);
      p.stroke(this.highlightColor);
      shape();
    }
    p.strokeWeight(this.weight);
    p.stroke(this.color);
    shape();
    p.pop();
  }
}

/*
 * DirectedRectLine(fx, fy, tx, ty)
 *  a 'triangular' line starting at (fx, fy), ending at (tx, ty).
 *  the thickness of the line diminishes from start towards end
 */
class DirectedRectLine extends BuildingBlock {
  constructor(fx, fy, tx, ty) {
    super();
    this.klass = 'DirectedRectLine';
    this.thicknessStart = 3;
    this.thicknessEnd = 0;
    this.color = 'black';
    this.refreshPoints(fx, fy, tx, ty);
  }

  refreshPoints(fx, fy, tx, ty) {
    this.fx = fx;
    this.fy = fy;
    this.tx = tx;
    this.ty = ty;
    this.x = Math.min(fx,tx);
    this.y = Math.min(fy,ty);
    this.width = Math.abs(fx-tx);
    this.height = Math.abs(fy-ty);
  }
  
  show(p) {
    let shape = function(inc) {
      let dx = this.tx-this.fx;
      let dy = this.ty-this.fy;
      let len = Math.sqrt(dx*dx + dy*dy);
      let ex = dy / len;
      let ey = dx / len;
      p.quad(this.fx - this.x + ex*(this.thicknessStart+inc), this.fy - this.y - ey*(this.thicknessStart+inc),
             this.tx - this.x + ex*(this.thicknessEnd+inc), this.ty - this.y - ey*(this.thicknessEnd+inc),
             this.tx - this.x - ex*(this.thicknessEnd+inc), this.ty - this.y + ey*(this.thicknessEnd+inc),
             this.fx - this.x - ex*(this.thicknessStart+inc), this.fy -this.y + ey*(this.thicknessStart+inc));
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    if (this.highlighted) {
      p.noStroke();
      p.fill(this.highlightColor);
      shape(2);
    }
    p.noStroke();
    p.fill(this.color);
    shape(0);
    p.pop();
  }
}

/*
 * HRectLine(fx, fy, tx, ty)
 *  a segmented line from (fx, fy) to (tx, ty).
 *  Currently 3 segments are shown, the middle one being horizontal.
 *  Corners are 'rounded'
 */
class HRectLine extends BuildingBlock {
  constructor(fx, fy, tx, ty) {
    super();
    this.klass = 'HRectLine';
    this.color = 'black';
    this.weight = 1;
    this.yMiddle = 10; // relative to fy
    this.curve = 2; // roundness of corners
    this.refreshPoints(fx, fy, tx, ty);
  }

  refreshPoints(fx, fy, tx, ty) {
    this.fx = fx;
    this.fy = fy;
    this.tx = tx;
    this.ty = ty;
    this.x = Math.min(fx,tx);
    this.y = Math.min(fy,ty);
    this.width = Math.abs(fx-tx);
    this.height = Math.abs(fy-ty);
  }
         
  show(p) {
    let shape = function() {
      p.strokeCap(p.SQUARE);
      p.noFill();
      p.beginShape();
      p.vertex(this.fx - this.x, this.fy - this.y);
      let sgn = (this.fx < this.tx) ? 1 : -1;
      if (Math.abs(this.fx - this.tx) < this.curve*2) {
        p.vertex(this.fx - this.x, this.fy+this.yMiddle - this.y);
      } else {
        p.vertex(this.fx - this.x, this.fy+this.yMiddle-this.curve - this.y);
        p.vertex(this.fx+this.curve*sgn - this.x, this.fy+this.yMiddle - this.y);
      }
      
      if (Math.abs(this.fx - this.tx) < this.curve*2) {
        p.vertex(this.tx - this.x, this.fy+this.yMiddle - this.y);
      } else {
        p.vertex(this.tx-this.curve*sgn - this.x, this.fy+this.yMiddle - this.y);
        p.vertex(this.tx - this.x, this.fy+this.yMiddle+this.curve - this.y);
      }
      p.vertex(this.tx - this.x, this.ty - this.y);
      p.endShape();
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    if (this.highlighted) {
      p.strokeWeight(this.weight+2);
      p.stroke(this.highlightColor);
      shape();
    }
    p.strokeWeight(this.weight);
    p.stroke(this.color);
    shape();
    p.pop();
  }
  
  atPos(x, y) {
    // bounding box isn't good enough
    if (!super.atPos(x, y)) return false;

    let eps = 2;
    let midy = this.y + this.yMiddle*this.scaleY;
    let rightx = this.x + this.width*this.scaleX;

    if (this.x == this.fx) {
      return (y < midy && this.x - eps <= x && x <= this.x + eps ||
              y > midy && rightx - eps <= x && x <= rightx + eps ||
              midy - eps <= y && y <= midy + eps);
    } else {
      return (y < midy && rightx - eps <= x && x <= rightx + eps ||
              y > midy && this.x - eps <= x && x <= this.x + eps ||
              midy - eps <= y && y <= midy + eps);
    }      
  }
}

/*
 * HArrow(fx, tx, y)
 *  a horizontal arrow from (fx, y) to (tx, y)
 *  currently its point dimensions are fixed
 */
class HArrow extends BuildingBlock {
  constructor(fx, tx, y) {
    super();
    this.klass = 'HArrow';
    this.pointHeight = 6;
    this.pointWidth = 10;
    this.color = 'black';
    this.weight = 1;
    this.dotted = false;
    this.dotsize = 3;
    this.refreshMe(fx, tx, y);
    this.fx;
    this.tx;
  }

  refreshMe(fx, tx, y) {
    this.fx = fx;
    this.tx = tx;
    this.x = Math.min(fx, tx);
    this.y = y - this.pointHeight/2;
    this.width = Math.abs(fx-tx);
    this.height = this.pointHeight;
  }

  move(x, y) {
    if (this.fx < this.tx) {
      this.refreshMe(x, x+this.tx-this.fx, y+this.pointHeight/2);
    } else {
      this.refreshMe(x+this.fx-this.tx, x, y+this.pointHeight/2);
    }
  }

  show(p) {
    let shape = function() {
      p.push();
      if (this.dotted) {
        p.drawingContext.setLineDash([this.dotsize, this.dotsize]);
      } 
      p.line(this.fx - this.x, this.pointHeight/2,
             this.tx - this.x, this.pointHeight/2);
      p.pop();
      let dir = (this.fx < this.tx) ? 1 : -1;
      p.triangle(this.tx - this.x, this.pointHeight/2,
                 this.tx-dir*this.pointWidth - this.x, 0,
                 this.tx-dir*this.pointWidth - this.x, this.pointHeight);
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    p.noFill();
    if (this.highlighted) {
      p.strokeWeight(this.weight+2);
      p.stroke(this.highlightColor);
      shape();
    }
    p.strokeWeight(this.weight);
    p.stroke(0);
    shape();
    p.pop();
  }
}

/*
 * Group(content)
 *  an grouping of its content, a list of BuildingBlock-s
 *  location or dimensions do not play a role
 */
class Group extends BuildingBlock {
  constructor(content) {
    super();
    this.klass = 'Group';
    this.setContent(content); // [buildingblock]
  }

  setContent(content) {
    content.forEach(bb => bb.setParent(this));
    this.content = content;
  }

  refresh() {
    this.content.forEach(c => c.refresh());
    this.refreshMe();
  }
  
  refreshMe() {
    if (this.content.length > 0) {
      let minx = 9999999;
      let maxx = -9999999;
      let miny = 9999999;
      let maxy = -9999999;
      this.content.forEach(c => {
        let bnd = c.bounds();
        minx = Math.min(minx, bnd.x);
        maxx = Math.max(maxx, bnd.x + bnd.width);
        miny = Math.min(miny, bnd.y);
        maxy = Math.max(maxy, bnd.y + bnd.height);
      });

      this.x = minx;
      this.y = miny;
      this.width = maxx - minx;
      this.height = maxy - miny;
    }
  }
  
  show(p) {
    this.content.forEach(c => c.show(p));
  }

  atPos(x, y) {
    // Group is dimensionless, so no restrictions
    return true;
  }
  
  objectsAt(x, y) {
    let result = [];
    this.content.forEach(c => result = result.concat(c.objectsAt(x, y)));
    return result;
  }
}

/*
 * Frame(content)
 *  an invisible frame around its content, a list of BuildingBlock-s
 *  Located at (0, 0) upon construction
 *  Function refreshMe updates the Frame's dimensions.
 */
class Frame extends BuildingBlock {
  constructor(content) {
    super();
    this.klass = 'Frame';
    this.setContent(content); // [buildingblock]
    this.move(0, 0);
  }

  setContent(content) {
    content.forEach(bb => bb.setParent(this));
    this.content = content;
    this.refreshMe();
  }

  refresh() {
    this.content.forEach(c => c.refresh());
    this.refreshMe();
  }
  
  refreshMe() {
    if (this.content.length > 0) {
      let minx = 9999999;
      let maxx = -9999999;
      let miny = 9999999;
      let maxy = -9999999;
      this.content.forEach(c => {
        let bnd = c.bounds();
        minx = Math.min(minx, bnd.x);
        maxx = Math.max(maxx, bnd.x + bnd.width);
        miny = Math.min(miny, bnd.y);
        maxy = Math.max(maxy, bnd.y + bnd.height);
      });

      this.width = maxx - minx;
      this.height = maxy - miny;

      this.content.forEach(c => {
        c.shift(-minx, -miny);
      });      
    }
  }
  
  show(p) {
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    if (DEBUG) {
      p.noFill();
      p.stroke('red');
      p.strokeWeight(2.5);
      p.rect(0, 0, this.width, this.height);
    }
    this.content.forEach(c => c.show(p));
    p.pop();
  }

  objectsAt(x, y) {
    if (this.atPos(x, y)) {
      let result = [];
      this.content.forEach(c => result = result.concat(c.objectsAt(x-this.x, y-this.y)));
      return result.concat([this]);
    } else {
      return [];
    }
  }
}

/*
 * Circle(radius, color)
 *  a circle with default radius 1 (float)
 */
class Circle extends BuildingBlock {
  constructor() {
    super();
    this.klass = 'Circle';
    this.radius = 1; // float
    this.color = 'black'; // p5 color
    this.shadowed = false;
    this.refreshMe();
  }

  refreshMe() {
    this.width = 2 * this.radius;
    this.height = 2 * this.radius;
  }

  show(p) {
    let shape = function() {
      p.circle(0, 0, this.radius*2);
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    p.translate(this.radius, this.radius);
    if (this.shadowed) {
      p.push();
      let eps = 3;
      p.translate(eps, eps);
      p.noStroke();
      p.fill(100,100,100,100);
      shape();
      p.pop();
    }
    p.fill(this.color);
    p.noStroke();
    shape();
    if (this.highlighted) {
      p.strokeWeight(.5);
      p.stroke(this.highlightColor);
      shape();
    }
    p.pop();
  }
}

/*
 * HexagonBox(content)
 *  a hexagon around its content (BuildingBlock). 
 *  Located at (0, 0) upon construction
 */
class HexagonBox extends BuildingBlock {
  constructor(content) {
    super();
    this.klass = 'HexagonBox';
    this.content = content; // buildingblock
    this.content.setParent(this);
    this.side = 1; // float
    this.padding = 1; // int
    this.color = 'black'; // p5 color
    this.strokeColor = 'black';
    this.strokeWeight = .5;
    this.shadowed = true;
    this.refreshMe();
  }

  setContent(content) {
    content.setParent(this);
    this.content = content;
    this.refreshMe();
  }

  refresh() {
    this.content.refresh();
    this.refreshMe();
  }

  refreshMe() {
    let cbnd = this.content.bounds();
    if (cbnd.width/cbnd.height > Math.sqrt(3)) {
      this.side = (cbnd.width + 2*this.padding)/Math.sqrt(3);
    } else {
      this.side = ((cbnd.width + 2*this.padding) * Math.sqrt(3) + (cbnd.height + 2*this.padding) * 3) /  6;
    }
    this.width = this.side * Math.sqrt(3);
    this.height = this.side * 2;
    this.content.move(this.width/2 - cbnd.width/2, this.height/2 - cbnd.height/2);
  }

  show(p) {
    let shape = function() {
      p.strokeCap(p.SQUARE);
      p.beginShape();
      p.vertex(this.side*Math.sqrt(3)/2, 0);
      p.vertex(this.side*Math.sqrt(3), this.side/2);
      p.vertex(this.side*Math.sqrt(3), this.side*3/2);
      p.vertex(this.side*Math.sqrt(3)/2, this.side*2);
      p.vertex(0, this.side*3/2);
      p.vertex(0, this.side/2);
      p.vertex(this.side*Math.sqrt(3)/2, 0);
      p.endShape();
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    if (this.shadowed) {
      p.push();
      let eps = 3;
      p.translate(eps, eps);
      p.noStroke();
      p.fill(100,100,100,100);
      shape();
      p.pop();
    }
    p.fill(this.color);
    p.noStroke();
    shape();
    p.noFill();
    if (this.highlighted) {
      p.strokeWeight(this.strokeWeight+2);
      p.stroke(this.highlightColor);
      shape();
    }
    p.strokeWeight(this.strokeWeight);
    p.stroke(this.strokeColor);
    shape();
    this.content.show(p);
    p.pop();
  }

  objectsAt(x, y) {
    if (this.atPos(x, y)) {
      return  this.content.objectsAt(x-this.x, y-this.y).concat([this]);
    } else {
      return [];
    }
  }
}

/*
 * RoundedBoundingBox(content)
 *  a rectangle with rounded corners around its content (BuildingBlock). 
 *  Located at (0, 0) upon construction
 *  If visible is false, only the content is shown.
 */
class RoundedBoundingBox extends BuildingBlock {
  constructor(content) {
    super();
    this.klass = 'RoundedBoundingBox';
    this.content = content; // buildingblock
    this.content.setParent(this);
    this.padding = 1; // int
    this.visible = true; // boolean
    this.color = 'white'; // p5 color
    this.strokeColor = 'black';
    this.strokeWeight = .5;
    this.round = 1; // in pixels
    this.minWidth = 0;
    this.shadowed = true;
    this.refreshMe();
  }

  setContent(content) {
    content.setParent(this);
    this.content = content;
    this.refreshMe();
  }

  refresh() {
    this.content.refresh();
    this.refreshMe();
  }

  refreshMe() {
    this.content.move(this.padding, this.padding);
    let cbnd = this.content.bounds();
    this.width = Math.max(this.minWidth, cbnd.width) + 2*this.padding;
    this.height = cbnd.height + 2*this.padding;
  }

  show(p) {
    let shape = function() {
      p.rect(0, 0, this.width, this.height, this.round);
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    if (this.visible) {
      if (this.shadowed) {
        p.push();
        let eps = 3;
        p.translate(eps, eps);
        p.noStroke();
        p.fill(100,100,100,100);
        shape();
        p.pop();
      }
      p.fill(this.color);
      p.noStroke();
      shape();
      p.noFill();
      if (this.highlighted) {
        p.strokeWeight(this.strokeWeight+2);
        p.stroke(this.highlightColor);
        shape();
      }
      p.strokeWeight(this.strokeWeight);
      p.stroke(this.strokeColor);
      shape();
    }
//    p.translate(-this.x, -this.y);
    this.content.show(p);
    p.pop();
  }

  objectsAt(x, y) {
    if (this.atPos(x, y)) {
      return  this.content.objectsAt(x-this.x, y-this.y).concat([this]);
    } else {
      return [];
    }
  }
}

/*
 * BoundingBox(content)
 *  a RoundedBoundingBox without rounded corners
 */
class BoundingBox extends RoundedBoundingBox {
  constructor(content) {
    super(content);
    this.klass = 'BoundingBox';
    this.round = 0;
  }

  show(p) {
    let shape = function() {
      p.rect(0, 0, this.width, this.height, this.round);
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    if (this.visible) {
      if (this.shadowed) {
        p.push();
        let eps = 3;
        p.translate(eps, eps);
        p.noStroke();
        p.fill(100,100,100,100);
        shape();
        p.pop();
      }
      p.fill(this.color);
      p.noStroke();
      shape();
      p.noFill();
      if (this.highlighted) {
        p.stroke(this.highlightColor);
        p.strokeWeight(this.strokeWeight+2);
        shape();
      }
      p.stroke(0);
      p.strokeWeight(this.strokeWeight);
      shape();
    }
    this.content.show(p);
    p.pop();
  }
}

/*
 * CorneredBoundingBox(content)
 *  a BoundingBox with some decoration at the corners
 */
class CorneredBoundingBox extends BoundingBox {
  constructor(content) {
    super(content);
    this.klass = 'CorneredBoundingBox';
    this.shadowed = false;
  }

  show(p) {
    super.show(p);
    if (this.visible) {
      let corner = function(x, y) {
        p.push();
        p.noStroke();
        p.fill(0);
        let s = 2;
        p.rect(x-s, y-s, 2*s, 2*s);
        p.pop();
      }.bind(this);
      p.push();
      p.translate(this.x, this.y);
      p.scale(this.scaleX, this.scaleY);
      corner(0, 0);
      corner(0, this.height);
      corner(this.width, 0);
      corner(this.width, this.height);
      p.pop();
    }
  }
}

/*
 * TextBox(text)
 *  a RoundedBoundingBox with as content text (Text)
 */
class TextBox extends RoundedBoundingBox {
  constructor(text) {
    super(text);
    this.klass = 'TextBox';
    this.round = 7;
  }
}

/*
 * Button(text, callback, parent)
 *  a button, represented as a RoundedBoundingBox
 *  with given text, function callback: nil --> nil,
 * and parent: BuildingBlock
 */
class Button extends RoundedBoundingBox {
  constructor(text, callback, manager) {
    super(new Text(text));
    this.klass = 'Button';
    this.padding = 2;
    this.color = '#EEEEEE';
    this.round = 1;
    this.shadowed = false;
    this.text = text;
    this.callback = callback;
    this.manager = manager;
  }

  show(p) {
    super.show(p);
  }
}

/* Table(content) 
 *   a two dimensional table of Text elements.
 */

class Table extends BuildingBlock {
  constructor(content) {
    super();
    this.klass = 'Table';
    this.content = content;
    this.hpad = 8;
    this.vpad = 8;
    this.hsep = 8;
    this.vsep = 8;
    this.centered = false;
    
    this.colWidth = [];
    this.rowHeight = [];
    this.refreshMe();
  }

  refreshMe() {
    if (this.content.length == 0) {
      this.colWidth = [];
      this.rowHeight = [];
    } else {
      this.colWidth = this.content[0].map(elt => 0);
      this.rowHeight = this.content.map(row => 0);
    }

    
    this.content.forEach(row => {
      row.forEach((elt, ci) => {
        this.colWidth[ci] = Math.max(this.colWidth[ci], elt.width);
      });  
    });
    this.content.forEach((row, ri) => {
      let rheight = 0;
      row.forEach(elt => {
        rheight = Math.max(rheight, elt.height);
      });
      this.rowHeight[ri] = rheight;
    });

    let twidth = this.hpad;
    this.colWidth.forEach(cw => { twidth += cw + this.hsep; });
    this.width = twidth - this.hsep + this.hpad;
    
    let theight = this.vpad;
    this.rowHeight.forEach(rh => { theight += rh + this.vsep; });
    this.height = theight - this.vsep + this.vpad;
  }

  show(p) {
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    let py = 0;
    this.content.forEach((row, ri) => {
      if (ri > 0) {
        py += this.vsep/2;
        p.stroke(100);
        p.strokeWeight(.5);
        p.line(this.hpad, py, this.width-this.hpad, py);
        py += this.vsep/2;
      } else {
        py += this.vpad;
      }
      let px = this.hpad;
      row.forEach((elt, ci) => {
        p.push();
        p.translate(px, py);
        if (this.centered) {
          elt.hCenter(this.colWidth[ci]/2);
        }
        elt.show(p);
        p.pop();
        px += this.colWidth[ci] + this.hsep;
      });
      py += this.rowHeight[ri]; 
    });
    p.pop();
  }
}

/* Alert
 *   an alert icon (red with white exclamation mark).
 *   drawn with p5 lines
 */

class Alert extends BuildingBlock {
  constructor() {
    super();
    this.class = 'Alert';
    this.width = 35;
    this.height = 35;
    this.color = '#E74745';
    this.shadowed = true;
  }

  show(p) {
    let shape = function() {
      p.beginShape();
      p.vertex(this.width/2, 0);
      p.vertex(this.width, this.height);
      p.vertex(0, this.height);
      p.endShape(p.CLOSE);
    }.bind(this);
    p.push();
    p.translate(this.x, this.y);
    p.scale(this.scaleX, this.scaleY);
    if (this.shadowed) {
      p.push();
      let eps = 3;
      p.translate(eps*1.5, eps);
      p.noStroke();
      p.fill(100,100,100,100);
      shape();
      p.pop();
    }
    p.fill(this.color);
    p.noStroke();
    shape();
    // exclamation mark:
    p.stroke('#FFFFFF');
    p.strokeWeight(4);
    p.line(this.width/2, 12, this.width/2, this.height-11);
    p.line(this.width/2, this.height-6, this.width/2, this.height-5);
    p.noFill();
    if (this.highlighted) {
      p.strokeWeight(2);
      p.stroke(this.highlightColor);
      shape();
    }
    p.pop();
  }
}
