;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2019,2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2019 Rob Wieringa <Rob.Wieringa@verum.com>
;;; Copyright © 2020 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;; Copyright © 2020 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (daemon Dzn_proxy)
  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)
  #:use-module (srfi srfi-1)
  #:use-module ((srfi srfi-26) #:select (cute))
  #:use-module ((srfi srfi-71))
  #:use-module (dzn runtime)
  #:use-module (dzn decoupler)
  #:use-module (oop goops)
  #:use-module (daemon daemon_interface)
  #:use-module (ide pipe)
  #:use-module (ide util)
  #:use-module (8sync)
  #:use-module (8sync actors)
  #:use-module (transport libenvelope)
  #:export (<daemon:Dzn_proxy> .dz))

(define-class <daemon:Dzn_proxy> (<dzn:component>)
  (dz #:accessor .dz #:init-form (make <daemon:Idzn>) #:init-keyword #:dz))

(define-method (initialize (o <daemon:Dzn_proxy>) args)
  (next-method)
  (set! (.dz o)
    (make <daemon:Idzn>
      #:in (make <daemon:Idzn.in>
        #:name "dz"
        #:self o
        #:command (lambda args (call-in o (lambda _ (apply dzn-command (cons o args))) `(,(.dz o) command))))
      #:out (make <daemon:Idzn.out>))))

(define-actor <pipeline> (<actor>)
  ((close pipeline-close)
   (open pipeline-open)
   (read pipeline-read)
   (write pipeline-write))

  (pids #:accessor .pids #:init-value #f #:init-keyword #:pids)
  (input-port #:accessor .input-port #:init-value #f #:init-keyword #:input-port)
  (output-port #:accessor .output-port #:init-value #f #:init-keyword #:output-port)
  (error-input-port #:accessor .error-input-port #:init-value #f #:init-keyword #:error-input-port)
  (error-output-port #:accessor .error-output-port #:init-value #f #:init-keyword #:error-output-port)
  (on-read #:accessor .on-read #:init-keyword #:on-read))

(define-method (pipeline-open (p <pipeline>) message commands)
  (match (pipe)
    ((error-input-port . error-output-port)
     (set-unbuffered! (set-non-blocking! error-input-port))
     (move->fdes error-output-port 2)
     (let ((output-port input-port pids
                        (with-error-to-port error-output-port
                          (cute pipeline commands))))
       (set! (.error-input-port p) error-input-port)
       (set! (.error-output-port p) error-output-port)
       (set! (.input-port p) (set-unbuffered! (set-non-blocking! input-port)))
       (set! (.output-port p) (set-unbuffered! (set-non-blocking! output-port)))
       (set! (.pids p) pids)))))

(define-method (pipeline-close (p <pipeline>) message)
  (self-destruct p))

(define-method (pipeline-read (p <pipeline>) message)
  ((.on-read p) p))

(define-method (pipeline-write (p <pipeline>) message data)
  (display data (.output-port p))
  (close (.output-port p)))

(define-method (8sync:pipe-command (o <daemon:Dzn_proxy>) name argv input)
  (let* ((hive (dzn:get (.locator o) <hive>))
         (pipeline
          (create-actor
           hive <pipeline>
           #:on-read
           (lambda (pipeline)
             (let* ((stdout (read-string (.input-port pipeline)))
                    (close-stderr-output (close (.error-output-port pipeline)))
                    (stderr (read-string (.error-input-port pipeline)))
                    (status (apply + (map
                                      (compose status:exit-val cdr waitpid)
                                      (.pids pipeline))))
                    (result (libenvelope:wrap
                             name
                             `((stdout . ,stdout)
                               (stderr . ,stderr)
                               (status . ,status)))))
               (action o .dz .out .result result)
               (<- (actor-id pipeline) 'close))))))

    (<- pipeline 'open `(,argv))
    (<- pipeline 'read)
    (<- pipeline 'write input)))

(define-method (blocking:pipe-command (o <daemon:Dzn_proxy>) name argv input)
  (let* ((stdout status stderr (pipeline*->string (list argv) #:input input))
         (result (libenvelope:wrap name `((stdout . ,stdout)
                                          (stderr . ,stderr)
                                          (status . ,status)))))
    (action o .dz .out .result result)))

(define-method (dzn-command (o <daemon:Dzn_proxy>) msg)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Dzn_proxy: and now run ~s\n" msg)
    (let* ((index (libenvelope:label msg))
           (command (libenvelope:data msg))
           (name (assoc-ref command "name"))
           (argv (assoc-ref command "argv"))
           (input (assoc-ref command "input")))
      (log-debug "Dzn_proxy: piping: ~s\n" (and argv (string-join argv)))
      (if (blocking-pipe?) (blocking:pipe-command o name argv input)
          (8sync:pipe-command o name argv input)))))
