;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2020 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands simulate)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)
  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)
  #:use-module (oop goops)
  #:use-module (dzn ast)
  #:use-module (dzn goops)
  #:use-module (dzn parse)
  #:use-module (ide shell-util)
  #:use-module (ide commander)
  #:use-module (ide commands daemon)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide pipe)
  #:use-module (ide util)
  #:use-module (ide command-line)
  #:export (sequence->error
            handle-simulate-result))

(define (parse-opts args)
  (let* ((option-spec
          '((help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))
            (queue-size (single-char #\q) (value #t))
            (strict (single-char #\s))
            (trail (single-char #\t) (value #t))
            (trail-file (single-char #\T) (value #t))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide simulate [OPTION]... [FILE]...
Run simulate and make a trace view available.

  -h, --help             display this help and exit
  -I, --import=DIR+      add directory DIR to import path
  -m, --model=MODEL      explore model MODEL
  -q, --queue-size=SIZE  use queue size=SIZE for simulation [3]
  -s, --strict           use strict matching of trail
  -t, --trail=TRAIL      use named event trail TRAIL
  -T, --trail-file=FILE  read named event trail from FILE

View:
  ~a
" (get-url "trace"))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define* (handle-simulate-result data #:key debug? verbose?)
  (let* ((stdout (assoc-ref data "stdout"))
         (labels (trace:get-info stdout 'labels))
         (eligible (trace:get-info stdout 'eligible))
         (trail (trace:get-trail stdout))
         (stderr (assoc-ref data "stderr"))
         (status (assoc-ref data "status")))
    (when stderr
      (display stderr (current-error-port)))
    (when (or debug? verbose?)
      (format #t "trail: ~a\n" trail)
      (format #t "labels: ~a\n" (string-join labels ","))
      (format #t "eligible: ~a\n" (string-join eligible ",")))
    status))

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (files (option-ref options '() #f))
         (file-name (car files))
         (imports (multi-opt options 'import))
         (input (piped-file->stream file-name #:debug? debug? #:imports imports))
         (model (or (option-ref options 'model #f)
                    (let ((ast (false-if-exception (string->ast input))))
                      (if (not ast) (exit EXIT_OTHER_FAILURE)
                          (and=> (ast:get-model ast) ast:dotted-name)))))
         (result EXIT_SUCCESS))

    (define (simulate-handler message)
      (let ((message (parse-message message)))
        (cond ((simulate? message)
               (let* ((data (simulate-data message))
                      (status (handle-simulate-result data #:debug? debug? #:verbose? verbose?)))
                 (unless (zero? status)
                   (set! result EXIT_FAILURE))
                 (when (isatty? (current-error-port))
                   (format (current-error-port) "See result: ~a\n" (get-url "trace")))))
              (else
               (format (current-error-port)
                       "ide simulate: error: received: ~s\n"
                       message)
               (set! result EXIT_FAILURE)))
        message))

    (unless model
      (format (current-error-port) "no model to simulate\n")
      (exit EXIT_OTHER_FAILURE))

    (daemon:start #:debug? debug? #:port %ide-port)
    (let* ((trail-file (option-ref options 'trail-file #f))
           (trail (and trail-file (with-input-from-file trail-file read-string)))
           (trail (option-ref options 'trail trail))
           (trail (or trail
                      (and (not (isatty? (current-input-port)))
                           (input-port? (current-input-port))
                           (read-string (current-input-port)))
                      ""))
           (trail trail-model (string->trail+model trail))
           (model (or model trail-model))
           (trail (string-join trail ","))
           (trail (if (mingw?) (string-append trail ",") trail))
           (queue-size (option-ref options 'queue-size #f))
           (strict? (option-ref options 'strict #f))
           (command (make-command
                     "simulate"
                     `("dzn" "simulate"
                       ,@(if model `("--model" ,model) '())
                       ,@(if strict? `("--strict") '())
                       ,@(if queue-size `("--queue-size" ,queue-size) '())
                       "--locations"
                       "--state"
                       "--trail" ,trail
                       "-")
                     #:input input
                     #:model model
                     #:queue-size queue-size
                     #:trail trail))
           (commander (make-commander #:debug? debug? #:verbose? verbose?))
           (result-handler
            (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                     simulate-handler))
           (result
            (+ (run-command commander command
                            #:result-handler result-handler
                            #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
               result)))
      (exit result))))
