;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands state)
  #:use-module (ice-9 getopt-long)
  #:use-module (ide commands daemon)
  #:use-module (ide commander)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide util))

(define (state-opts args)
  (let* ((option-spec
          '((help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide state [OPTION]... [FILE]...
Make a state view available.

  -h, --help             display this help and exit
  -I, --import=DIR+      add directory DIR to import path
  -m, --model=MODEL      explore model MODEL

View:
  ~a
" (get-url "state"))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (main args)
  (daemon-config-set!)
  (let* ((options (state-opts args))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (result EXIT_SUCCESS))

    (define (state-handler message)
      (let ((state (parse-message message)))
        (cond ((state? state)
               (let* ((data (state-data state))
                      (status (assoc-ref data "status"))
                      (stderr (assoc-ref data "stderr")))
                 (when stderr
                   (display stderr (current-error-port)))
                 (cond
                  ((zero? status)
                   (when (isatty? (current-error-port))
                     (format (current-error-port) "See result: ~a\n" (get-url "state"))))
                  (else
                   (set! result EXIT_FAILURE)))))
              (else
               (format (current-error-port)
                       "ide state: error: received: ~s\n"
                       message)
               (set! result EXIT_FAILURE)))
        message))

    (daemon:start #:debug? debug? #:port %ide-port)
    (let* ((files (option-ref options '() #f))
           (file-name (car files))
           (imports (multi-opt options 'import))
           (model (option-ref options 'model #f))
           (skip-wfc? (ide:command-line:get 'skip-wfc #f))
           (debug? (ide:command-line:get 'debug))
           (input (piped-file->stream file-name #:debug? debug? #:imports imports))
           (command (make-command
                     "state"
                     `("dzn"
                       ,@(if skip-wfc? '("--skip-wfc") '())
                       "explore"
                       "--state-diagram"
                       "--format=json"
                       ,@(if model `("--model" ,model) '())
                       "-")
                     #:input input
                     #:model model))
           (commander (make-commander #:debug? debug? #:verbose? verbose?))
           (result-handler
            (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                     state-handler))
           (result
            (+ (run-command commander command
                            #:result-handler result-handler
                            #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
               result)))
      (exit result))))
