/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

class StateDiagramP5 extends DiagramP5 {
  
  constructor(parent) {
    super(parent);
    
    /*
     * super defines interface to the outside world, through functions
     *   in.draw(data)
     *   in.dimensions(px, py, width, height)
     *   out.selected(location)
     *
     * extension:
     */
    this.in.reset = function() {
      this.sketch.data = null;
      if (this.sketch.set_up) {
        this.initDiagram(null);
        this.sketch.redraw();
      }
    }.bind(this);
    
    this.phase = {status: 'none', // {'none', 'simple', 'full', 'stable'}
                  stability: 0};
    
    this.simpleDiagram = null;
    this.fullDiagram = null;
    
    this.dataLimit = 4000;
    this.nrNodes = 0;
    this.detail = null;
    
    this.layoutAborted = false;    
    this.diagram = this.Message();

    this.timeOut = 0;
    this.sketch.loop();
  }
  
  /*
   * These abstract methods have to be defined in this class:
   *
   * initDiagram(data) { }
   * draw(p) { }
   * selection(px, py) { }
   * handleKey(p) { }
   * handleMouseClick(p, e) { }
   * dragIt(px, py) { }
   */
     
  initDiagram(data) {
    this.resetEnvironment();
    if (data == null) {
      this.diagram = null;
      this.sketch.noLoop();
    } else {
      this.nrNodes = data.states.length + data.transitions.length;
      if (this.nrNodes > this.dataLimit) {
        let msg = 
            '    more than '+ this.dataLimit + ' nodes   \n' +
            '    cowardly refusing to draw    \n\n';
        this.data = {states:[{id:'0', state:msg}], transitions:[]};
      } else  {
        this.data = data;
      }
      // start timing...
      this.timeOut = this.sketch.millis();
      this.incTimeOut();
      this.simpleDiagram = new StateDiagramSimple(this.data);
      this.simpleDiagram.initViz();
      this.simpleDiagram.refresh();
      this.phase.status = 'simple';
      this.phase.stability = this.stabilize(this.simpleDiagram, 1, this.timeOut);
      
      this.diagram = this.Message();
      this.sketch.loop();
    }
  }

  resetEnvironment() {
    //pre: this.set_up
    this.world.scale = 1;
    this.world.x0 = 0;
    this.world.y0 = 0;
    this.diagram = null;
    this.simpleDiagram = null;
    this.fullDiagram = null;
    this.phase.status = 'none';
    this.drag.dragging = false;
    this.highlightedObj = null;
    this.detail = null;
  }
  
  Message() {
    let message = new RoundedBoundingBox(new Text(''));
    message.color = 'white';
    message.padding = 20;
    message.center(this.windowWidth/2, this.windowHeight/2);
    message.refresh();
    return message;
  }
  
  incTimeOut() {
    let time = this.sketch.millis();
    while (this.timeOut < time) this.timeOut += 1000;
  }

  setMessage(diagram, msg) {
    diagram.content.text = msg;
    diagram.content.refresh();
    diagram.center(this.windowWidth/2, this.windowHeight/2);
    diagram.refresh();
  }
  
  stabilize(diagram, accuracy, timeOut) {
    if (diagram.maxEnergyPerNode == null) diagram.maxEnergyPerNode = diagram.energyPerNode;
    let result = 0;
    let time = this.sketch.millis();
    let iter = 0;
    while (time < timeOut && diagram.energyPerNode > accuracy) {
      let logEN = Math.log(accuracy/diagram.energyPerNode);
      let minEN = Math.min(Math.log(accuracy/diagram.maxEnergyPerNode), logEN);
      result = Math.floor((1 - logEN/minEN) * 100);
      diagram.refresh();
      iter++;
      diagram.maxEnergyPerNode = Math.max(diagram.maxEnergyPerNode, diagram.energyPerNode);
      time = this.sketch.millis();
    }
    if (diagram.energyPerNode <= accuracy) result = 100;
    console.log('at ' + Math.floor(time/1000) + ' sec: iterations: ' + iter + '; stability: ' + result);
    return result;
  }
  
  draw(p) {
    // for svg: use this.world.graphics to generate image i.s.o. p
    let gr = this.world.graphics || p;
    gr.clear();
    gr.background(255);
    gr.push();
    if (this.diagram == null) {
      p.noLoop();
    } else if (this.phase.status == 'none') {
      p.noLoop();
    } else if (this.phase.status == 'simple') {
      gr.push();
      this.world.set(gr);
      
      if (this.phase.stability < 100) {
        this.setMessage(this.diagram,
                   'calculating layout ' + this.nrNodes + ' nodes\n\n' +
                   'PHASE I accuracy ' + this.phase.stability + '%');
        this.diagram.show(gr);
        this.incTimeOut();
        this.phase.stability = this.stabilize(this.simpleDiagram, 1, this.timeOut);
      } else {
        this.fullDiagram = new StateDiagram(this.data);
        this.fullDiagram.initViz();
        this.fullDiagram.copyLayout(this.simpleDiagram);
        this.fullDiagram.refresh();
        console.log('----------- PHASE II -----------');
        this.phase.status = 'full';
        // accuracy = .01 * Math.sqrt(this.nrNodes);
        this.accuracy = .001 * this.nrNodes;
        this.phase.stability = this.stabilize(this.fullDiagram, this.accuracy, this.timeOut);
      }
      gr.pop();
    } else if (this.phase.status == 'full') {
      gr.push();
      this.world.set(gr);
      this.incTimeOut();
      this.phase.stability = this.stabilize(this.fullDiagram, this.accuracy, this.timeOut);
      if (this.phase.stability < 100 && !this.layoutAborted) {
        this.setMessage(this.diagram,
                   'calculating layout ' + this.nrNodes + ' nodes\n\n' +
                   'PHASE I accuracy 100%\n\n' +
                   'PHASE II accuracy ' + this.phase.stability + '%');
        this.diagram.show(gr);
      } else {
        this.diagram = this.fullDiagram;
        this.world.fit(this.diagram.bounds());
        this.phase.status = 'stable';
        this.sketch.cursor(this.sketch.ARROW);
      }
      gr.pop();
    } else {
      gr.push();
      this.world.set(gr);
      if (!this.drag.ctrl && !this.layoutAborted) this.diagram.refresh();
      this.diagram.show(gr);
      if (this.drag.dragging && this.drag.obj && !this.drag.ctrl) {
        // fixate the node position to the mouse point
        let wpt = this.world.mousePoint();
        this.drag.obj.drag(wpt.x-this.drag.offset.x, wpt.y-this.drag.offset.y);
      } else if (this.diagram.energyPerNode < this.accuracy || this.layoutAborted) {
        p.noLoop();
      }
      gr.pop();
      this.showDetails(gr);
    }
    gr.pop();
    if (this.world.graphics)
      p.image(gr, 0, 0);
  }
  
  mouseWheel(p, e) {
    if (!this.mouseInCanvas(p)) return;
    this.world.mouseWheel(e);
    p.redraw();
    return false;
  }

  handleMouseClick(p, e) {
    let wpt = this.world.mousePoint();
    let node = this.selection(wpt.x, wpt.y);
    if (node) {
      this.highlight(node);
      if (node.data.location) this.out.selected({...node.data.location, 'working-directory': this.data['working-directory']});
    } else {
      this.resetHighlight();
    }
    p.redraw();
  }
  
  // override super:
  mouseMoved(p, e) {
    if (!this.mouseInCanvas(p)) return;
    let node = this.nodeAtMouse(p);
    if (node != this.detail) {
      if (this.detail) {
        if (this.detail.isA('Transition')) this.detail.hover(false);
        this.detail = null;
      }
      if (node) {
        this.detail = node;
        if (this.detail.isA('Transition')) this.detail.hover(true);
      }
      p.redraw();
    }
  }
  
  nodeAtMouse(p) {
    if (!this.mouseInCanvas(p)) return null;
    if (!this.diagram) return null;
    let wpt = this.world.mousePoint();
    return this.selection(wpt.x, wpt.y);
  }
  
  selection(px, py) {
    let sel = this.diagram.objectsAt(px, py);
    return this.selectionOfKlass(sel, 'State')
      || this.selectionOfKlass(sel, 'InitialState')
      || this.selectionOfKlass(sel, 'Transition');
  }
  
  handleKey(p) {
    if (p.keyCode == p.ESCAPE) {
      this.layoutAborted = !this.layoutAborted;
      if (!this.layoutAborted) {
        p.loop();
        p.redraw();
      }
    } else if (p.key == 's') {
      if (p.keyIsDown(p.CONTROL)) {
        this.saveAsSvg(p, 'state.svg');
      }
    } else {
      // default behaviour for unbound keys:
      return;
    }
    // suppress default behaviour:
    return false;
  }

  dragIt(px, py) {
    if (this.drag.ctrl) {
      this.diagram.move(px, py);
    } else if (this.drag.obj) {
      this.drag.obj.drag(px, py);
      this.sketch.loop();
    }
  }
  
  showDetails(p) {
    if (!this.mouseInCanvas(p)) return;
    if (this.detail && this.world.scale < .75) {
      p.push();
      //        console.log('show detail %j', this.detail);
      p.translate(-this.detail.x+10, -this.detail.y+10);
      this.detail.show(p);
      p.pop();
    }
  }
}

