/*
 * Copyright (C) 2020,2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

/*
 * Port
 *   a Model's port.
 *   visualisation: downward triangle in a box
 */
class Port extends Viz {
  constructor(data, datamodels, model) {
    super(data); // {name, location, interface}
    this.klass = 'Port';
    this.name = data.name;
    this.location = data.location;
    this.dinterface = datamodels.find(m => m.name == data.interface);
    this.model = model;
  }
  
  initViz() {
    let box = new Box(this.p);
    box.width = 10;
    box.height = 10;
    box.color = '#FFFCB0';
    let pnt = new DTriangle(this.p);
    pnt.width = pnt.height = 6;
    pnt.color = 'black';
    pnt.move(2,2);
    this.setViz(new Frame([box, pnt]));
  }

  highlight(on) {
    this.viz.content[0].highlight(on);
  }

  getInternalBinding() {
    if (this.model.klass == 'System') {
      return this.model.getBinding(this);
    } else {
      return null;
    }
  }
  
  getExternalBinding() {
    let parent = this.model.instance.parentSystem;
    return parent && parent.getBinding(this);
  }
}

/*
 * Ports
 *   either all provides or all requires ports of a Model
 *   functions:
 *     refreshMe: float --> nil
 *       set the object's width, and layout the ports.
 *     layoutPorts: nil --> nil
 *       evenly spread the ports. FIXME: handle large amount of ports.
 */
class Ports extends Viz {
  constructor(ports, datamodels, model) {
    super(ports); // [port]
    this.klass = 'Ports';
    this.model = model;
    this.ports = ports.map(port => new Port(port, datamodels, model));
    this.modelWidth = 1;
  }
  
  initViz() {
    this.ports.forEach(port => port.initViz());
    this.setViz(new Frame(this.ports));
  }
  
  refresh() {
    this.viz.refresh();
    this.refreshMe();
  }

  setWidth(width) {
    this.modelWidth = width;
    this.refreshMe();
  }
  
  refreshMe() {
    this.layoutPorts();
    this.viz.refreshMe();
  }
  
  layoutPorts() {
    let nrports = this.ports.length;
    let spread = this.modelWidth/(nrports+1);
    this.ports.forEach((port, i) => {
      port.move(i*spread, 0);
    });
  }
}

/*
 * Summary
 *   a Model's summary
 *   represented as a BoundingBox containing the given name as Text
 */
class Summary extends Viz {
  constructor(data) {
    super(data); // {name}
    this.klass = 'Summary';
  }

  initViz(color) {
    let name = new Text(this.data.name);
    let box = new BoundingBox(name);
    box.padding = 15;
    box.color = color;
    this.setViz(box);
  }
}

/*
 * ComponentOrForeign
 *   represented as a Frame containing:
 *   . a BoundingBox containing the given name as Text
 *   . the provides Ports at the top (centered)
 *   . the requires ports on the bottom (centered)
 */
class ComponentOrForeign extends Viz {
  constructor(data, datamodels, instance) {
    super(data); // {name, location, provides, requires}
    this.klass = 'ComponentOrForeign';
    this.location = data.location;
    this.instance = instance;
    this.summary = new Summary(data);
    this.provides = new Ports(data.provides, datamodels, this);
    this.requires = new Ports(data.requires, datamodels, this);
    this.bbox;
  }
  
  initViz(color) {
    this.summary.initViz(color);
    let name = new Text(this.data.name);
    this.bbox = new BoundingBox(name);
    this.bbox.padding = 15;
    this.bbox.color = color;
    this.provides.initViz();
    this.requires.initViz();
    this.setViz(new Frame([this.bbox, this.provides, this.requires]));
    this.refreshMe();
  }
  
  refresh() {
    this.viz.refresh();
    this.refreshMe();
  }
  
  refreshMe() {
    let bbnd = this.bbox.bounds();
    this.provides.refreshMe();
    this.provides.setWidth(bbnd.width);
    this.provides.center(bbnd.x+bbnd.width/2, bbnd.y);
    this.requires.refreshMe();
    this.requires.setWidth(bbnd.width);
    this.requires.center(bbnd.x+bbnd.width/2, bbnd.y+bbnd.height);
    this.viz.refreshMe();
  }

  highlight(on) {
    this.viz.content[0].highlight(on);
  }
}

/*
 * Component
 *   represented as a Frame containing:
 *   . a BoundingBox containing the given name as Text
 *   . the provides Ports at the top (centered)
 *   . the requires ports on the bottom (centered)
 */
class Component extends ComponentOrForeign {
  constructor(data, datamodels, instance) {
    super(data, datamodels, instance); // {name, provides, requires}
    this.klass = 'Component';
  }
  
  initViz() {
    super.initViz('#AEE8A0');
  }
}

/*
 * Foreign
 *   represented as a Frame containing:
 *   . a BoundingBox containing the given name as Text
 *   . the provides Ports at the top (centered)
 *   . the requires ports on the bottom (centered)
 */
class Foreign extends ComponentOrForeign {
  constructor(data, datamodels, instance) {
    super(data, datamodels, instance); // {name, provides, requires}
    this.klass = 'Foreign';
  }
  
  initViz() {
    super.initViz('#E5FFE5'); // TODO
  }
}

/*
 * Instance
 *   a model instance is represented as the Model itself
 *   To get the model, a lookup in the list of all models is needed.
 */
class Instance extends Viz {
  constructor(data, datamodels, parent) {
    super(data); // {name, location, model}
    this.klass = 'Instance';
    this.name = data.name;
    this.location = data.location;
    let dmodel = datamodels.find(m => m.name == data.model);
    if (! dmodel) throw('Instance: model of instance ' + data + ' not found');
    this.model = (dmodel.kind == 'component') ? new Component(dmodel, datamodels, this)
      : (dmodel.kind == 'foreign') ? new Foreign(dmodel, datamodels, this)
      : (dmodel.kind == 'system') ? new System(dmodel, datamodels, this)
      : null;
    this.parentSystem = parent;
  }
  
  initViz(light) {
    this.model.initViz(light);
    this.setViz(new Frame([this.model]));
  }

  highlight(on) {
    this.viz.content[0].highlight(on);
  }

  changing() {
    return (this.model.klass == 'System') && this.model.changing();
  }
}

/*
 * Binding
 *   represented as a HRectLine between two Ports
 *   The Ports are found using the encapsulating System
 *   Before calling refresh() on a binding, take care to refresh te
 *   corresponding Ports!
 */
class Binding extends Viz {
  constructor(data, system) {
    super(data); // {from, to, location}
    this.klass = 'Binding';
    this.location = data.location;
    this.system = system;
    this.fromInstance;
    this.fromPort;
    this.toInstance;
    this.toPort;
    
    let from = this.data.from;
    let to = this.data.to;

    // check and switch form/to if necessary
    let port = this.findFromPort(from);
    if (!port) {
      this.fromPort = this.findFromPort(to);
      this.toPort = this.findToPort(from);
    } else {
      this.fromPort = port;
      this.toPort = this.findToPort(to);
    }
    if (! this.fromPort) throw('Binding: port of ' + from + ' not found')
    if (! this.toPort) throw('Binding: port of ' + to + ' not found')
  }

  findFromPort(from) {
    if (from.inst) {
      this.fromInstance = this.system.instances.find(inst => inst.data.name == from.inst);
      if (! this.fromInstance) throw('Binding: from instance of ' + from.inst + ' not found')
      return this.fromInstance.model.requires.ports.find(req => req.data.name == from.port);
    } else {
      return this.system.provides.ports.find(prov => prov.data.name == from.port);
    }
  }
  
  findToPort(to) {
    if (to.inst) {
      this.toInstance = this.system.instances.find(inst => inst.data.name == to.inst);
      if (! this.toInstance) throw('Binding: to instance of ' + to.inst + ' not found')
      return this.toInstance.model.provides.ports.find(prov => prov.data.name == to.port);
    } else {
      return this.system.requires.ports.find(req => req.data.name == to.port);
    }
  }

  initViz() {
    this.setViz(new HRectLine(0, 0, 1, 1));
    this.refresh();
  }
  
  refresh() {  
    let fbnd = this.fromPort.relativeBounds(this.system.viz);
    let tbnd = this.toPort.relativeBounds(this.system.viz);
    this.viz.refreshPoints(fbnd.x+fbnd.width/2, fbnd.y+fbnd.height,
                           tbnd.x+tbnd.width/2, tbnd.y);
  }
  
  show(p) {
    this.refresh();
    this.viz.show(p);
  }
  
}

/*
 * Layer
 *   One layer of instances in a System
 */
class Layer extends Viz {
  constructor(instances) {
    super(instances);
    this.klass = 'Layer';
    this.instances = instances;
  }

  initViz(light) {
    this.instances.forEach(inst => inst.initViz(!light));
    this.setViz(new Frame(this.instances)); 
  }

  refreshMe() {
    let hsep = 20;
    let px = 0;
    this.viz.content.forEach(r => {
      r.move(px, 0);
      let bnd = r.bounds();
      px += bnd.width + hsep;
    });
    this.viz.refreshMe();
  }

}

/*
 * System
 *   a system component, which can be represented open or closed. Open
 *   means all instances are shown. Opening and closing a System is
 *   done with a button. Default is open.
 *   an open System is represented as a Frame containing
 *   . a BoundingBox containing the given name as Text,
 *     and all Instances, subdivided in Layers
 *     Instances are ordered in layers, such that bindings point downward.
 *     Each layer containing Instances with equal 'rank'.
 *   . the provides Ports at the top (centered)
 *   . the requires ports on the bottom (centered)
 *   . a '-' Button to close the System
 *   . all bindings between Ports
 *   a closed System is represented as a Frame containing
 *   . a BoundingBox containing the given name as Text
 *   . the provides Ports at the top (centered)
 *   . the requires ports on the bottom (centered)
 *   . a '+' Button to open the System
 *
 *  functions:
 *    initViz: nil --> nil
 *      (recursively) create all BuildingBlocks.
 *      Both buttons are created, with common callback function
 *      buttonOpenClose();
 *    buttonOpenClose: nil --> nil
 *      forward to function openClose();
 *    openClose: nil --> nil
 *      swap boolean isOpen, and refresh the layout
 *   refreshMe: nil --> nil
 *     re-layout, using the isOpen boolean.
 *     Take care to re-use all created BuildingBlocks, using function
 *     setContent to update.
 *   refreshLayout: frame --> nil
 *     move each layer below the previous one, leaving a gap
 *     center all layers
 **/
class System extends Viz {
  constructor(data, datamodels, instance) {
    super(data); // {name, location, provides, requires, instances, bindings}
    this.klass = 'System';
    this.summary = new Summary(data);
    this.name;
    this.location = data.location;
    this.provides = new Ports(data.provides, datamodels, this);
    this.requires = new Ports(data.requires, datamodels, this);
    this.instances = this.data.instances.map(inst => new Instance(inst, datamodels, this));
    this.bindings = this.data.bindings.map(binding => new Binding(binding, this));
    this.layers = this.makeLayers(this.instances);
    this.instance = instance;
    this.changeSteps = 11;
    this.changeCounter = 0;
    this.isOpen;
    this.openViz;
    this.bindingSep = 3;
  }
  
  initViz(light) {
    let color = light ? '#C9FFC9' : '#C0F7BC'; // '#C3FAC0'; 
    this.summary.initViz(color);
    this.name = new Text(this.data.name);
    this.layers.forEach(layer => layer.initViz(light));

    let column = this.layers;
    // add a last 'empty' layer, used for layouting
    this.instFrame = new Frame(this.layers.concat(new Frame([])));
    
    this.provides.initViz();
    this.requires.initViz();
    this.bindings.forEach(binding => binding.initViz(this));
    this.openButton = new Button(' - ', this.buttonOpenClose, this);
    this.closedButton = new Button(' + ', this.buttonOpenClose, this);

    this.bbox = new BoundingBox(new Frame([this.openButton, this.name, this.instFrame]));
    this.bbox.padding = 15;
    this.bbox.color = color;
    this.frame = new Frame([this.bbox, this.provides, this.requires]
                           .concat(this.bindings));
    
    // CLOSED:
    // this.bbox = new CorneredBoundingBox(new Frame([this.closedButton, this.name]));
    // this.frame = new Frame([this.bbox, this.provides, this.requires]);
    
    this.isOpen = true;
    this.changeCounter = 0;    
    this.setViz(this.frame);
    this.refreshMe();
  }
  
  highlight(on) {
    this.viz.content[0].highlight(on);
  }

  buttonOpenClose(ctrl) {
    // take care: this.isA('Button')
    console.log('PRESSED with CTRL ' + ctrl);
    this.manager.openClose(ctrl)
  }
  
  openClose(ctrl) {
    this.setOpenClose(!this.isOpen, ctrl);
    this.changeCounter = this.changeSteps;
    this.refreshMe();
  }

  setOpenClose(open, ctrl) {
    this.isOpen = open;
    if (ctrl) {
      this.instances.forEach(inst => {
        if (inst.model.klass == 'System') {
          inst.model.setOpenClose(open, ctrl);
        }
      });
    }
  }

  changing() {
    return this.changeCounter != 0 || this.instances.find(inst => inst.changing());
  }
  
  updateViz(frame) {
    this.viz.setContent(frame.content);
  }
  
  refresh() {
    this.viz.refresh();
    this.refreshMe();
  }
  
  refreshMe() {
    let bboxFrame = this.bbox.content;
    let padding = 6;
    let py = padding;
    let showClosed = !this.isOpen; //  && this.changeCounter == 0;
    let button = showClosed ? this.closedButton : this.openButton;
    button.move(0, py);
    let buttonbnd = button.bounds();
    this.name.move(buttonbnd.width + padding, py);

    if (showClosed) {
      bboxFrame.setContent([this.closedButton, this.name]);
    } else {
      let nbnd = this.name.bounds();
      py += nbnd.height + padding;
      this.refreshLayout(this.instFrame);
      this.instFrame.move(0, py);
      bboxFrame.setContent([this.openButton, this.name, this.instFrame]);
    }
    this.bbox.minWidth = this.name.width + this.openButton.width + 10;
    this.bbox.refreshMe();
    let bbnd = this.bbox.bounds();
    this.provides.setWidth(this.bbox.width);
    this.provides.center(bbnd.x+bbnd.width/2, bbnd.y);
    this.requires.setWidth(this.bbox.width);
    this.requires.center(bbnd.x+bbnd.width/2, bbnd.y+bbnd.height);
    
    if (showClosed) {
      this.frame.setContent([this.bbox, this.provides, this.requires]);
    } else {
      this.layoutBindings();
      this.frame.setContent([this.bbox, this.provides, this.requires]
                            .concat(this.bindings));
    }
    this.viz.refreshMe();
    this.viz.scale(1, 1); 
    let bounds = this.bounds();
    if (this.changeCounter == 0) {
      this.saveBounds = bounds;
    } else {
      this.changeCounter--;
      let amt = this.changeCounter/this.changeSteps;
      let width = bounds.width + (this.saveBounds.width-bounds.width)*amt;
      let height = bounds.height + (this.saveBounds.height-bounds.height)*amt;
      let scaleWidth = width / bounds.width;
      let scaleHeight = height / bounds.height;
      this.viz.scale(scaleWidth,scaleHeight); 
    }
  }

  show(p) {
    this.refresh();
    this.viz.show(p);
  }

  makeLayers(instances) {
    let nodes = instances.map(inst => { return {instance: inst, rank: -1, children: []}; });
    nodes.forEach(node => {
      node.children = this.bindings
        .filter(b => (b.fromInstance &&
                      b.fromInstance.data.name == node.instance.data.name &&
                      b.toInstance))
        .map(b => {
          return { node: getnode(b.toInstance.data.name), binding: b };
        });
    });
    
    function getnode(name) {
      return nodes.find(n => n.instance.data.name == name);
    }
    
    function order(node, rank) {
      if (node.rank >= rank) return;
      node.rank = rank;
      node.children.forEach(c => { if (c.node) order(c.node, rank+1); });
    }
    
    nodes.forEach(node => order(node, 0));
    
    function getInstances(rank) {
      return nodes.filter(n => n.rank == rank).map(n => n.instance);
    }
    
    let layers = [];
    for (let rank = 0; rank < nodes.length; rank++) {
      let inst = getInstances(rank);
      if (inst.length > 0) {
        let layer = new Layer(inst);
        layers.push(layer);
      }
      else break;
    }
    return layers;
  }

  refreshLayout(instFrame) {
    
    function maxheight(layer) {
      if (!layer.instances) return 0;
      let result = 0;
      layer.instances.forEach(inst => {
        let bnd = inst.bounds();
        result = Math.max(result, bnd.height);
      });
      return result;
    }

    function maxwidth(column) {
      let result = 0;
      column.forEach(r => {
        let bnd = r.bounds();
        result = Math.max(result, bnd.width);
      });
      return result;
    }

    let vsep = 20;
    let py = 0;
    instFrame.content.forEach(layer => {
      layer.refreshMe();
      layer.move(0, py);
      py += maxheight(layer) + vsep + this.bindingHeight(layer)*this.bindingSep;
    });
    // adapt location of last 'empty' layer:
    let last = instFrame.content[instFrame.content.length-1];
    last.shift(0, -15); // -15 being the padding of the system's bounding box
    let width = maxwidth(instFrame.content);
    instFrame.content.forEach(layer => layer.hCenter(width/2));
    instFrame.refreshMe();
  }

  bindingHeight(layer) {
    if (!layer.instances) return 0;
    
    let h = 0;
    let layerBindings = this.bindings.filter(bind => {
      return bind.fromInstance && layer.instances.find(c => c == bind.fromInstance);
    });
    let hBinds = layerBindings.map(bind => {
      return {bind: bind, height: 0};
    });
    let maxHeight = 0;
    for (let i1 = 0; i1 < hBinds.length; i1++) {
      let h1 = hBinds[i1].height;
      for (let i2 = 0; i2 < i1; i2++) {
        if (this.overlapping(hBinds[i1].bind, hBinds[i2].bind)) {
          h1 = Math.max(h1, hBinds[i2].height + 1);
        }
      }
      hBinds[i1].height = h1;
      maxHeight = Math.max(maxHeight, h1);
    }
    return maxHeight;
  }
    
  overlapping(binding1, binding2) {
    let f1y = binding1.y + binding1.viz.fy;
    let f2y = binding2.y + binding2.viz.fy;
    if (f1y != f2y) return false;
    
    let f1x = binding1.x + binding1.viz.fx;
    let t1x = binding1.x + binding1.viz.tx;
    let f2x = binding2.x + binding2.viz.fx;
    let t2x = binding2.x + binding2.viz.tx;
    return Math.max(f1x, t1x) >= Math.min(f2x, t2x) && Math.max(f2x, t2x) >= Math.min(f1x, t1x)
  }
  
  layoutBindings() {
    // avoid binding overlaps
    // first reset:
    this.bindings.forEach(bind => {
      bind.viz.yMiddle = 10;
      if (bind.fromInstance) {
        let fibnd = bind.fromInstance.bounds();
        let fipbnd = bind.fromInstance.parent.bounds(); // current layer
        bind.viz.yMiddle += fipbnd.height - fibnd.height;
      }
    });
    // now check for overlap:
    for (let i1 = 0; i1 < this.bindings.length; i1++) {
      let mid1 = this.bindings[i1].viz.yMiddle;
      for (let i2 = 0; i2 < i1; i2++) {
        if (this.overlapping(this.bindings[i1], this.bindings[i2])) {
          mid1 = Math.max(mid1, this.bindings[i2].viz.yMiddle + this.bindingSep);
        }
      }
      this.bindings[i1].viz.yMiddle = mid1;
    }
  }

  // navigation:
  layerIndex(instance) {
    return this.layers.findIndex(layer => {
      let inst = layer.instances.find(inst => inst == instance);
      return inst;
    });
  }
  
  nextInstance(instance) {
    let layerIndex = this.layerIndex(instance);
    let layer = this.layers[layerIndex];
    let i = layer.instances.findIndex(inst => inst == instance);
    if (i < layer.instances.length-1) return layer.instances[i+1];
    else if (layerIndex < this.layers.length-1) {
      let layer1 = this.layers[layerIndex+1];
      return layer1.instances[0];
    } else return null;
  }
  previousInstance(instance) {
    let layerIndex = this.layerIndex(instance);
    let layer = this.layers[layerIndex];
    let i = layer.instances.findIndex(inst => inst == instance);
    if (i > 0) return layer.instances[i-1];
    else if (layerIndex > 0) {
      let layer1 = this.layers[layerIndex-1];
      return layer1.instances[layer1.instances.length-1];
    }
    else return null;
  }
  nextLayer(instance){
    let layerIndex = this.layerIndex(instance);
    if (layerIndex < this.layers.length-1) return this.layers[layerIndex+1];
    else return null;
  }
  previousLayer(instance) {
    let layerIndex = this.layerIndex(instance);
    if (layerIndex > 0) return this.layers[layerIndex-1];
    else return null;
  }

  getBinding(port) {
    return this.bindings.find(bind => bind.fromPort == port || bind.toPort == port);
  }
}

/*
 * SUT
 *   System Under Test
 *   Identified by name, so a lookup in the list of all models is required
 *   Mainly an instance of the corresponding Model without parent System
 */
class SUT extends Viz {
  constructor(data) {
    super(data); // {sut, models}
    this.klass = 'SUT';
    let dmodel = data.models.find(m => m.name == data.sut);
    let location = dmodel && dmodel.location;
    this.instance = new Instance({name: data.sut, model: data.sut, location: location},
                                 data.models, null);
  }
  
  initViz() {
    this.instance.initViz(true);
    this.setViz(new Frame([this.instance]));
  }

  highlight(on) {
    this.viz.content[0].highlight(on);
  }

  changing() {
    return this.instance.changing();
    
  }
}
