/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */

class SystemDiagramP5 extends DiagramP5 {
  
  constructor(parent) {
    super(parent);
    /*
     * super defines interface to the outside world, through functions
     *   in.draw(data)
     *   in.dimensions(px, py, width, height)
     *   out.selected(location)
     *
     * extension: none
     */
    
    this.tooltipObj = null;
    this.tooltip = null;
    this.highlightedExtras = [];
    
    this.nrFocusFrames = 10;
    this.focusShift = {x:0, y:0};
    this.focusFrame = this.nrFocusFrames;
    
    this.diagramStack = []; // [Instance]
  }  
  
  /*
   * These abstrace methods have to be defined in this class:
   *
   * initDiagram(data) { }
   * draw(p) { }
   * selection(px, py) { }
   * handleKey(p) { }
   * handleMouseClick(p, e) { }
   * dragIt(px, py) { }
   */

  initDiagram(data) {
    this.data = data;
    this.diagram = new SUT(data).instance;
    this.diagram.initViz(false);
    this.diagram.move(100, 100);
    this.diagramStack.push(this.diagram);
    this.sketch.cursor(this.sketch.ARROW);
    this.sketch.noLoop();
  }
  
  draw(p) {
    // for svg: use this.world.graphics to generate image i.s.o. p
    let gr = this.world.graphics || p;
    gr.clear();
    gr.background(255);
    if (!this.diagram) return;
    gr.push();
    this.world.set(gr);
    this.diagram.refresh();
    this.diagram.show(gr);
    if (this.diagram.changing() || this.focusAnimating()) p.loop(); else p.noLoop();
    gr.pop();
    this.showDetails(gr);
    this.showTooltip(gr, p.mouseX, p.mouseY);
    if (this.world.graphics)
      p.image(gr, 0, 0);
  }
  
  mouseMoved(p, e) {
    if (!this.mouseInCanvas(p)) return;
    let obj = this.objAtMouse(p);
    if (obj && obj.isA('Port'))
      this.setTooltip(obj);
    else
      this.setTooltip(null);
  }
  
  setTooltip(obj) {
    let old = this.tooltipObj;
    this.tooltipObj = obj;
    if (this.tooltipObj && this.tooltipObj != old) {
      if (this.tooltipObj.isA('Port')) {
        let text = this.tooltipObj.name;
        this.tooltip = new BoundingBox(new Text(text));
        this.tooltip.padding = 3;
        this.tooltip.color = '#FFFCB0';
        this.tooltip.refreshMe();
      }
    }
    if (this.tooltipObj || old)
      this.sketch.redraw();
  }
  
  highlight(obj) {
    super.highlight(obj);
    this.resetHighlightExtras();
  }
  
  resetHighlight() {
    super.resetHighlight();
    this.resetHighlightExtras();
    this.sketch.redraw();
  }
  
  highlightExtras(lst) {
    this.highlightedExtras = lst;
    this.highlightedExtras.forEach(h => h.highlight(true));
    this.sketch.redraw();
  }
  
  resetHighlightExtras() {
    this.highlightedExtras.forEach(h => h.highlight(false));
    this.highlightedExtras = [];
  }
  
  focus(obj) {
    let sbnd = this.diagram.bounds();
    let obnd = obj.relativeBounds(this.diagram.viz);
    // absolute coordinates:
    let oldX = (obj == this.diagram) ? sbnd.x : sbnd.x+obnd.x;
    let oldY = (obj == this.diagram) ? sbnd.y : sbnd.y+obnd.y;
    let newX = oldX;
    let newY = oldY;
    let topLeft = this.world.canvasToWorld(0, 0);
    let bottomRight = this.world.canvasToWorld(this.world.canvas.width, this.world.canvas.height);
    if (oldX < topLeft.x || oldX + obnd.width > bottomRight.x) newX = 100/this.world.scale;
    if (oldY < topLeft.y || oldY + obnd.height > bottomRight.y) newY = 100/this.world.scale;
    
    if (newX != oldX || newY != oldY) {
      this.startFocusAnimation(newX - oldX, newY - oldY);
    }
  }
  
  startFocusAnimation(dx, dy) {
    this.focusShift = {x: 1/this.nrFocusFrames*dx, y: 1/this.nrFocusFrames*dy};
    this.diagram.shift(this.focusShift.x, this.focusShift.y);
    this.focusFrame = 1;
    this.sketch.redraw();
  }
  
  focusAnimating() {
    if (this.focusFrame == this.nrFocusFrames) return false;
    this.diagram.shift(this.focusShift.x, this.focusShift.y);
    this.focusFrame++;
    return true;
  }
  
  highlightAndFocus(obj) {
    this.highlight(obj);
    this.focus(obj);
  }
  
  handleMouseClick(p, e) {
    let obj = this.objAtMouse(p);
    if (obj) {
      if (obj.klass == 'Button') {
        obj.callback(e.ctrlKey);
        p.redraw();
      } else {
        this.highlight(obj);
        if (obj.klass == 'Port') {
          this.highlightExtras(this.path(obj));
          p.redraw();
        }
        let location = obj.location;
        if (p.keyIsDown(p.CONTROL)) {
          if (obj.klass == 'Instance')
            location = obj.model && obj.model.location;
          else if (obj.klass == 'Port')
            location = obj.dinterface.location;
          else if (obj.klass == 'Binding')
            location = obj.fromPort.dinterface.location;
        }
        if (location) this.out.selected({...location, 'working-directory': this.data['working-directory']});
      }
    } else {
      this.resetHighlight();
    }
  }
  
  objAtMouse(p) {
    if (!this.mouseInCanvas(p)) return null;
    if (!this.diagram) return null;
    let wpt = this.world.mousePoint();
    return this.selection(wpt.x, wpt.y);
  }

  selection(px, py) {
    let sel = this.diagram.objectsAt(px, py);
    return this.selectionOfKlass(sel, 'Button') ||
      this.selectionOfKlass(sel, 'Binding') ||
      this.selectionOfKlass(sel, 'Port') ||
      this.selectionOfKlass(sel, 'Instance') ||
      this.selectionOfKlass(sel, 'Component') ||
      this.selectionOfKlass(sel, 'Foreign') ||
      this.selectionOfKlass(sel, 'System');
  }
  
  path(port) {
    function follow(port, bnd) {
      let port2 = (bnd.fromPort == port) ? bnd.toPort : bnd.fromPort;
      let result = [port2];
      if (port2.model.klass == 'System') {
        let bnd2 = (port2.model == bnd.system) ? port2.getExternalBinding() : port2.getInternalBinding();
        if (bnd2) {
          result.push(bnd2);
          result = result.concat(follow(port2, bnd2));
        }
      }
      return result;
    }
    
    let result = [port];
    let ibnd = port.getInternalBinding();
    let ebnd = port.getExternalBinding();
    if (ibnd) {
      result.push(ibnd);
      result = result.concat(follow(port, ibnd));
    }
    if (ebnd) {
      result.push(ebnd);
      result = result.concat(follow(port, ebnd));
    }
    return result;
  }
  
  handleKey(p) {
    if (p.keyCode == p.ENTER) {
      this.enterInstance();
    } else if (p.keyCode == p.BACKSPACE) {
      this.exitInstance();
    } else if (p.keyCode == p.DOWN_ARROW ||
               p.keyCode == p.UP_ARROW ||
               p.keyCode == p.RIGHT_ARROW ||
               p.keyCode == p.LEFT_ARROW) {
      this.navigateInstance(p, p.keyCode);
    } else if (p.key === '-') {
      if (p.keyIsDown(p.CONTROL)) {
        this.world.zoomAround(0, 0, this.world.zoomOutFactor);
        p.redraw();
      } else {
        this.closeSystem();
      }
    } else if (p.key === '+' || p.key == '=') {
      if (p.keyIsDown(p.CONTROL)) {
        this.world.zoomAround(0, 0, this.world.zoomInFactor);
        p.redraw();
      } else {
        this.openSystem();
      }
    } else if (p.key === '0') {
      if (p.keyIsDown(this.CONTROL)) {
        this.world.fit(this.diagram.bounds());
        p.redraw();
      } 
    } else if (p.key === '1') {
      if (p.keyIsDown(p.CONTROL)) {
        this.world.scale = 1;
        this.world.zoomAround(0, 0, 1);
        p.redraw();
      } 
    } else if (p.key == 's') {
      if (p.keyIsDown(p.CONTROL)) {
        this.saveAsSvg(p, 'system.svg');
      }
    } else {
      // default behaviour for unbound keys:
      return;
    }
    // suppress default behaviour:
    return false;
  }
  
  getHighlightedInstance() {
    if (this.highlightedObj && this.highlightedObj.isA('Instance')) return this.highlightedObj;
    else return null;
  }
  
  getHighlightedModel() {
    if (!this.highlightedObj) return null;
    else if (this.highlightedObj.isA('Instance')) return this.highlightedObj.model;
    else if (this.highlightedObj.isA('Component') ||
             this.highlightedObj.isA('Foreign') ||
             this.highlightedObj.isA('System')) return this.highlightedObj;
    else return null;
  }
  
  getHighlightedSystem() {
    let model = this.getHighlightedModel();
    if (model && model.isA('System')) return model;
    else return null;
  }
  
  openSystem() {
    let system = this.getHighlightedSystem();
    if (system && !system.isOpen) {
      system.openClose();
      this.sketch.redraw();
    }
  }
  
  closeSystem() {
    let system = this.getHighlightedSystem();
    if (system && system.isOpen) {
      system.openClose();
      this.sketch.redraw();
    }
  }
  
  enterInstance() {
    let inst = this.getHighlightedInstance();
    if (inst) {
      this.diagram = inst;
      this.diagramStack.push(this.diagram);
      this.highlightAndFocus(this.diagram);
    }
  }
  
  exitInstance() {
    if (this.diagramStack.length > 1) {
      this.diagramStack.pop();
      this.diagram = this.diagramStack[this.diagramStack.length-1];
      this.highlightAndFocus(this.diagram);
    }
  }
  
  navigateInstance(p, keyCode) {
    let inst = this.getHighlightedInstance();
    if (inst) {
      this.navigate(inst, p, keyCode)
    } else if (!this.highlightedObj) {
      this.highlightAndFocus(this.diagram);
    }
  }
  
  navigate(instance, p, keyCode) {
    if (p.keyIsDown(p.CONTROL)) {
      // navigate inward or outward
      if (keyCode == p.DOWN_ARROW) {
        if (instance.model.isA('System') && instance.model.isOpen) {
          let layer = instance.model.layers[0];
          if (layer) this.highlightAndFocus(layer.instances[0]);
        }
      } else if (keyCode == p.UP_ARROW) {
        // do not navigate outside diagram:
        if (instance != this.diagram) this.highlightAndFocus(instance.parentSystem.instance);
      }
    } else {
      // navigate in current system
      // do not navigate outside diagram:
      if (instance == this.diagram) return;
      let system = instance.parentSystem;
      if (keyCode == p.DOWN_ARROW) {
        let layer = system.nextLayer(instance);
        if (layer) this.highlightAndFocus(layer.instances[0]);
      } else if (keyCode == p.UP_ARROW) {
        let layer = system.previousLayer(instance);
        if (layer) this.highlightAndFocus(layer.instances[0]);
      } else if (keyCode == p.RIGHT_ARROW) {
        let sel = system.nextInstance(instance);
        if (sel) this.highlightAndFocus(sel);
      } else if (keyCode == p.LEFT_ARROW) {
        let sel = system.previousInstance(instance);
        if (sel) this.highlightAndFocus(sel);
      }
    }
  }

  dragIt(px, py) {
    if (this.drag.ctrl) {
      // drag whole diagram....
      this.diagram.move(px, py);
    }
  }
  
  showDetails(p) {
    if (this.highlightedObj && this.world.scale < .75) {
      let model = this.highlightedObj.isA('Instance') ? this.highlightedObj.model : this.highlightedObj;
      let summary = model.summary;
      p.push();
      p.translate(-summary.x+10, -summary.y+10);
      summary.show(p);
      p.pop();
    }
  }
  
  showTooltip(p, mx, my) {
    if (this.tooltipObj) {
      p.push();
      p.translate(mx+20, my);
      this.tooltip.show(p);
      p.pop();
    }
  }
}
