/*
 * Copyright (C) 2021 Rob Wieringa <rma.wieringa@gmail.com>
 *
 * This file is part of Dezyne-P5.
 * Dezyne-P5 offers Dezyne web views based on p5.js
 *
 * Dezyne-P5 is free software, it is distributed under the terms of
 * the GNU General Public Licence version 3 or later.
 * See <http://www.gnu.org/licenses/>.
 */


/*
 * Abstract class for drawing a diagram
 *
 * Visualisation is done using standard functions defined in p5js
 * Zooming and dragging are facilitated. The World object handles the
 * appropriate transformations.
 */

class DiagramP5 {

  /*
   * These abstract methods have to be defined by any subclass
   */
  initDiagram(data) { }
  draw(p) { }
  selection(px, py) { }
  handleKey(p) { }
  handleMouseClick(p, e) { }
  dragIt(px, py) { }

  constructor(parent) {

    /*
     * interface to the outside world, through functions
     *   in.draw(data)
     *     show the data as a diagram
     *   in.dimensions(px, py, width, height)
     *     set the location (px, py) and size (with, height) of the
     *     diagram
     *   out.selected(location)
     *     process the selected node, using its location
     *
     *   out functions can be (re)defined by the outside world.
     *   sub classes can extend this interface
     */
    this.in = {
      draw: function (data) {
        this.data = data;
        if (this.set_up) {
          this.initDiagram(data);
          this.sketch.redraw();
        }
      }.bind(this),
      dimensions: function (px, py, width, height) {
        this.dimensions(px, py, width, height);
      }.bind(this),
    };
    this.out = {
      selected: function (location) {
        console.log('selected location: %j', location);
      }
    };

    // default dimensions: whole window
    this.px = 0;
    this.py = 0;
    this.width = window.innerWidth;
    this.height = window.innerHeight;
    this.fixedSize = false;

    this.set_up = false;
    this.diagram = null;
    this.world = null;

    // dragging:
    this.drag = {dragging: false,
                 ctrl: false,
                 obj: null,
                 offset: {x: 0, y: 0} // dragging startpoint
                };

    // highlighting:
    this.highlightedObj = null;

    // timeout for repeating key handling:
    this.timeout = {
      delay: 50, // const
      handler: null,
      start: 0,
      wait: 0,
      running: false
    }

    /*
     * interface with sketch:
     * P5 event handling
     * All events are implemented by the 'this' DiagramP5 object
     */
    let diagramSketch = function(p) {
      p.setup = function() {
        this.setup(p);
      }.bind(this);
      p.draw = function() {
        this.draw(p);
      }.bind(this);
      p.windowResized = function() {
        this.windowResized(p);
      }.bind(this);
      p.keyPressed = function() {
        return this.keyPressed(p);
      }.bind(this);
      p.keyReleased = function() {
        return this.keyReleased(p);
      }.bind(this);
      p.mouseMoved = function(e) {
        return this.mouseMoved(p, e);
      }.bind(this);
      p.mousePressed = function(e) {
        return this.mousePressed(p, e);
      }.bind(this);
      p.mouseDragged = function(e) {
        return this.mouseDragged(p, e);
      }.bind(this);
      p.mouseReleased = function(e) {
        return this.mouseReleased(p, e);
      }.bind(this);
      p.mouseWheel = function(e) {
        return this.mouseWheel(p, e);
      }.bind(this);
    }.bind(this);

    this.sketch = new p5(diagramSketch, parent);
  }


  dimensions(px, py, width, height) {
    this.px = px;
    this.py = py;
    this.width = width;
    this.height = height;
    this.fixedSize = true;
    if (this.set_up) {
      this.world.positionCanvas(px, py);
      this.world.resizeCanvas(width, height);
      this.sketch.redraw();
    }
  }

  setup(p) {
    let px = this.px;
    let py = this.py;
    this.width = this.fixedSize ? this.width : p.windowWidth;
    this.height = this.fixedSize ? this.height : p.windowHeight;
    this.world = new World(p, this.width, this.height);
    if (px != null && py != null) this.world.positionCanvas(px, py);
    this.set_up = true;
    p.cursor(p.WAIT);
    if (this.data) {
      this.initDiagram(this.data);
    } else {
      p.noLoop();
    }
  }

  mouseInCanvas(p) {
    if (!this.fixedSize) return true; // canvas spans whole window
    if (!(0 <= p.mouseX && p.mouseX <= this.width)) return false;
    if (!(0 <= p.mouseY && p.mouseY <= this.height)) return false;
    return true;
  }

  windowResized(p) {
    if (!this.fixedSize) {
      this.width = p.windowWidth;
      this.height = p.windowHeight;
      this.world.resizeCanvas(this.width, this.height);
    }
  }

  selectionOfKlass(sel, klass) {
    let obj = sel.find(s => s.isA(klass));
    return obj;
  }

  highlight(obj) {
    if (this.highlightedObj) {
      this.highlightedObj.highlight(false);
    }
    this.highlightedObj = obj;
    this.highlightedObj.highlight(true);
    this.sketch.redraw();
  }

  resetHighlight() {
    if (this.highlightedObj) {
      this.highlightedObj.highlight(false);
      this.highlightedObj = null;
    }
  }

  mouseMoved(p, e) {
    // no default behaviour
  }

  mousePressed(p, e) {
    if (!this.mouseInCanvas(p)) return;
    if (!this.diagram) return;
    let wpt = this.world.mousePoint();
    this.drag.ctrl = e.ctrlKey;
    this.drag.obj = null;
    if (e.ctrlKey) {
      this.drag.obj = this.diagram;
      this.drag.offset.x = wpt.x - this.diagram.x;
      this.drag.offset.y = wpt.y - this.diagram.y;
    } else {
      let obj = this.selection(wpt.x, wpt.y);
      if (obj) {
        // do not drag buttons
        if (obj.klass == 'Button') {
          this.buttonSelected = obj;
          obj.highlight(true);
          p.redraw();
        } else {
          this.drag.obj = obj;
          this.drag.offset.x = wpt.x - obj.x;
          this.drag.offset.y = wpt.y - obj.y;
        }
      }
    }
    p.redraw();
  }

  mouseDragged(p, e) {
    if (!this.mouseInCanvas(p)) return;
    if (!this.diagram) return;
    this.drag.dragging = true;
    let wpt = this.world.mousePoint();
    this.dragIt(wpt.x-this.drag.offset.x, wpt.y-this.drag.offset.y);
    p.redraw();
  }

  mouseReleased(p, e) {
    if (this.drag.dragging) {
      this.drag.dragging = false;
    }
    let buttonSelected = this.buttonSelected;
    if (buttonSelected) {
      buttonSelected.highlight(false);
      p.redraw();
    }
    this.buttonSelected = null;
    if (!this.mouseInCanvas(p)) return;
    if (!this.diagram) return;
    this.handleMouseClick(p, e);
  }

  keyPressed(p) {
    if (!this.mouseInCanvas(p)) return;
    if (!this.diagram) return;
    this.startTimeout();
    return this.handleKey(p);
  }

  keyReleased(p) {
    if (!this.mouseInCanvas(p)) return;
    this.stopTimeout();
    return false;
  }

  startTimeout() {
    this.timeout.running = true;
    this.timeout.wait = this.timeout.delay * 15; // initial delay is larger
    this.timeout.start = Date.now();
  }

  stopTimeout() {
    this.timeout.running = false;
    if (this.timeout.handler != null) {
      clearTimeout(this.timeout.handler);
      this.timeout.handler = null;
    }
  }

  // might be called in this.draw(p):
  checkRepeatKey(p) {
    if (this.timeout.running && this.timeout.handler == null) {
      let now = Date.now();
      let delta = now - this.timeout.start;
      let rerun = function() {
        // restart timer:
        this.timeout.start = Date.now();
        this.timeout.wait = this.timeout.delay;
        this.timeout.handler = null;
        this.handleKey(p);
      }.bind(this);
      this.timeout.handler = setTimeout(rerun, this.timeout.wait - delta);
    }
  }

  mouseWheel(p, e) {
    if (!this.mouseInCanvas(p)) return;
    if (!this.diagram) return;
    this.world.mouseWheel(e);
    p.redraw();
    return false;
  }

  saveAsSvg(p, title) {
    if (!this.diagram) return;
    let bnd = this.diagram.bounds();
    let padding = 10;
    // store state:
    let worldSave = this.world.getState();
    // svg graphics
    let graphics = p.createGraphics(bnd.width+2*padding, bnd.height+2*padding, p.SVG);
    this.world.setState({
      scale: 1,
      x0: padding - bnd.x,
      y0: padding - bnd.y,
      canvas: this.world.canvas,
      graphics: graphics});
    p.draw();
    p.save(graphics, title);
    // restore state:
    this.world.setState(worldSave);
  }
}
