;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2019,2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2019,2020 Rob Wieringa <Rob.Wieringa@verum.com>
;;; Copyright © 2019,2020 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;; Copyright © 2020,2021 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (daemon Data)
  #:use-module (oop goops)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 match)
  #:use-module (8sync)
  #:use-module (dzn runtime)
  #:use-module (dzn decoupler)
  #:use-module (ide util)
  #:use-module (ide shell-util)
  #:use-module (transport transport_interface)
  #:use-module (transport libtransport)
  #:use-module (transport libenvelope)
  #:use-module (daemon daemon_interface)
  #:duplicates (merge-generics)
  #:export (<daemon:Data>)
  #:re-export (.data))

(define-class <daemon:Data> (<dzn:component>)
  (data #:accessor .data #:init-form (make <daemon:Idata>) #:init-keyword #:data)

  (hello-command #:accessor .hello-command #:init-value #f)
  (parse-command #:accessor .parse-command #:init-value #f)
  (simulate-command #:accessor .simulate-command #:init-value #f)
  (state-command #:accessor .state-command #:init-value #f)
  (verify-command #:accessor .verify-command #:init-value #f)

  (request-data #:accessor .request-data #:init-value (list))
  (selection-data #:accessor .selection-data #:init-value #f)

  (hello-data #:accessor .hello-data #:init-value #f)
  (parse-data #:accessor .parse-data #:init-value #f)
  (verify-data #:accessor .verify-data #:init-value #f)
  (simulate-data #:accessor .simulate-data #:init-value #f)
  (trace-data #:accessor .trace-data #:init-value #f)
  (state-data #:accessor .state-data #:init-value #f))

(define-method (initialize (o <daemon:Data>) args)
  (next-method)
  (set! (.data o)
        (make <daemon:Idata>
          #:in (make <daemon:Idata.in>
                 #:name "data"
                 #:self o
                 #:store_command (lambda args (call-in o (lambda _ (apply data-store_command (cons o args))) `(,(.data o) store_command)))
                 #:store_result (lambda args (call-in o (lambda _ (apply data-store_result (cons o args))) `(,(.data o) store_result)))
                 #:request (lambda args (call-in o (lambda _ (apply data-request (cons o args))) `(,(.data o) request)))
                 #:get_selection (lambda args (call-in o (lambda _ (apply data-get_selection (cons o args))) `(,(.data o) get_selection)))
                 #:get_location (lambda args (call-in o (lambda _ (apply data-get_location (cons o args))) `(,(.data o) get_location)))
                 #:get_command_simulate (lambda args (call-in o (lambda _ (apply data-get_command_simulate (cons o args))) `(,(.data o) get_command_simulate)))
                 #:get_command_simulate_back (lambda args (call-in o (lambda _ (apply data-get_command_simulate_back (cons o args))) `(,(.data o) get_command_simulate_back))))
          #:out (make <daemon:Idata.out>))))


(define (json-get json key) ;; REMOVEME
  (let ((key (if (string? key) (string->symbol key) key)))
    (or (assq-ref json key)
        (assoc-ref json (symbol->string key)))))

(define-method (data-store_command (o <daemon:Data>) command)
  (let ((log (dzn:get (.locator o) <procedure> 'log))
        (log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Data: store_command: command: ~s\n" command)
    (let* ((command-data (libenvelope:data command))
           (type (assoc-ref command-data "name")))
      (log-debug "Data: store_command: type: ~s\n" type)
      (log "command: ~a\n" type)
      (case (string->symbol type)
        ((bye) (set! (.hello-command o) command-data))
        ((hello) (set! (.hello-command o) command-data))
        ((parse) (set! (.parse-command o) command-data))
        ((simulate) (set! (.simulate-command o) command-data))
        ((state) (set! (.state-command o) command-data))
        ((verify) (set! (.verify-command o) command-data))
        (else (log-error "Data: store_command: no such command: ~s\n" type))))))

(define-method (data-store_result (o <daemon:Data>) data out-stored)
  (define (symbol< a b)
    (string< (symbol->string a) (symbol->string b)))
  (let ((log (dzn:get (.locator o) <procedure> 'log))
        (log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Data: store_result: data: ~s\n" data)
    (let* ((label (libenvelope:label data))
           (data (libenvelope:data data))
           (stdout (assoc-ref data 'stdout))
           (stderr (assoc-ref data 'stderr))
           (connections (.request-data o))
           (connections (sort connections symbol<))
           (data (case label
                   ((hello)
                    `(,@data
                      (connections ,@connections)))
                   ((parse)
                    `((stdout . "") ; truncate for performance
                      ,@(filter (compose not (cute eq? <> 'stdout) car) data)))
                   (else
                    data))))
      (case label
        ((bye) (set! (.hello-data o) stdout))
        ((hello) (set! (.hello-data o) stdout))
        ((parse) (set! (.parse-data o) stdout))
        ((verify) (set! (.verify-data o) stdout))
        ((simulate) (set! (.simulate-data o) stdout))
        ((state) (set! (.state-data o) stdout))
        (else (log-error "Data: store_result: no such label: ~s\n" label)))
      (set! (.request-data o) (list))
      (when (eq? label 'simulate)
        (let ((trace-data
               (catch #t
                 (cute trace:trace->json (string-append stdout stderr))
                 (const #f)
                 (lambda (key . args)
                   (log-debug "Data: store_result: trace->json: ~a,~s\n" key args)
                   (display-backtrace (make-stack #t) (current-error-port))))))
          (log-debug "Data: store_result: trace-data: ~s\n" trace-data)
          (set! (.trace-data o) trace-data)))
      (let ((envelope (libenvelope:wrap label data)))
        (set-cdr! out-stored envelope)))))

(define-method (data-request (o <daemon:Data>) notification out-data)
  (let ((log (dzn:get (.locator o) <procedure> 'log))
        (log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Data.in.request notification: ~s\n" notification)
    (let* ((label (assoc-ref notification "label"))
           (label (if (string? label) (string->symbol label) label))
           (data (case label
                   ((system) (.parse-data o))
                   ((verify) (.verify-data o))
                   ((state) (.state-data o))
                   ((trace) (.trace-data o))
                   (else (log-error "Data.in.request: no such label: ~s\n" label))))
           (result (libenvelope:wrap label data))
           (origin (assoc-ref notification "origin"))
           (origin (and=> (memq origin '(state system trace)) car)))
      (when origin
        (set! (.request-data o) (cons origin (.request-data o))))
      (set-cdr! out-data result))))

(define-method (data-get_selection (o <daemon:Data>) out-selection)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "data-get_selection\n")
    (set-cdr! out-selection (.selection-data o))
    (.selection-data o)))

(define (seqdiag:seqdiag-selection->location seqdiag-selection working-directory)
  (let* ((file-name (assoc-ref seqdiag-selection "file"))
         (line (assoc-ref seqdiag-selection "line"))
         (column (assoc-ref seqdiag-selection "column"))
         (end (assoc-ref seqdiag-selection "end"))
         (end-line (and end (assoc-ref end "line")))
         (end-column (and end (assoc-ref end "column"))))
    (and file-name
         `(("working-directory" . ,working-directory)
           ("file-name" . ,file-name)
           ("line" . ,line)
           ("column" . ,column)
           ,@(if end-line `(("end-line" . ,end-line)) '())
           ,@(if end-column `(("end-column" . ,end-column)) '())))))

(define-method (seqdiag:index->location (o <daemon:Data>) index)
  (let* ((data (libtransport:unserialize (.simulate-data o)))
         (working-directory (assoc-ref data "working-directory"))
         (seqdiag-location (list-ref (assoc-ref data "seqdiag") index))
         (seqdiag-selections (assoc-ref seqdiag-location "selection")))
    (and (pair? seqdiag-selections)
         (seqdiag:seqdiag-selection->location (car seqdiag-selections) working-directory))))

(define-method (data-get_location (o <daemon:Data>) selection out-location)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "data-get_location ~s\n" selection)
    (match selection
      (((("index" . (? number? index))))
       (let ((location (seqdiag:index->location o index)))
         (and location
              (set-cdr! out-location location))))
      ((("selection" . (location)))
       (set-cdr! out-location location))
      (_
       (set-cdr! out-location '())))))

(define* (update-command command #:key event trail)
  (let* ((name (assoc-ref command "name"))
         (model (assoc-ref command "model"))
         (queue-size (assoc-ref command "queue-size"))
         (input (assoc-ref command "input"))
         (trail (or trail (assoc-ref command "trace")))
         (trail (string-split trail #\,))
         (trail (cond (event (append trail (string-split event #\,)))
                      (else trail)))
         (trail (string-join trail ","))
         (trail (if (and (mingw?) (string-null? trail)) "," trail))
         (argv `("dzn"
                 "--skip-wfc"
                 "simulate"
                 ,@(if model `("--model" ,model) `())
                 ,@(if queue-size `("--queue-size" ,queue-size) `())
                 "--state"
                 "--locations"
                 "--trail" ,trail
                 "-")))
    `(("command" . (("name" . ,name)
                    ("model" . ,model)
                    ("trace" . ,trail)
                    ("input" . ,input)
                    ("argv" . ,argv))))))

(define-method (data-get_command_simulate (o <daemon:Data>) event out-command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "data-get_command_simulate event=~s\n" event)
    (let* ((simulate (.simulate-command o))
           (command (assoc-ref simulate "command"))
           (command (or command simulate)) ;; FIXME
           (trail (and=> (.simulate-data o) trace:get-trail))
           (command (update-command command #:event event #:trail trail)))
      (log-debug "data-get_command_simulate command= ~s\n" command)
      (set! (.simulate-command o) command)
      (set-cdr! out-command command))))

(define-method (data-get_command_simulate_back (o <daemon:Data>) out-command)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "data-get_command_back\n")
    (let* ((simulate (.simulate-command o))
           (command (assoc-ref simulate "command"))
           (command (or command simulate)) ;; FIXME
           (trail (and=> (.simulate-data o) trace:get-back-trail))
           (command (update-command command #:trail trail)))
      (log-debug "data-get_command_back command= ~s\n" command)
      (set! (.simulate-command o) command)
      (set-cdr! out-command command))))
