;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2021 Paul Hoogendijk <paul@dezyne.org>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;; Commentary:
;;;
;;; Dezyne Language Server for the Language Server Protocol
;;; https://microsoft.github.io/language-server-protocol
;;;
;;; Code:

(define-module (ide commands lsp)
  #:use-module (ice-9 getopt-long)
  #:use-module (ide command-line)
  #:use-module (ide config)
  #:use-module (ide lsp))

(define (parse-opts args)
  (let* ((option-spec
          '((help   (single-char #\h))
            (import (single-char #\I) (value #t))))
         (options (getopt-long args option-spec))
         (help?   (option-ref options 'help #f))
         (files   (option-ref options '() '()))
         (usage?  (and (not help?) (pair? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide lsp [OPTION]...
Run a Language Protocol Server for Dezyne over stdio.

  -h, --help                      display this help and exit
  -I, --import=DIR+               add DIR to import path
")
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (main args)
  (let* ((options   (parse-opts args))
         (debug?    (ide:command-line:get 'debug))
         (imports   (multi-opt options 'import))
         (files     (option-ref options '() '()))
         (file-name (and (pair? files) (car files))))
    (lsp:stdio #:debug? debug? #:file-name file-name #:imports imports)))
