;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020, 2022 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (ide command-line)
  #:use-module (srfi srfi-1)
  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 match)
  #:use-module (ide script)
  #:export (command-line:get
            command-line:get-number
            command-line:debugity
            command-line:verbosity
            ide:command-line:get
            ide:debugity
            ide:multi-opt
            ide:verbosity
            multi-opt
            EXIT_OTHER_FAILURE))

(define EXIT_OTHER_FAILURE 2)

(define multi-options
  '(import))

(define (command)
  (let ((files (option-ref (parse-opts (command-line)) '() '(#f))))
    (match files
      ((command t ...) command)
      (_ "dzn"))))

(define* (command-line:get option #:optional default #:key (multi-options multi-options))
  (and (> (length (command-line)) 1)
       (let* ((files (option-ref (parse-opts (command-line)) '() '(#f)))
              (command (string->symbol (command)))
              (parse-opts (let ((module (resolve-module `(ide commands ,command))))
                            (module-ref module 'parse-opts)))
              (options (if command (parse-opts files)
                           (parse-opts (command-line))))
              (multi-opt (lambda (option) (lambda (o) (and (eq? (car o) option) (cdr o))))))
         (if (not (member option multi-options)) (option-ref options option default)
             (filter-map (multi-opt option) options)))))

(define* (command-line:get-number option #:optional (default 0))
  (let* ((value (command-line:get option (number->string default)))
         (number (string->number value)))
    (unless number
      (format (current-error-port)
              "~a: number expected for option --~a, got: `~a'\n"
              (command) option value)
      (exit EXIT_OTHER_FAILURE))
    number))

(define* (ide:command-line:get option #:optional default)
  (and (> (length (command-line)) 1)
       (let ((options (parse-opts (command-line))))
         (option-ref options option default))))

(define (multi-opt options name)
  (let ((opt? (lambda (o) (and (eq? (car o) name) (cdr o)))))
    (filter-map opt? options)))

(define (ide:multi-opt name)
  (or (and (> (length (command-line)) 1)
           (multi-opt (parse-opts (command-line)) name))
      '()))

(define (ide:debugity)
  (or (and (pair? (command-line))
           (member ((compose basename car command-line))
                   '("dzn" ".dzn-real" "ide" ".ide-real"))
           (and=> (ide:multi-opt 'debug) length))
      0))

(define* (command-line:get-number option #:optional (default 0))
  (let* ((value (command-line:get option (number->string default)))
         (number (string->number value)))
    (unless number
      (format (current-error-port) "~a:number expected for option --~a, got: `~a'\n"
              (command) option value)
      (exit EXIT_OTHER_FAILURE))
    number))

(define (command-line:debugity)
  (or (and=> (false-if-exception
              (command-line:get 'debug #:multi-options '(debug))) length)
      0))

(define (ide:verbosity)
  (or (and (pair? (command-line))
           (member ((compose basename car command-line))
                   '("dzn" ".dzn-real" "ide" ".ide-real"))
           (length (ide:multi-opt 'verbose)))
      0))

(define (command-line:verbosity)
  (or (and=> (false-if-exception
              (command-line:get 'verbose #:multi-options '(verbose))) length)
      0))
