;;; Dezyne-IDE --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021, 2022 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2020 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Dezyne-IDE.
;;;
;;; Dezyne-IDE is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commander)
  #:use-module (srfi srfi-9 gnu)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 match)
  #:use-module (oop goops)
  #:use-module (8sync)
  #:use-module (8sync systems websocket client)
  #:use-module (ide config)
  #:use-module (ide util)
  #:use-module (dzn runtime)
  #:use-module (command command)
  #:use-module (transport transport)
  #:use-module (transport Websocket)
  #:export (bye?
            bye-data

            hello?
            hello-data

            info?
            info-data

            parse?
            parse-data

            simulate?
            simulate-data

            state?
            state-data

            make-command
            make-commander
            make-command-handler
            make-finalizer
            make-result-handler
            parse-message
            run-command))

(define-actor <decoupler> (<actor>)
  ((decouple decouple)))

(define-method (decouple (o <decoupler>) message thunk)
  (thunk))

(define* (make-command name argv #:key input model queue-size trail
                       no-compliance? no-deadlock? no-interface-livelock?
                       no-queue-full? no-refusals? no-strict?)
  `(("name" . ,name)
    ("argv" . ,argv)
    ,@(if input `(("input" . ,input)) '())
    ,@(if model `(("model" . ,model)) '())
    ,@(if no-compliance? `(("no-compliance?" . ,no-compliance?)) '())
    ,@(if no-deadlock? `(("no-deadlock?" . ,no-deadlock?)) '())
    ,@(if no-interface-livelock?
          `(("no-interface-livelock?" . ,no-interface-livelock?)) '())
    ,@(if no-queue-full? `(("no-queue-full?" . ,no-queue-full?)) '())
    ,@(if no-refusals? `(("no-refusals?" . ,no-refusals?)) '())
    ,@(if queue-size `(("queue-size" . ,queue-size)) '())
    ,@(if no-strict? `(("no-strict?" . ,no-strict?)) '())
    ,@(if trail `(("trace" . ,trail)) '())))

(define* (make-commander #:key debug? verbose?)
  (let* ((locator (make <dzn:locator>))
         (locator (if debug? locator
                      (dzn:set! locator (const #t) 'trace)))
         (locator (dzn:set! locator (if (or debug? verbose?) log-verbose (const #t)) 'log))
         (locator (dzn:set! locator (if (or debug? verbose?) log-debug (const #t)) 'log-debug))
         (runtime (make <dzn:runtime>))
         (locator (dzn:set! locator runtime))
         (hive (make-hive))
         (locator (dzn:set! locator hive))
         (decoupler (bootstrap-actor hive <decoupler>))
         (decoupler ((@@ (8sync actors) hive-resolve-local-actor) hive decoupler))
         (locator (dzn:set! locator decoupler)))
    (make <command:Commander> #:locator locator #:name "commander")))

(define (make-command-handler commander command)
  (lambda (message)
    (let ((decoupler (actor-id (dzn:get (.locator commander) <decoupler>))))
      (<- decoupler 'decouple
          (lambda _
            (let ((cmd `(("command" . ,command))))
              (action commander .control .in .command cmd)))))))

(define* (make-finalizer commander #:key debug? verbose?)
  (lambda (message)
    (force-output)
    (force-output (current-error-port))
    (let* ((ws (.websocket (.transprt (.daemon commander))))
           (hive (dzn:get (.locator commander) <hive>))
           (clients (slot-ref ws 'clients)))
      (false-if-exception
        (map (compose self-destruct
                      websocket-close
                      (cute (@@ (8sync actors) hive-resolve-local-actor) hive <>))
             clients))
      message)))

(define* (make-result-handler command #:key debug? verbose?)
  (let ((name (match command
                ((("name" . name) rest ...) name)
                (_ "ide"))))
    (lambda (message)
      (when debug?
        (format (current-error-port) "received: ~s\n" message))
      (when (or debug? verbose?)
        (format (current-error-port) "\ninitialization done\n")
        (force-output (current-error-port))
        (let ((url (get-url name)))
          (when (and (string? url) (not (string-null? url)))
            (format #t "~a\n" url))))
      message)))

(define-immutable-record-type <bye>
  (make-bye data)
  bye?
  (data bye-data))

(define-immutable-record-type <hello>
  (make-hello data)
  hello?
  (data hello-data))

(define-immutable-record-type <info>
  (make-info data)
  info?
  (data info-data))

(define-immutable-record-type <parse>
  (make-parse data)
  parse?
  (data parse-data))

(define-immutable-record-type <simulate>
  (make-simulate data)
  simulate?
  (data simulate-data))

(define-immutable-record-type <state>
  (make-state data)
  state?
  (data state-data))

(define (parse-message message)
  (log-debug  "received: ~s\n" message)
  (match message
    ((("bye" . bye))
     (make-bye bye))
    ((("hello" data ...))
     (make-hello data))
    ((("info" data ...))
     (make-info data))
    ((("parse" data ...))
     (make-parse data))
    ((("state" data ...))
     (make-state data))
    ((("simulate" . data))
     (make-simulate data))
    (_ #f)))

(define* (run-command commander command  #:key result-handler ide-port debug? verbose?)
  (let* ((hive (dzn:get (.locator commander) <hive>))
         (decoupler (dzn:get (.locator commander) <decoupler>))
         (decoupler (actor-id (dzn:get (.locator commander) <decoupler>)))
         (finalizer (make-finalizer commander #:debug? debug? #:verbose? verbose?))
         (result-handler (or result-handler
                             (compose finalizer
                                      (make-result-handler command #:debug? debug? #:verbose? verbose?))))
         (name (match command
                 ((("name" . name) rest ...) name)
                 (_ "ide")))
         (setup-handler (make-command-handler commander command))
         (result EXIT_SUCCESS))

    (define (error-handler message)
      (format (current-error-port) "ide ~a: error: ~a\n" name message)
      (set! result EXIT_FAILURE)
      message)

    (set! (.set_up (.out (.control commander))) setup-handler)
    (set! (.result (.out (.control commander))) result-handler)
    (set! (.error (.out (.control commander))) (compose finalizer error-handler))

    (run-hive hive (list (bootstrap-message
                          hive decoupler 'decouple
                          (lambda _
                            (action commander .control .in .setup
                                    (format #f "ws://127.0.0.1:~a" ide-port))))))
    result))
