// dzn-runtime -- Dezyne runtime library
//
// Copyright © 2020, 2023 Rutger van Beusekom <rutger@dezyne.org>
//
// This file is part of dzn-runtime.
//
// All rights reserved.
//
//
// Commentary:
//
// Code:

#include <cassert>
#include <condition_variable>
#include <functional>
#include <future>
#include <mutex>
#include <queue>
#include <vector>

namespace dzn
{
  namespace thread
  {
    class task;
    class pool
    {
      class task;
      friend class task;
      std::vector<std::shared_ptr<task const> > tasks_;
      std::queue<task*> idle_tasks_;
      std::mutex mut_;
    public:
      pool() {}
      std::future<void> defer(const std::function<void()>& work)
      {
        std::unique_lock<std::mutex> lock(mut_);
        if(idle_tasks_.empty())
        {
          task *pt = new task(*this);
          tasks_.push_back(std::shared_ptr<const task>(pt));
          return pt->assign(work);
        }
        else
        {
          std::future<void> fut = idle_tasks_.front()->assign(work);
          idle_tasks_.pop();
          return fut;
        }
      }
    private:
      pool& operator = (const pool&);
      pool(const pool&);

      void idle(task* t)
      {
        std::unique_lock<std::mutex> lock(mut_);
        idle_tasks_.push(t);
      }
      class task
      {
        pool& pool_;
        bool running_;
        std::function<void()> work_;
        std::promise<void> promise_;
        std::mutex mut_;
        std::condition_variable con_;
        std::thread thread_;
      public:
        task(pool& p)
          : pool_(p)
          , running_(true)
          , work_()
          , mut_()
          , con_()
          , thread_(std::bind(&task::worker, self()))
        {}
        ~task()
        {
          std::unique_lock<std::mutex> lock(mut_);
          running_ = false;
          con_.notify_one();
          thread_.detach();
        }
        std::future<void> assign(std::function<void()> work)
        {
          std::unique_lock<std::mutex> lock(mut_);
          assert(!work_);
          work_.swap(work);
          promise_ = std::promise<void>();
          con_.notify_one();
          return promise_.get_future();
        }
      private:
        task* self() { return this; }
        void worker()
        {
          std::unique_lock<std::mutex> lock(mut_);
          do
          {
            while(running_ && !work_)
            {
              con_.wait(lock);
            }
            if(work_)
            {
              std::function<void()> work;
              work_.swap(work);
              lock.unlock();
              work();
              lock.lock();
              promise_.set_value();
              pool_.idle(this);
            }
          }
          while(running_);
        }
        task(const task&);
        task& operator = (const task&);
      };
    };

    std::future<void> defer(const std::function<void()>& work)
    {
      static thread::pool tp;
      return tp.defer(work);
    }
  }
}
//version: 2.15.5
