;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2020,2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (ide service)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)
  #:use-module (ice-9 receive)
  #:use-module (ide pipe)
  #:export (kill-daemon
            read-pid-file
            start-daemon))

(define (default-daemon-directory)
  "The default directory for daemons."
  (or (getenv "HOME")
      (getcwd)))

(define* (read-pid-file file-name #:key (timeout 5))
  "Return content of FILE-NAME if it is an integer, retry for TIMEOUT
seconds."
  (let ((start (current-time)))
    (let loop ()
      (let* ((pid (false-if-exception (with-input-from-file file-name read-string)))
             (pid (and pid (string->number (string-trim-both pid)))))
        (cond ((integer? pid) pid)
              ((< (current-time) (+ start timeout))
               (sleep 1)
               (loop)))))))

(define* (piped-process+exec-command command
                                     #:key
                                     (directory (default-daemon-directory))
                                     log-file
                                     debug?
                                     verbose?)
  "Use piped-process to start COMMAND, a list of strings, as a daemon,
running in directory DIRECTORY and writing output to LOG-FILE."

  (chdir directory)
  (match (pipe)
    ((error-input-port . error-output-port)
     (let* ((command (if (not log-file) command
                         (append command (list (string-append "--log-file=" log-file)))))
            (command (if (not debug?) command
                         (append command (list "--debug"))))
            (command (if (not verbose?) command
                         (append command (list "--verbose"))))
            (commands (list command)))
       (set-unbuffered! (set-non-blocking! error-input-port))
       (move->fdes error-output-port 2)
       (when debug?
         (format (current-error-port) "starting daemon\n"))
       (receive (input-port output-port pids)
           (with-error-to-port error-output-port (cute pipeline commands))
         (when debug?
           (format (current-error-port) "detaching daemon\n"))
         (close error-output-port)
         (close output-port)
         (let ((input-port (set-unbuffered! (set-non-blocking! input-port))))
           (close input-port)
           (car pids)))))))

(define* (start-daemon command
                       #:key
                       (directory (default-daemon-directory))
                       log-file
                       pid-file
                       (pid-file-timeout 5)
                       debug?
                       verbose?)
  "Start COMMAND, a list of strings, as a daemon, running in directory
DIRECTORY and writing output to LOG-FILE, and return its process id.
When PID-FILE is a string, wait for PID-FILE-TIMEOUT seconds for it to
appear before returning."
  (when (file-exists? pid-file)
    (when debug?
      (format (current-error-port) "deleting pid-file: ~s\n" pid-file))
    (delete-file pid-file))

  (let ((pid (piped-process+exec-command command
                                         #:directory directory
                                         #:log-file log-file
                                         #:debug? debug?
                                         #:verbose? verbose?)))
    (when pid-file
      (read-pid-file pid-file #:timeout pid-file-timeout))
    pid))

(define* (kill-daemon pid #:optional (signal SIGTERM))
  "Kill the daemon with process id PID."
  (kill pid signal))
