;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2020, 2021, 2022, 2023 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2020,2021 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;; Copyright © 2022 Rutger van Beusekom <rutger.van.beusekom@verum.com>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

;;
;;; Commentary:
;;;
;;; Code:

(define-module (ide commands verify)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)
  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 rdelim)
  #:use-module (ice-9 regex)
  #:use-module (dzn ast goops)
  #:use-module (dzn ast)
  #:use-module (dzn parse)
  #:use-module (ide commands daemon)
  #:use-module (ide commander)
  #:use-module (ide config)
  #:use-module (ide daemon-config)
  #:use-module (ide command-line)
  #:use-module (ide commands simulate)
  #:use-module (ide pipe)
  #:use-module (ide util)
  #:use-module (ide shell-util))

(define (parse-opts args)
  (let* ((option-spec
          '((help (single-char #\h))
            (import (single-char #\I) (value #t))
            (model (single-char #\m) (value #t))
            (no-constraint (single-char #\C))
            (no-interfaces)
            (no-unreachable (single-char #\U))
            (queue-size (single-char #\q) (value #t))
            (queue-size-defer (value #t))
            (queue-size-external (value #t))))
	 (options (getopt-long args option-spec))
	 (help? (option-ref options 'help #f))
	 (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: ide verify [OPTION]... [FILE]...
Run dzn verify and make a trace view available.

  -h, --help             display this help and exit
  -C, --no-constraint    do not use a constraining process
  -I, --import=DIR+      add directory DIR to import path
  -m, --model=MODEL      explore model MODEL
      --no-interfaces    skip interface verification
  -q, --queue-size=SIZE  use queue size=SIZE for verification [~a]
      --queue-size-defer=SIZE
                         use defer queue size=SIZE for verification [~a]
      --queue-size-external=SIZE
                         use external queue size=SIZE for verification [~a]
  -U, --no-unreachable   skip the unreachable code check

View:
  ~a
" %queue-size %queue-size-defer %queue-size-external
(get-url "trace"))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define (execute input procs process-line)
  (define (read-string port)
    (let loop ((res ""))
      (let ((line (read-line port)))
        (if (eof-object? line) res
            (loop (string-append res "\n" (process-line line)))))))
  (let ((to from pids (pipeline procs)))
    (display input to)
    (catch #t (lambda _ (close to)) (const #f))
    (let ((output (read-string from)))
      (catch #t (lambda _ (close from)) (const #f))
      (values output (apply + (map (compose status:exit-val cdr waitpid) pids))))))

(define (display-any-status json)
  (let* ((data (json-string->alist-scm json))
         (data (if (and (pair? data) (null? (cdr data))) (car data) data)))
    (let ((res (cond ((assoc-ref data "result") (assoc-ref data "result"))
                     ((assoc-ref data "trace") "fail")
                     (else #f))))
      (if res (format #t "verify: ~a: check: ~a: ~a\n" (assoc-ref data "model") (assoc-ref data "assert") res))))
  json)

(define (main args)
  (daemon-config-set!)
  (let* ((options (parse-opts args))
         (debug? (ide:command-line:get 'debug))
         (verbose? (ide:command-line:get 'verbose))
         (queue-size-check (command-line:get-number 'queue-size))
         (queue-size (option-ref options 'queue-size #f))
         (queue-size-defer-check (command-line:get-number 'queue-size-defer))
         (queue-size-defer (option-ref options 'queue-size #f))
         (queue-size-external-check (command-line:get-number
                                     'queue-size-external))
         (queue-size-external (option-ref options 'queue-size-external #f))
         (skip-wfc? (ide:command-line:get 'skip-wfc #f))
         (files (option-ref options '() #f))
         (file-name (car files))
         (imports (multi-opt options 'import))
         (model (option-ref options 'model #f))
         (no-constraint? (option-ref options 'no-constraint #f))
         (no-interfaces? (option-ref options 'no-interfaces #f))
         (no-unreachable? (option-ref options 'no-unreachable #f))
         (result EXIT_SUCCESS))

    (define (simulate-handler message)
      (let ((message (parse-message message)))
        (cond ((simulate? message)
               (let* ((data (simulate-data message))
                      (status (handle-simulate-result data #:debug? debug? #:verbose? verbose?)))
                 (unless (zero? status)
                   (set! result EXIT_FAILURE))
                 (when (isatty? (current-error-port))
                   (format (current-error-port) "See result: ~a\n" (get-url "trace")))))
              (else
               (format (current-error-port)
                       "ide verify: error: received: ~s\n"
                       message)
               (set! result EXIT_FAILURE)))
        message))

    (define* (run-verify input #:key debug? model verbose?)
      (let* ((verify-command
              `("dzn" "--verbose"
                ,@(if skip-wfc? '("--skip-wfc") '())
                "verify"
                ,@(if model `("--model" ,model) '())
                ,@(if no-constraint? `("--no-constraint") '())
                ,@(if no-interfaces? `("--no-interfaces") '())
                ,@(if no-unreachable? `("--no-unreachable") '())
                ,@(if (not queue-size) '()
                      `("--queue-size" ,queue-size))
                ,@(if (not queue-size-defer) '()
                      `("--queue-size-defer" ,queue-size-defer))
                ,@(if (not queue-size-external) '()
                      `("--queue-size-external" ,queue-size-external))
                "-"))
             (output status
                     (execute input (list verify-command) display-any-status)))
        (when verbose?
          (format #t "~a\n" output))
        (if (and (not (zero? status)) (equal? output "")) status
            (let* ((trail (string-split output #\newline))
                   (trail (filter (negate (cute equal? <> "")) trail))
                   (trail (filter (negate (cute string-prefix? "error:" <>)) trail))
                   (trail (filter (negate (cute string-prefix? "verify:" <>)) trail))
                   (trail (string-join trail "\n")))
              trail))))

    (define* (make-simulate-command trail #:key input)
      (let* ((model-match (string-match "(^[ \n]*model: ?([^ \n,]+))" trail))
             (no-compliance? (not (string-contains trail "<non-compliance>")))
             (trail (if model-match
                        (substring trail (match:end model-match))
                        trail))
             (model (or model
                        (let ((ast (identity (string->ast input))))
                          (if (not ast) (exit EXIT_OTHER_FAILURE)
                              (and=> (ast:get-model ast) ast:dotted-name)))))
             (model (if model-match (match:substring model-match 2) model))
             (trail (string-map (lambda (c) (if (eq? c #\newline) #\, c)) trail))
             (trail (if (mingw?) (string-append trail ",") trail)))

        (unless model
          (format (current-error-port) "~a: No dezyne model found\n" file-name)
          (exit EXIT_OTHER_FAILURE))

        (make-command "simulate"
                      `("dzn"
                        "--skip-wfc"
                        "simulate"
                        "--model" ,model
                        ,@(if no-compliance? `("--no-compliance") '())
                        ,@(if (not queue-size) '()
                              `("--queue-size" ,queue-size))
                        ,@(if (not queue-size-defer) '()
                              `("--queue-size-defer" ,queue-size-defer))
                        ,@(if (not queue-size-external) '()
                              `("--queue-size-external" ,queue-size-external))
                        "--trail" ,trail
                        "--locations"
                        "-")
                      #:input input
                      #:model model
                      #:no-compliance? no-compliance?
                      #:queue-size queue-size
                      #:queue-size-defer queue-size-defer
                      #:queue-size-external queue-size-external
                      #:trail trail)))

    (let* ((files (option-ref options '() #f))
           (file-name (car files))
           (imports (multi-opt options 'import))
           (model (option-ref options 'model #f))
           (input (piped-file->stream file-name #:debug? debug? #:imports imports))
           (verify-result (run-verify input #:debug? debug? #:model model #:verbose? verbose?)))

      (when (and (number? verify-result)
                 (not (zero? verify-result)))
        (exit EXIT_OTHER_FAILURE))

      (daemon:start #:debug? debug? #:port %ide-port)
      (let* ((command (make-simulate-command verify-result #:input input))
             (commander (make-commander #:debug? debug? #:verbose? verbose?))
             (result-handler
              (compose (make-finalizer commander #:debug? debug? #:verbose? verbose?)
                       simulate-handler))
             (result
              (+ (run-command commander command
                              #:result-handler result-handler
                              #:ide-port %ide-port #:debug? debug? #:verbose? verbose?)
                 result)))
        (exit result)))))
